#undef BLOGD_EXT
#ifndef  LOG_BUFFER_SIZE
# define LOG_BUFFER_SIZE	65536
#endif
#ifndef  TRANS_BUFFER_SIZE
# define TRANS_BUFFER_SIZE	 4096
#endif
#ifndef  BOOT_LOGFILE
# define BOOT_LOGFILE		"/var/log/boot.msg"
#endif
#ifndef  _PATH_BLOG_FIFO
# define _PATH_BLOG_FIFO	"/dev/blog"
#endif
#include <sys/time.h>
#include <sys/types.h> /* Defines the macros major and minor */
#include <sys/stat.h>
#include <sys/ioctl.h>
#include <sys/socket.h>
#include <sys/param.h>
#include <sys/un.h>
#include <stdlib.h>
#include <stdio.h>
#include <stdarg.h>
#include <string.h>
#include <dirent.h>
#include <fcntl.h>
#include <termios.h>
#include <unistd.h>
#include <signal.h>
#include <fcntl.h>
#include <signal.h>
#include <limits.h>
#include <errno.h>
#include <pthread.h>
#include <sched.h>
#include <syscall.h>
#include "libconsole.h"
#include "listing.h"

extern void error (const char *fmt, ...);
extern void warn  (const char *fmt, ...);
#define STRERR	strerror(errno)

/*
 * push and popd direcotry changes
 */

typedef struct pwd_struct {
    list_t      deep;
    char        *pwd;
} pwd_t;
#define getpwd(list)	list_entry((list), struct pwd_struct, deep)
static list_t pwd = { &(pwd), &(pwd) }, * topd = &(pwd);

void pushd(const char * path)
{
    pwd_t *  dir;

    dir = (pwd_t *)malloc(sizeof(pwd_t));
    if (dir) {
	if (!(dir->pwd = getcwd(NULL, 0)))
	    goto err;
	insert(&(dir->deep), topd->prev);
	goto out;
    }
err:
    error("%s", STRERR);
out:
    if (chdir(path) < 0)
	error ("pushd() can not change to directory %s: %s\n", path, STRERR);
}

void popd(void)
{
    list_t * tail = topd->prev;
    pwd_t *  dir;

    if (list_empty(topd))
	goto out;
    dir = getpwd(tail);
    if (chdir(dir->pwd) < 0)
	error ("popd() can not change directory %s: %s\n", dir->pwd, STRERR);
    free(dir->pwd);
    delete(tail);
    free(dir);
out:
    return;
}

/*
 * Follow the link to its full deep, this because
 * to get the real file name stored in lnk[].
 */
static char lnk[PATH_MAX+1];
static int rlstat(char ** file, struct stat *st)
{
    int ret = -1;
    unsigned int deep = 0;
    static ino_t iprev;			// Recurive loop detection

    if (lstat(*file, st) < 0)
	goto out;

    if (iprev && iprev == st->st_ino)	// We're run on the previous
	goto out;
    iprev = st->st_ino;

    do {
	size_t cnt;
	char *prev_file;

	if (!S_ISLNK(st->st_mode))	// Not a symlink
	    break;

	if (deep++ > MAXSYMLINKS) {
	    errno = EMLINK;
	    goto out;
	}

	prev_file = strdupa(*file);
	if (!prev_file)
	    error("strdupa(): %s\n", STRERR);
	if ((cnt = readlink(*file, lnk, sizeof(lnk) - 1)) < 0)
	    goto out;
	lnk[cnt] = '\0';
	if (lnk[0] != '/') {
	    /* Construct a new valid file name */
	    char *lastslash;

	    lastslash = strrchr (prev_file, '/');
	    if (lastslash != NULL) {
		size_t dirname_len = lastslash - prev_file + 1;
		if (dirname_len + cnt > PATH_MAX)
		    cnt = PATH_MAX - dirname_len;
		memmove(&lnk[dirname_len], &lnk[0], cnt + 1);
		memcpy(&lnk[0], prev_file, dirname_len);
	    }
	}
	*file = lnk;

	if (lstat(*file, st) < 0)
	    goto out;

    } while (S_ISLNK(st->st_mode));
    ret = 0;
    if (*file != (void *)&lnk) {
	strcpy(lnk, *file);
	*file = lnk;
    }
out:
    return ret;
}

/*
 * Arg used: safe out
 */
static void (*vc_reconnect)(int fd) = NULL;
static inline void safeout (int fd, const char *ptr, size_t s)
{
    int saveerr = errno;
    int repeated = 0;
    static int eiocount;

    while (s > 0) {
	ssize_t p = write (fd, ptr, s);
	if (p < 0) {
	    if (errno == EPIPE) {
		warn("error on writing to fd %d: %s\n", fd, STRERR);
		exit (0);
	    }
	    if (errno == EINTR) {
		errno = 0;
		continue;
	    }
	    if (errno == EAGAIN) {	/* Kernel 2.6 seems todo this very often */
		int ret;
		fd_set check;

		if (repeated++ > 1000)
		    error("repeated error on writing to fd %d: %s\n", fd, STRERR);

		FD_ZERO (&check);
		FD_SET (fd, &check);

		/* Avoid high load: wait upto two seconds if system is not ready */
		errno = 0;
		do {
		    struct timeval two = {2, 0};
		    ret = select(fd + 1, (fd_set*)0, &check, (fd_set*)0, &two);

		} while ((ret < 0) && (errno == EINTR));

		if (ret < 0)
		    error("can not write to fd %d: %s\n", fd, STRERR);

		errno = 0;
		continue;
	    }
	    if (errno == EIO) {
		if ((eiocount++ > 10) || !vc_reconnect)
		    error("can not write to fd %d: %s\n", fd, STRERR);
		(*vc_reconnect)(fd);
		errno = 0;
		continue;
	    }
	    error("can not write to fd %d: %s\n", fd, STRERR);
	}
	repeated = 0;
	ptr += p;
	s -= p;
    }
    errno = saveerr;
}

/*
 * Twice used: safe in
 */
static inline ssize_t safein  (int fd, char *ptr, size_t s)
{
    int saveerr = errno;
    ssize_t r = 0;
    int t;
    static int repeated;

    if (s > SSIZE_MAX)
	error("Can not read from fd %d: %m", fd, strerror(EINVAL));

    if ((ioctl(fd, FIONREAD, &t) < 0) || (t == 0)) {
	fd_set check;
	struct timeval zero = {0, 0};

	do {
	    FD_ZERO (&check);
	    FD_SET (fd, &check);

	    /* Avoid deadlock: do not read if nothing is in there */
	    if (select(fd + 1, &check, (fd_set*)0, (fd_set*)0, &zero) <= 0)
		break;

	    r = read (fd, ptr, s);

	} while (r < 0 && (errno == EINTR || errno == EAGAIN));

	/* Do not exit on a broken FIFO */
	if (r < 0 && errno != EPIPE)
	    error("Can not read from fd %d: %s\n", fd, STRERR);

	goto out;
    }

    if (t > 0 && (size_t)t > s)
	t = s;

    repeated = 0;
    while (t > 0) {
	ssize_t p = read (fd, ptr, t);
	if (p < 0) {
	    if (repeated++ > 1000)
		error("Repeated error on reading from fd %d: %s\n", fd, STRERR);
	    if (errno == EINTR || errno == EAGAIN) {
		errno = 0;
		continue;
	    }
	    error("Can not read from fd %d: %s\n", fd, STRERR);
	}
	repeated = 0;
	ptr += p;
	r += p;
	t -= p;
    }
out:
    errno = saveerr;
    return r;
}

/*
 * The stdio file pointer for our log file
 */
static FILE * flog = NULL;
static int fdwrite = -1;
static int fdsec   = -1;
static int fdread  = -1;
static int fdfifo  = -1;

/*
 * Signal control for writing on log file
 */
static volatile sig_atomic_t nsigio = -1;
static sigset_t save_oldset;

/* One shot signal handler */
static void sigio(int sig)
{
    /* Currently no thread active */
    if (nsigio == 0) {
	(void)sigprocmask(SIG_SETMASK, &save_oldset, NULL);
	(void)signal(sig, SIG_IGN);
    }
    nsigio = sig;
}

/*
 * Our ring buffer
 */

typedef struct _mutex {
    int locked;
    pthread_mutex_t mutex;
    pthread_t thread;
} mutex_t;

static inline void lock(mutex_t *mutex)
{
    if (mutex->thread != pthread_self() || !mutex->locked) {
	pthread_mutex_lock(&mutex->mutex);
	mutex->thread = pthread_self();
    }
    mutex->locked++;
}

static inline void unlock(mutex_t *mutex)
{
    if (!--mutex->locked) {
	mutex->thread = 0;
	pthread_mutex_unlock(&mutex->mutex);
    }
}

static mutex_t llock = { 0, PTHREAD_MUTEX_INITIALIZER, 0 };
static mutex_t ljoin = { 0, PTHREAD_MUTEX_INITIALIZER, 0 };
static pthread_cond_t lcond = PTHREAD_COND_INITIALIZER;
static pthread_t    lthread;
static volatile int running;

static       unsigned char data[LOG_BUFFER_SIZE];
static const unsigned char *const end = data + sizeof(data);
static       unsigned char *     head = data;
static       unsigned char *     tail = data;
static volatile ssize_t avail;
#define THRESHOLD	64

static inline void resetlog(void) { tail = head = data; avail = 0; }

static inline void storelog(const unsigned char *const buf, const size_t len)
{
    lock(&llock);
    if (len > end - tail) {
	static int be_warned = 0;
	if (!be_warned) {
	    warn("log buffer exceeded\n");
	    be_warned++;
	}
	goto xout;
    }
    memcpy(tail, buf, len);
    avail = (tail += len) - head;
xout:
    unlock(&llock);
    return;
}

static inline void addlog(const unsigned char c)
{
    lock(&llock);
    if (end - tail <= 0) {
	static int be_warned = 0;
	if (!be_warned) {
	    warn("log buffer exceeded\n");
	    be_warned++;
	}
	goto xout;
    }
    *tail = c;
    avail = (tail += 1) - head;
xout:
    unlock(&llock);
    return;
}

static inline void writelog(void)
{
    if (!flog)
	goto xout;;
    clearerr(flog);
    lock(&llock);
    while (avail > 0) {
	size_t ret = (size_t)avail;

	if (avail > TRANS_BUFFER_SIZE)
	    ret = TRANS_BUFFER_SIZE;

	if (!flog)
	    goto xout;;
	ret = fwrite(head, sizeof(unsigned char), ret, flog);
	if (!ret && ferror(flog))
	    break;
	head += ret;

	if (head >= tail) {		/* empty, reset buffer */
	    resetlog();
	    break;
	}
	if (head > data) {		/* buffer not empty, move contents */
	    avail = tail - head;
	    head = (unsigned char *)memmove(data, head, avail);
	    tail = head + avail;
	}
    }
    unlock(&llock);
    if (!flog)
	goto xout;;
    fflush(flog);
    fdatasync(fileno(flog));
xout:
    return;
}

static inline void flushlog(void)
{
    if (running) pthread_cond_broadcast(&lcond);
}

static inline int poll(int msec, mutex_t *outer)
{
    int ret = 1;

    if (avail <= THRESHOLD) {
	struct timeval now;

	if (gettimeofday(&now, NULL) == 0) {
	    struct timespec abstime;
	    int err;

	    now.tv_usec += msec * 1000;
	    while (now.tv_usec >= 1000000) {
		now.tv_sec++;
		now.tv_usec -= 1000000;
	    }
	    abstime.tv_sec  = now.tv_sec;
	    abstime.tv_nsec = now.tv_usec * 1000;

	    do {
		int locked = outer->locked;
		/* Note: pthread_cond_timedwait() atomically unlocks the mutex */
		outer->locked = 0;
		err = pthread_cond_timedwait(&lcond, &outer->mutex, &abstime);
		outer->locked = locked;
	    } while (err == EINTR);

	    if (err == ETIMEDOUT || err == EBUSY)
		ret = 0;
	}
    } else
	pthread_yield();

    return ret;
}

/*
 * Our transfer buffer
 */
static unsigned char trans[TRANS_BUFFER_SIZE];

/*
 * Remove Escaped sequences and write out result (see
 * linux/drivers/char/console.c in do_con_trol()).
 */
enum {	ESnormal, ESesc, ESsquare, ESgetpars, ESgotpars, ESfunckey,
	EShash, ESsetG0, ESsetG1, ESpercent, ESignore, ESnonstd,
	ESpalette };
#define NPAR 16
static unsigned int state = ESnormal;
static int npar, nl;
static unsigned long int line;

/*
 * Workaround for spinner of fsck/e2fsck
 * Uses ascii lines ending with '\r' only
 */
static int spin;
#define PROGLEN 192
static unsigned char prog[PROGLEN];

static void parselog(unsigned char *buf, const size_t s)
{
    int c;
    ssize_t r = s, up;
    unsigned char uprt[16];

    while (r > 0) {
	c = (unsigned char)*buf;

	switch(state) {
	case ESnormal:
	default:
	    state = ESnormal;
	    switch (c) {
	    case 0  ...  8:
	    case 16 ... 23:
	    case 25:
	    case 28 ... 31:
		nl = 0;
		spin = 0;
		addlog('^'); addlog(c + 64);
		break;
	    case '\n':
		if (spin > 4)	/* last spinner line */
		    storelog(prog, strlen((char*)prog));
		nl = 1;
		line++;
		spin = 0;
		addlog(c);
		break;
	    case '\r':
		spin++;
		if (spin < 5) {
		    if (spin > 1)
			addlog('\n');
		    nl = 1;
		}
		if (spin == 5)
		    storelog((unsigned char*)"\n<progress bar skipped>\n", 24);
		break;
	    case 14:
	    case 15:
		/* ^N and ^O used in xterm for rmacs/smacs  *
		 * on console \033[10m and \033[11m is used */
	    case 24:
	    case 26:
		spin = 0;
		break;
	    case '\033':
		spin = 0;
		state = ESesc;
		break;
	    case '\t':
	    case  32 ... 126:
	    case 160 ... 255:
		if (spin < 5) {
		    if (spin == 1 && nl)
			addlog('\n');
		    addlog(c);
		} else {		/* Seems to be a lengthy spinner line */
		    static   int old = 0;
		    static ssize_t p = 0;
		    if (old != spin) {
			old = spin;	/* Next line overwrite on tty */
			p = 0;
			bzero(prog, PROGLEN);
		    }
		    if (p < PROGLEN)
			prog[p++] = c;	/* buffer always current line */
		}
		nl = 0;
		break;
	    case 127:
		nl = 0;
		spin = 0;
		addlog('^'); addlog('?');
		break;
	    case 128 ... 128+26:
	    case 128+28 ... 159:
		nl = 0;
		spin = 0;
		if ((up = snprintf((char*)uprt, sizeof(uprt), "\\%03o", c)) > 0)
		    storelog(uprt, (size_t)up);
		break;
	    case 128+27:
		spin = 0;
		state = ESsquare;
		break;
	    default:
		nl = 0;
		spin = 0;
		if ((up = snprintf((char*)uprt, sizeof(uprt), "0x%X", c)) > 0)
		    storelog(uprt, (size_t)up);
		break;
	    }
	    break;
	case ESesc:
	    state = ESnormal;
	    switch((unsigned char)c) {
	    case '[':
		state = ESsquare;
		break;
	    case ']':
		state = ESnonstd;
		break;
	    case '%':
		state = ESpercent;
		break;
	    case 'E':
	    case 'D':
		if (spin > 4)	/* last spinner line */
		    storelog(prog, strlen((char*)prog));
		nl = 1;
		line++;
		spin = 0;
		addlog('\n');
		break;
	    case '(':
		state = ESsetG0;
		break;
	    case ')':
		state = ESsetG1;
		break;
	    case '#':
		state = EShash;
		break;
#ifdef BLOGD_EXT
	    case '^':				/* Boot log extension */
		state = ESignore;
		break;
#endif
	    default:
		break;
	    }
	    break;
	case ESnonstd:
	    if        (c == 'P') {
		npar = 0;
		state = ESpalette;
	    } else if (c == 'R')
		state = ESnormal;
	    else
		state = ESnormal;
	    break;
	case ESpalette:
	    if ((c>='0'&&c<='9') || (c>='A'&&c<='F') || (c>='a'&&c<='f')) {
		npar++;
		if (npar==7)
		    state = ESnormal;
	    } else
		state = ESnormal;
	    break;
	case ESsquare:
	    npar = 0;
	    state = ESgetpars;
	    if (c == '[') {
		state = ESfunckey;
		break;
	    }
	    if (c == '?')
		break;
	case ESgetpars:
	    if (c==';' && npar<NPAR-1) {
		npar++;
		break;
	    } else if (c>='0' && c<='9') {
		break;
	    } else
		state = ESgotpars;
	case ESgotpars:
	    state = ESnormal;
	    break;
	case ESpercent:
	    state = ESnormal;
	    break;
	case ESfunckey:
	case EShash:
	case ESsetG0:
	case ESsetG1:
	    state = ESnormal;
	    break;
#ifdef BLOGD_EXT
	case ESignore:				/* Boot log extension */
	    state = ESesc;
	    {
		unsigned char echo[64];
		ssize_t len;

		if ((len = snprintf(echo, sizeof(echo), "\033[%lu;%dR", line, nl)) > 0)
		    safeout(fdread, echo, len);
		else
		    safeout(fdread, "\033R", 2);
		tcdrain(fdread);
	    }
	    break;
#endif
	}
	buf++;
	r--;
    }
}

static void copylog(const unsigned char *buf, const size_t s)
{
    storelog(buf, s);
}

static void *action(void *dummy)
{
    sigset_t sigset, save_oldset;
    sigemptyset(&sigset);

    sigaddset(&sigset, SIGTTIN);
    sigaddset(&sigset, SIGTTOU);
    sigaddset(&sigset, SIGTSTP);
    sigaddset(&sigset, SIGHUP);
    sigaddset(&sigset, SIGINT);
    sigaddset(&sigset, SIGQUIT);
    sigaddset(&sigset, SIGTERM);
    (void)pthread_sigmask(SIG_BLOCK, &sigset, &save_oldset);

    lock(&ljoin);
    running = 1;
    while (running) {

	if (!poll(30, &ljoin))
	    continue;

	if (!flog)
	    break;

	writelog();
    }
    unlock(&ljoin);

    (void)pthread_sigmask(SIG_SETMASK, &save_oldset, NULL);
    return NULL;
}

/*
 * Prepare I/O
 */
static const char *fifo_name = _PATH_BLOG_FIFO;

void prepareIO(void (*rfunc)(int), const int in, const int out, const int second)
{
    vc_reconnect = rfunc;
    fdread  = in;
    fdwrite = out;
    fdsec   = second;

    if (fifo_name && fdfifo < 0) {
	struct stat st;
	errno = 0;
	/* udev support: create /dev/blog if not exist */
	if (stat(fifo_name, &st) < 0 && errno == ENOENT)
	    (void)mkfifo(fifo_name, 0600);
	errno = 0;
	if (!stat(fifo_name, &st) && S_ISFIFO(st.st_mode)) {
	    if ((fdfifo = open(fifo_name, O_RDWR|O_NOCTTY)) < 0)
		warn("can not open named fifo %s: %s\n", fifo_name, STRERR);
	}
    }
}

/*
 * Seek for input, more input ...
 */
static void more_input (struct timeval *timeout)
{
    fd_set watch;
    int nfds, wfds;

    FD_ZERO (&watch);
    FD_SET (fdread, &watch);

    if (fdfifo > 0) {
	FD_SET (fdfifo, &watch);
	wfds = (fdread > fdfifo ? fdread : fdfifo) + 1;
    } else
	wfds = fdread + 1;

    nfds = select(wfds, &watch, (fd_set*)0, (fd_set*)0, timeout);

    if (nfds < 0) {
	timeout->tv_sec  = 0;
	timeout->tv_usec = 0;
	if (errno != EINTR)
	    error ("select(): %s\n", STRERR);
	goto nothing;
    }

    if (!nfds)
	goto nothing;

    if (FD_ISSET(fdread, &watch)) {
	const ssize_t cnt = safein(fdread, (char*)trans, sizeof(trans));

	if (cnt > 0) {
	    parselog(trans, cnt);		/* Parse and make copy of the input */

	    safeout(fdwrite, (char*)trans, cnt);	/* Write copy of input to real tty */
	    (void)tcdrain(fdwrite);

	    if (fdsec > 0) {
		safeout(fdsec, (char*)trans, cnt);	/* Write copy of input to second tty */
		(void)tcdrain(fdsec);
	    }

	    flushlog();
	}
    }

    if (fdfifo > 0 && FD_ISSET(fdfifo, &watch)) {
	const ssize_t cnt = safein(fdfifo, (char*)trans, sizeof(trans));

	if (cnt > 0) {
	    copylog(trans, cnt);		/* Make copy of the input */
	    flushlog();
	}
    }
    errno = 0;
nothing:
    return;
}

/*
 *  The main routine for blogd.
 */
void safeIO (void)
{
    struct timeval timeout;
    static int log = -1;

    if (!nsigio) /* signal handler set but no signal recieved */
	goto skip;

    if (log < 0) {
#ifdef DEBUG_SIGIO
	if (nsigio < 0)
	   goto skip;
#else
	if (nsigio < 0) {
	    /*
	     * Maybe access() does lie under kernel 2.6
	     * or the root file system is rw mounted
	     */
	    const char *runlevel = getenv("RUNLEVEL");
	    if (runlevel && (*runlevel < '0' || *runlevel > '6'))
		goto skip;
	}
#endif
	if (access(BOOT_LOGFILE, W_OK) < 0) {
	    if (errno != ENOENT && errno != EROFS)
		error("Can not write to %s: %s\n", BOOT_LOGFILE, STRERR);
	    goto skip;
	}
	if ((log = open(BOOT_LOGFILE, O_WRONLY|O_NOCTTY|O_APPEND)) < 0) {
	    if (errno != ENOENT && errno != EROFS)
		error("Can not open %s: %s\n", BOOT_LOGFILE, STRERR);
	    goto skip;
	}
	if ((flog = fdopen (log, "a")) == NULL)
		error("Can not open %s: %s\n", BOOT_LOGFILE, STRERR);

	nsigio = SIGIO; /* We do not need a signal handler */
	(void)signal(SIGIO, SIG_IGN);
    }

skip:
    if (nsigio < 0) { /* signal handler not set, so do it */
	struct sigaction new;
	sigset_t sigset;

	/* Currently no thread active */
	nsigio = 0;

	new.sa_handler = sigio;
	sigemptyset(&new.sa_mask);
	new.sa_flags = SA_RESTART;
	do {
	    if (sigaction(SIGIO, &new, NULL) == 0)
		break;
	} while (errno == EINTR);

	sigemptyset(&sigset);
	sigaddset(&sigset, SIGIO);
	(void)sigprocmask(SIG_UNBLOCK, &sigset, &save_oldset);
    }

    timeout.tv_sec  = 5;
    timeout.tv_usec = 0;
    more_input(&timeout);

    if (flog && !running) {
	int policy = SCHED_RR;
	struct sched_param param;

	pthread_getschedparam(pthread_self(), &policy, &param);
	pthread_create(&lthread, NULL, &action, NULL);

	policy = SCHED_RR;
	param.sched_priority = sched_get_priority_max(policy)/2 + 1;
	pthread_setschedparam(pthread_self(), policy, &param);
	pthread_setschedparam(lthread, policy, &param);
    }
}

/*
 *
 */
void closeIO(void)
{
    struct timeval timeout;
    int n = 6;

    /* Maybe we've catched a signal, therefore */
    if (flog) {
	fflush(flog);			/* Clear out stdio buffers   */
	fdatasync(fileno(flog));
    } else
	warn("no message logging because /var file system is not accessible\n");
    (void)tcdrain(fdwrite);		/* Hold in sync with console */
    if (fdsec > 0)
	(void)tcdrain(fdsec);		/* Hold in sync with second console */

    do {
	/*
	 * Repeat this as long as required,
	 * but not more than 3 seconds
	 */

	if (!n)
	    break;
	n--;

	timeout.tv_sec  = 0;
	timeout.tv_usec = 5*100*1000;	/* A half second */

    	more_input(&timeout);

	if (!flog)
	    break;
    } while (timeout.tv_sec || timeout.tv_usec);

    if (running) {
	lock(&ljoin);
	running = 0;
	unlock(&ljoin);
	flushlog();
	pthread_cancel(lthread);
    }

    if (!flog)
	goto xout;

    writelog();

    if (!nl)
	fputc('\n', flog);

    if (!flog)
	goto xout;

    (void)fclose(flog);
    flog = NULL;
xout:
    (void)tcdrain(fdwrite);
    if (fdsec > 0)
	(void)tcdrain(fdsec);

    return;
}

/*
 * Fetch our real underlying terminal character device,
 * returned name should be freed if not used anymore.
 */
static void ctty(pid_t pid, unsigned int * tty, pid_t * ttypgrp)
{
    char fetched[NAME_MAX+1];
    ssize_t cnt;
    int fd;

    sprintf(fetched, "/proc/%d/stat", (int)pid);
    if ((fd = open(fetched, O_RDONLY|O_NOCTTY)) < 0)
	error("can not open(%s): %s\n", fetched, STRERR);
    cnt = safein(fd, fetched, sizeof(fetched));
    close(fd);

    if (cnt <= 0)
	error("can not read from proc: %s\n", STRERR);
    else	
	/* format pid comm state ppid pgrp session tty tpgid */
	if (sscanf(fetched,"%*d %*s %*c %*d %*d %*d %u %u %*u", tty, ttypgrp) != 2)
	    error("can not sscanf for my tty: %s\n", STRERR);
}

/* fall back routine to fetch tty */
static unsigned int fallback(const pid_t pid, const pid_t ppid)
{
    unsigned int tty = 0;
    pid_t ttypgrp = -1;
    pid_t  pgrp = getpgid(pid);
    pid_t ppgrp = getpgid(ppid);

    ctty(pid, &tty, &ttypgrp);

    if (pgrp != ttypgrp && ppgrp != ttypgrp) {
	int fdfrom[2];
	pid_t pid = -1;  /* Inner pid */

	if (pipe(fdfrom) < 0)
	    error("can not create a pipe: %s\n", STRERR);

	switch ((pid = fork())) {
	case 0:
	    {   void (*save_sighup);

		dup2( fdfrom[1], 1);
		close(fdfrom[1]);
		close(fdfrom[0]);

		pid = getpid();	/* our pid is not zero */

		if (pid != getsid(pid)) {
		    if (pid == getpgid(pid))
			setpgid(0, ppgrp);
		    setsid();
		}

		/* Remove us from any controlling tty */
		save_sighup = signal(SIGHUP, SIG_IGN);
		if (ttypgrp > 0)
		    ioctl(0, TIOCNOTTY, (void *)1);
		(void)signal(SIGHUP, save_sighup);

		/* Take stdin as our controlling tt< */
		if (ioctl(0, TIOCSCTTY, (void *)1) < 0)
		    warn("can not set controlling tty: %s\n", STRERR);

		ctty(pid, &tty, &ttypgrp);

		/* Never hold this controlling tty */
		save_sighup = signal(SIGHUP, SIG_IGN);
		if (ttypgrp > 0)
		    ioctl(0, TIOCNOTTY, (void *)1);
		(void)signal(SIGHUP, save_sighup);

		printf("|%u|%u|", tty, ttypgrp);  /* stdout to pipe synchronize ... */

		exit(0);
	    } break;
	case -1:
	    error("can not execute: %s\n", STRERR);
	    break;
	default:
	    {   int fd = dup(0);
		dup2( fdfrom[0], 0);
		close(fdfrom[0]);
		close(fdfrom[1]);

		scanf("|%u|%u|", &tty, &ttypgrp); /* ... with stdin from pipe here  */

		dup2(fd, 0);
		close(fd);
	    } break;
	}
    }

    return tty;
}

static int checkdev(char ** retname, unsigned int tty, DIR * dev)
{
    int found = 0;
    struct dirent * d;
    struct stat st;
    static int deep;

    memset(&st, 0, sizeof(struct stat));
    while ((d = readdir(dev))) {
	char * name = d->d_name;

	if (*name == '.')
	    continue;

	if (deep == 0) {
	    /*
	     * No terminals therein
	     */
	    if (!strcmp(name, "shm"))
		continue;

	    if (!strncmp(name, "snd", 3))
		continue;

	    if (!strncmp(name, "ram", 3))
		continue;

	    if (!strcmp(name, "cciss"))
		continue;

	    if (!strcmp(name, "ataraid"))
		continue;

	    if (!strcmp(name, "cbd"))
		continue;

	    if (!strcmp(name, "cpu"))
		continue;

	    if (!strcmp(name, "dvb"))
		continue;

	    if (!strcmp(name, "i2o"))
		continue;

	    if (!strcmp(name, "ida"))
		continue;

	    if (!strcmp(name, "inet"))
		continue;

	    if (!strcmp(name, "input"))
		continue;

	    if (!strcmp(name, "mvideo"))
		continue;

	    if (!strcmp(name, "raw"))
		continue;

	    if (!strcmp(name, "rd"))
		continue;

	    if (!strcmp(name, "scsi"))
		continue;
	}

	if (rlstat(&name, &st) < 0) {
	    if (errno != ENOENT)
		warn("can not follow %s: %s\n", name, STRERR);
	    continue;
	}

	if (!strncmp(name, "/proc", 5))
	    continue;

	if (S_ISDIR(st.st_mode)) {
	    DIR * cur = opendir(name);
	    if (!cur)
		error("can not opendir(%s): %s\n", name, STRERR);
	    pushd(name);
	    deep++;
	    found = checkdev(retname, tty, cur);
	    deep--;
	    popd();
	    closedir(cur);

	    if (found)
		break;

	    continue;
	}

	if (!S_ISCHR(st.st_mode))
	    continue;

	if ((dev_t)tty != st.st_rdev)
	    continue;

	/*
	 * rlstat() uses lnk[] as buffer to which points
	 * the pointer name after rlstat().
	 */
	if (*name && *name != '/') {
	    char * ptr = strdupa(name);
	    if (!ptr)
		error("checkdev(): %s\n", STRERR);
	    getcwd(name, PATH_MAX - 1 - strlen(ptr));
	    strcat(name, "/");
	    strcat(name, ptr);
	}

	found++;

	/*
	 * Allocate memory to be able to return several
	 * different buffers for different files names.
	 */
	name = strdup(name);
	if (!name)
	    error("checkdev(): %s\n", STRERR);
	*retname = name;
	break;
    }

    return found;
}

/* main routine to fetch tty */
char * fetchtty(const pid_t pid, const pid_t ppid, unsigned int *mjmi)
{
    unsigned int tty = 0, found = 0;
    char * name = NULL;
    DIR * dev;

#ifdef TIOCGDEV
    if (ioctl (0, TIOCGDEV, &tty) < 0) {
	if (errno == EINVAL && !getenv("NOTIOCGDEV"))
	    warn("Warning: the ioctl TIOCGDEV is not known by the kernel\n");
#else
#       error The ioctl TIOCGDEV is not defined (SuSE TIOCGDEV patch is missed)
#endif
	if (!(name = ttyname(0)) || !strcmp(name, "/dev/console"))
	    tty = fallback(pid, ppid);
	else {
	    name = strdup(name);
	    if (!name)
		error("fetchtty(): %s\n", STRERR);
	    goto out;
	}
#ifdef TIOCGDEV
    }
#endif
    if (mjmi) *mjmi = tty;

    if (!(dev = opendir("/dev")))
	error("can not opendir(/dev): %s\n", STRERR);
    pushd("/dev");
    found = checkdev(&name, tty, dev);
    popd();
    closedir(dev);

    if (!name)
	goto out;

    if (!found) {
	free(name);
	name = (char*)0;
    }
out:
    return name;
}

/* Do we have some more system console around? */
char * secondtty(char * compare)
{
    char buf[1024], *ptr, *shcmp, *res = (char*)0;
    unsigned int tty = 0, found = 0;
    int fd = -1, len;
    char * name = (char*)0;
    DIR * dev;

    if ((fd = open("/proc/cmdline", O_RDONLY|O_NOCTTY)) < 0) {
	warn("can not open /proc/cmdline\n");
	goto out;
    }

    if ((len = read(fd, buf, sizeof(buf) - 1)) < 0) {
	warn("can not read /proc/cmdline\n");
	close(fd);
	goto out;
    }
    close(fd);

    if (len == 0)
	goto out;

    shcmp = compare;
    if (!strncmp(shcmp, "/dev/", 5))
	shcmp += 5;

    /*
     * Check for e.g. /dev/tty[1-9] which is equal to /dev/tty0
     */
    if (!strncmp(shcmp, "tty", 3)) {
	size_t len = strspn(shcmp + 3, "123456789");

	if (strlen(shcmp) == len + 3) {
	    compare = "/dev/tty0";
	    shcmp   = "tty0";
	}
    }

    ptr = &buf[len];
    while (ptr >= &buf[0]) {
	if (*ptr == ',' || *ptr == ' ' || *ptr == '\t' || *ptr == '\r' || *ptr == '\n') {
	    *ptr-- = 0;
	    continue;
	}
	if (*ptr == 'c' && !strncmp(ptr, "console=", 8)) {
	    char * console  = ptr + 8;

	    if (!strncmp(console, "/dev/", 5))
		console += 5;

	    /*
	     * Compare known console tty with that of the kernel command
	     * line.  If already known skip this console tty and search
	     * for the next one.
	     */
	    if (strcmp(shcmp, console)) {
		res = console;		/* New device not identical to tty */
		break;
	    }
	}
	ptr--;
    }

    if (!res)
	goto out;

    if (!(dev = opendir("/dev")))
	error("can not opendir(/dev): %s\n", STRERR);
    pushd("/dev");

    /* Open second console e.g. /dev/tty0 */
    if ((fd = open(res, O_RDWR|O_NONBLOCK|O_NOCTTY)) < 0)
	goto out;

    /*
     * We do this because if we would write out the buffered
     * messages to e.g. /dev/tty0 after this we would (re)read
     * those and buffer them again which leads to an endless loop.
     */
#ifdef TIOCGDEV
    if (ioctl (fd, TIOCGDEV, &tty) < 0) {
	if (errno == EINVAL && !getenv("NOTIOCGDEV"))
	    warn("Warning: the ioctl TIOCGDEV is not known by the kernel\n");
	close(fd);
	popd();
	closedir(dev);
	goto out;
    }
#else
#   error The ioctl TIOCGDEV is not defined (SuSE TIOCGDEV patch is missed)
#endif
    close(fd);

    /* Try to open the real device e.g. /dev/tty1 */
    found = checkdev(&name, tty, dev);

    popd();
    closedir(dev);

    if (!name)
	goto out;

    if (!found) {
	free(name);
	name = (char*)0;
	goto out;
    }

    if (!strcmp(compare, name)) {
	free(name);
	name = (char*)0;
	goto out;		/* Already in use */
    }

    return name;
out:
    return (char*)0;
}
