#!/usr/bin/python
# Copyright (C) 2016-2022 Musarubra US LLC. All Rights Reserved.
# Script validates the prefs.xml using the XML schema of product version 10.2.2
# For any xml parsing error or schema error, it will exit with a return code of 1
# For any other error like missing python module, missing file, it will exit with a return code of 0
# For a xml which was validated correctly, it will exit with a return code of 0
import os
import sys
# If lxml python module is not available in the system, then exit the script
try:
    import lxml.etree as ET
except:
    print("Unable to validate prefs.xml as lxml is not available")
    sys.exit(0)
from io import StringIO

def usage():
    print("Pass the absolute path of the prefs.xml file to validate")
    sys.exit(0)

if len(sys.argv) > 1:
    file_to_validate = sys.argv[1]
    if not os.path.isfile(file_to_validate):
        usage()
else:
    usage()

valid_DNDOASProfile = StringIO(u'''\
   <xs:schema xmlns:xs="http://www.w3.org/2001/XMLSchema" elementFormDefault="qualified" attributeFormDefault="unqualified">
       <xs:element name="DNDOASProfile">
         <xs:complexType>
               <xs:all>
                 <xs:element name="DetectJokesEnabled" type="xs:int"></xs:element>
                 <xs:element name="DetectUnknownMacrosEnabled" type="xs:int"></xs:element>
                 <xs:element name="DetectUnknownProgramsEnabled" type="xs:int"></xs:element>
                 <xs:element name="DetectUnwantedProgramsEnabled" type="xs:int"></xs:element>
                 <xs:element name="FileTypeForDefaultAndSpecified">
                   <xs:complexType>
                     <xs:sequence>
                           <xs:element name="FileType" maxOccurs="unbounded" type="xs:string"></xs:element>
                     </xs:sequence>
                     <xs:attribute name="delimiter" type="xs:string"></xs:attribute>
                   </xs:complexType>
                 </xs:element>
                 <xs:element name="FileTypeForOnlySpecified">
                   <xs:complexType>
                     <xs:sequence>
                           <xs:element name="FileType" maxOccurs="unbounded" type="xs:string"></xs:element>
                     </xs:sequence>
                     <xs:attribute name="delimiter" type="xs:string"></xs:attribute>
                   </xs:complexType>
                 </xs:element>
                 <xs:element name="FileTypeToScan" type="xs:int"></xs:element>
                 <xs:element name="InfectionPrimaryAction" type="xs:int"></xs:element>
                 <xs:element name="InfectionSecondaryAction" type="xs:int"></xs:element>
                 <xs:element name="PUPPrimaryAction" type="xs:int"></xs:element>
                 <xs:element name="PUPSecondaryAction" type="xs:int"></xs:element>
                 <xs:element name="ProcessList">
                   <xs:complexType>
                     <xs:sequence>
                           <xs:element name="Process" maxOccurs="unbounded" type="xs:string"></xs:element>
                     </xs:sequence>
                     <xs:attribute name="delimiter" type="xs:string"></xs:attribute>
                     <xs:attribute name="matchallprocess" type="xs:int"></xs:attribute>
                   </xs:complexType>
                 </xs:element>
                 <xs:element name="ProfileKeyword" type="xs:string"></xs:element>
                 <xs:element name="ProfileName" type="xs:string"></xs:element>
                 <xs:element name="ReadExcludeFoldersAndSubFolders">
                   <xs:complexType>
                     <xs:choice maxOccurs="unbounded">
                           <xs:element name="CLIFolder" maxOccurs="unbounded" type="xs:string"></xs:element>
                           <xs:element name="ePOFolder" maxOccurs="unbounded" type="xs:string"></xs:element>
                     </xs:choice>
                     <xs:attribute name="delimiter" type="xs:string"></xs:attribute>
                   </xs:complexType>
                 </xs:element>
                 <xs:element name="ReadExclusionFiles">
                   <xs:complexType>
                     <xs:choice maxOccurs="unbounded">
                           <xs:element name="CLIFile" maxOccurs="unbounded" type="xs:string"></xs:element>
                           <xs:element name="ePOFile" maxOccurs="unbounded" type="xs:string"></xs:element>
                     </xs:choice>
                     <xs:attribute name="delimiter" type="xs:string"></xs:attribute>
                   </xs:complexType>
                 </xs:element>
                 <xs:element name="ReadExclusionPaths">
                   <xs:complexType>
                     <xs:choice maxOccurs="unbounded">
                           <xs:element name="CLIPath" maxOccurs="unbounded" type="xs:string"></xs:element>
                           <xs:element name="ePOPath" maxOccurs="unbounded" type="xs:string"></xs:element>
                     </xs:choice>
                     <xs:attribute name="delimiter" type="xs:string"></xs:attribute>
                   </xs:complexType>
                 </xs:element>
                 <xs:element name="ReadExclusionTypes">
                   <xs:complexType>
                     <xs:choice maxOccurs="unbounded">
                           <xs:element name="CLIFileType" maxOccurs="unbounded" type="xs:string"></xs:element>
                           <xs:element name="ePOFileType" maxOccurs="unbounded" type="xs:string"></xs:element>
                     </xs:choice>
                     <xs:attribute name="delimiter" type="xs:string"></xs:attribute>
                   </xs:complexType>
                 </xs:element>
                 <xs:element name="ReadWriteExcludeFoldersAndSubFolders">
                   <xs:complexType>
                     <xs:choice maxOccurs="unbounded">
                           <xs:element name="CLIFolder" maxOccurs="unbounded" type="xs:string"></xs:element>
                           <xs:element name="ePOFolder" maxOccurs="unbounded" type="xs:string"></xs:element>
                     </xs:choice>
                     <xs:attribute name="delimiter" type="xs:string"></xs:attribute>
                   </xs:complexType>
                 </xs:element>
                 <xs:element name="ReadWriteExclusionFiles">
                   <xs:complexType>
                     <xs:choice maxOccurs="unbounded">
                           <xs:element name="CLIFile" maxOccurs="unbounded" type="xs:string"></xs:element>
                           <xs:element name="ePOFile" maxOccurs="unbounded" type="xs:string"></xs:element>
                     </xs:choice>
                     <xs:attribute name="delimiter" type="xs:string"></xs:attribute>
                   </xs:complexType>
                 </xs:element>
                 <xs:element name="ReadWriteExclusionPaths">
                   <xs:complexType>
                     <xs:choice maxOccurs="unbounded">
                           <xs:element name="CLIPath" maxOccurs="unbounded" type="xs:string"></xs:element>
                           <xs:element name="ePOPath" maxOccurs="unbounded" type="xs:string"></xs:element>
                     </xs:choice>
                     <xs:attribute name="delimiter" type="xs:string"></xs:attribute>
                   </xs:complexType>
                 </xs:element>
                 <xs:element name="ReadWriteExclusionTypes">
                   <xs:complexType>
                     <xs:choice maxOccurs="unbounded">
                           <xs:element name="CLIFileType" maxOccurs="unbounded" type="xs:string"></xs:element>
                           <xs:element name="ePOFileType" maxOccurs="unbounded" type="xs:string"></xs:element>
                     </xs:choice>
                     <xs:attribute name="delimiter" type="xs:string"></xs:attribute>
                   </xs:complexType>
                 </xs:element>
                 <xs:element name="ScanArchiveEnabled" type="xs:int"></xs:element>
                 <xs:element name="ScanErrorAction" type="xs:int"></xs:element>
                 <xs:element name="ScanExtension" type="xs:int"></xs:element>
                 <xs:element name="ScanMacros" type="xs:int"></xs:element>
                 <xs:element name="ScanMailEnabled" type="xs:int"></xs:element>
                 <xs:element name="ScanMode" type="xs:int"></xs:element>
                 <xs:element name="ScanOnNetwork" type="xs:int"></xs:element>
                 <xs:element name="ScanTimeoutAction" type="xs:int"></xs:element>
                 <xs:element name="WriteExcludeFoldersAndSubFolders">
                   <xs:complexType>
                     <xs:choice maxOccurs="unbounded">
                           <xs:element name="CLIFolder" maxOccurs="unbounded" type="xs:string"></xs:element>
                           <xs:element name="ePOFolder" maxOccurs="unbounded" type="xs:string"></xs:element>
                     </xs:choice>
                     <xs:attribute name="delimiter" type="xs:string"></xs:attribute>
                   </xs:complexType>
                 </xs:element>
                 <xs:element name="WriteExclusionFiles">
                   <xs:complexType>
                     <xs:choice maxOccurs="unbounded">
                           <xs:element name="CLIFile" maxOccurs="unbounded" type="xs:string"></xs:element>
                           <xs:element name="ePOFile" maxOccurs="unbounded" type="xs:string"></xs:element>
                     </xs:choice>
                     <xs:attribute name="delimiter" type="xs:string"></xs:attribute>
                   </xs:complexType>
                 </xs:element>
                 <xs:element name="WriteExclusionPaths">
                   <xs:complexType>
                     <xs:choice maxOccurs="unbounded">
                           <xs:element name="CLIPath" maxOccurs="unbounded" type="xs:string"></xs:element>
                           <xs:element name="ePOPath" maxOccurs="unbounded" type="xs:string"></xs:element>
                     </xs:choice>
                     <xs:attribute name="delimiter" type="xs:string"></xs:attribute>
                   </xs:complexType>
                 </xs:element>
                 <xs:element name="WriteExclusionTypes">
                   <xs:complexType>
                     <xs:choice maxOccurs="unbounded">
                           <xs:element name="CLIFileType" maxOccurs="unbounded" type="xs:string"></xs:element>
                           <xs:element name="ePOFileType" maxOccurs="unbounded" type="xs:string"></xs:element>
                     </xs:choice>
                     <xs:attribute name="delimiter" type="xs:string"></xs:attribute>
                   </xs:complexType>
                 </xs:element>
               </xs:all>
               <xs:attribute name="type" type="xs:int"></xs:attribute>
         </xs:complexType>
       </xs:element>
   </xs:schema>
''')

valid_ODSProfile = StringIO(u'''\
   <xs:schema xmlns:xs="http://www.w3.org/2001/XMLSchema" elementFormDefault="qualified" attributeFormDefault="unqualified">
       <xs:element name="ODSProfile">
         <xs:complexType>
               <xs:all>
                 <!-- Depending on the type of ODS Profile, FileTypeForDefaultAndSpecified may not be present -->
                 <xs:element name="FileTypeForDefaultAndSpecified" minOccurs="0">
                   <xs:complexType>
                     <xs:sequence>
                           <xs:element name="FileType" maxOccurs="unbounded" type="xs:string"></xs:element>
                     </xs:sequence>
                     <xs:attribute name="delimiter" type="xs:string"></xs:attribute>
                   </xs:complexType>
                 </xs:element>
                 <!-- Depending on the type of ODS Profile, FileTypeForOnlySpecified may not be present -->
                 <xs:element name="FileTypeForOnlySpecified" minOccurs="0">
                   <xs:complexType>
                     <xs:sequence>
                           <xs:element name="FileType" maxOccurs="unbounded" type="xs:string"></xs:element>
                     </xs:sequence>
                     <xs:attribute name="delimiter" type="xs:string"></xs:attribute>
                   </xs:complexType>
                 </xs:element>
                 <xs:element name="DetectJokesEnabled" type="xs:int"></xs:element>
                 <xs:element name="DetectUnknownMacrosEnabled" type="xs:int"></xs:element>
                 <xs:element name="DetectUnknownProgramsEnabled" type="xs:int"></xs:element>
                 <xs:element name="DetectUnwantedProgramsEnabled" type="xs:int"></xs:element>
                 <xs:element name="FileExclusions">
                   <xs:complexType>
                     <xs:sequence>
                           <xs:element name="File" maxOccurs="unbounded" type="xs:string"></xs:element>
                     </xs:sequence>
                     <xs:attribute name="delimiter" type="xs:string"></xs:attribute>
                   </xs:complexType>
                 </xs:element>
                 <xs:element name="FileTypeToScan" type="xs:int"></xs:element>
                 <xs:element name="FileTypesToExclude">
                   <xs:complexType>
                     <xs:sequence>
                           <xs:element name="FileType" maxOccurs="unbounded" type="xs:string"></xs:element>
                     </xs:sequence>
                     <xs:attribute name="delimiter" type="xs:string"></xs:attribute>
                   </xs:complexType>
                 </xs:element>
                 <xs:element name="FoldersAndSubFoldersExclusion">
                   <xs:complexType>
                     <xs:sequence>
                           <xs:element name="Folder" maxOccurs="unbounded" type="xs:string"></xs:element>
                     </xs:sequence>
                     <xs:attribute name="delimiter" type="xs:string"></xs:attribute>
                   </xs:complexType>
                 </xs:element>
                 <xs:element name="GroupID" type="xs:int"></xs:element>
                 <xs:element name="InfectionPrimaryAction" type="xs:int"></xs:element>
                 <xs:element name="InfectionSecondaryAction" type="xs:int"></xs:element>
                 <xs:element name="LastRunSummary">
                   <xs:complexType>
                     <xs:all>
                           <xs:element name="ODSMajorDATVersion" type="xs:int"></xs:element>
                           <xs:element name="ODSMajorEngineVersion" type="xs:int"></xs:element>
                           <xs:element name="ODSMinorDATVersion" type="xs:int"></xs:element>
                           <xs:element name="ODSMinorEngineVersion" type="xs:int"></xs:element>
                           <xs:element name="NoOfCleanFiles" type="xs:int"></xs:element>
                           <xs:element name="NoOfFiles" type="xs:int"></xs:element>
                           <xs:element name="NoOfFilesCleaned" type="xs:int"></xs:element>
                           <xs:element name="NoOfFilesDeleted" type="xs:int"></xs:element>
                           <xs:element name="NoOfFilesInfected" type="xs:int"></xs:element>
                           <xs:element name="NoOfFilesSkipped" type="xs:int"></xs:element>
                           <xs:element name="NoOfCleanGTIFiles" minOccurs="0" type="xs:int"></xs:element>
                           <xs:element name="NoOfCacheHit" type="xs:int"></xs:element>
                           <xs:element name="NoOfFilesExcluded" type="xs:int"></xs:element>
                           <xs:element name="NoOfFilesNotScannedDueToError" type="xs:int"></xs:element>
                           <xs:element name="NoOfFilesNotScannedDueToTimeout" type="xs:int"></xs:element>
                           <xs:element name="OdsStatus" type="xs:int"></xs:element>
                           <xs:element name="TaskEndTime" type="xs:int"></xs:element>
                           <xs:element name="TaskStartTime" type="xs:int"></xs:element>
                           <xs:element name="TimeTaken" type="xs:float"></xs:element>
                     </xs:all>
                   </xs:complexType>
                 </xs:element>
                 <xs:element name="PUPPrimaryAction" type="xs:int"></xs:element>
                 <xs:element name="PUPSecondaryAction" type="xs:int"></xs:element>
                 <xs:element name="PathExclusions">
                   <xs:complexType>
                     <xs:sequence>
                           <xs:element name="Path" maxOccurs="unbounded" type="xs:string"></xs:element>
                     </xs:sequence>
                     <xs:attribute name="delimiter" type="xs:string"></xs:attribute>
                   </xs:complexType>
                 </xs:element>
                 <xs:element name="ScanArchiveEnabled" type="xs:int"></xs:element>
                 <xs:element name="ScanExtension" type="xs:int"></xs:element>
                 <xs:element name="ScanLocalDrives" type="xs:int"></xs:element>
                 <xs:element name="ScanMacros" type="xs:int"></xs:element>
                 <xs:element name="ScanMailEnabled" type="xs:int"></xs:element>
                 <xs:element name="ScanOnNetwork" type="xs:int"></xs:element>
                 <xs:element name="ScanPaths">
                   <xs:complexType>
                     <xs:sequence>
                           <xs:element name="Path" maxOccurs="unbounded" type="xs:string"></xs:element>
                     </xs:sequence>
                     <xs:attribute name="delimiter" type="xs:string"></xs:attribute>
                   </xs:complexType>
                 </xs:element>
                 <xs:element name="ScanSubFolder" type="xs:int"></xs:element>
                 <xs:element name="ScanTmpFolders" type="xs:int"></xs:element>
                 <xs:element name="TaskName" type="xs:string"></xs:element>
                 <xs:element name="TaskSource" type="xs:int"></xs:element>
                 <xs:element name="TaskSchedule"></xs:element>
                 <xs:element name="Threads" type="xs:int"></xs:element>
                 <xs:element name="MaxCPULimit"  minOccurs="0" type="xs:int"></xs:element>
                 <xs:element name="MaxScanTimeout" type="xs:int"></xs:element>
                 <xs:element name="IsGTIEnabled"  minOccurs="0" type="xs:int"></xs:element>
                 <xs:element name="GTISensitivityLevel" minOccurs="0" type="xs:int"></xs:element>
                 <xs:element name="UseScanCache" type="xs:int"></xs:element>
                 <xs:element name="UserID" type="xs:int"></xs:element>
               </xs:all>
               <xs:attribute name="type" type="xs:string"></xs:attribute>
         </xs:complexType>
       </xs:element>
   </xs:schema>
''')

valid_DNDOASGlobalSettings = StringIO(u'''\
   <xs:schema xmlns:xs="http://www.w3.org/2001/XMLSchema" elementFormDefault="qualified" attributeFormDefault="unqualified">
       <xs:element name="DNDOASGlobalSettings">
         <xs:complexType>
               <xs:all>
                 <xs:element name="EnableMultipleProfiles" type="xs:int"></xs:element>
                 <xs:element name="UseFanotify" type="xs:int" minOccurs="0"></xs:element>
                 <xs:element name="UseKernelModule" type="xs:int"></xs:element>
                 <xs:element name="IsFanotifySupported" type="xs:int" minOccurs="0"></xs:element>
                 <xs:element name="KallsymsLookUpNameAddress" type="xs:string" minOccurs="0"></xs:element>
                 <xs:element name="SyscallTableAddress" type="xs:string"></xs:element>
                 <xs:element name="SyscallTableAddress32Bit" type="xs:string" minOccurs="0"></xs:element>
                 <xs:element name="NetlinkRequestSocket" type="xs:int"></xs:element>
                 <xs:element name="NetlinkResponseSocket" type="xs:int"></xs:element>
                 <xs:element name="IsOASEnabled" type="xs:int"></xs:element>
                 <xs:element name="IsOASSupported" type="xs:int" minOccurs="0"></xs:element>
                 <xs:element name="MaxScanTimeout" type="xs:int"></xs:element>
                 <xs:element name="MaxOASCPULimit" type="xs:int" minOccurs="0"></xs:element>
                 <xs:element name="MaxValueOfScanTime" type="xs:int"></xs:element>
                 <xs:element name="MinValueOfScanTime" type="xs:int"></xs:element>
                 <xs:element name="MaxRequestQueueSize" type="xs:int"></xs:element>
                 <xs:element name="RequestSizeMonitorInterval" type="xs:int"></xs:element>
                 <xs:element name="RequestQueueReducePercentage" type="xs:int"></xs:element>
                 <xs:element name="ActionWhenRequestQueueOverflows" type="xs:string"></xs:element>
                 <xs:element name="OASExcludedFileSystems">
                   <xs:complexType>
                     <xs:sequence>
                           <xs:element name="FileSystem" maxOccurs="unbounded" type="xs:string"></xs:element>
                     </xs:sequence>
                     <xs:attribute name="delimiter" type="xs:string"></xs:attribute>
                   </xs:complexType>
                 </xs:element>
                 <xs:element name="OASPrefixKeyword"></xs:element>
                 <xs:element name="OASPrefixKeywordAutomatically" type="xs:int"></xs:element>
                 <xs:element name="ProfileName" type="xs:string"></xs:element>
                 <xs:element name="ScanManagerThreads" type="xs:int"></xs:element>
                 <xs:element name="EnableOASMgrWatchDog" type="xs:int"></xs:element>
                 <xs:element name="OASMgrWatchDogInterval" type="xs:int"></xs:element>
                 <xs:element name="OASMgrTimeoutInterval" type="xs:int"></xs:element>
                 <xs:element name="CacheMaxLimit" type="xs:int"></xs:element>
                 <xs:element name="CachePurgeLimit" type="xs:int"></xs:element>
                 <xs:element name="UseDeferredScan" type="xs:int" minOccurs="0"></xs:element>
               </xs:all>
               <xs:attribute name="type" type="xs:int"></xs:attribute>
         </xs:complexType>
       </xs:element>
   </xs:schema>
''')

valid_DNDGlobalSettings = StringIO(u'''\
   <xs:schema xmlns:xs="http://www.w3.org/2001/XMLSchema" elementFormDefault="qualified" attributeFormDefault="unqualified">
       <xs:element name="DNDGlobalSettings">
         <xs:complexType>
               <xs:all>
                 <xs:element name="ExtraDATVirusCount" type="xs:int"></xs:element>
                 <xs:element name="ExtraDATVirusList">
                   <xs:complexType>
                     <xs:sequence>
                           <xs:element name="VirusName" maxOccurs="unbounded" type="xs:string"></xs:element>
                     </xs:sequence>
                     <xs:attribute name="delimiter" type="xs:string"></xs:attribute>
                   </xs:complexType>
                 </xs:element>
                 <xs:element name="LastUpdatedTimeStamp"></xs:element>
                 <xs:element name="LicenseStatus" type="xs:int"></xs:element>
                 <xs:element name="MajorDATVersion" type="xs:int"></xs:element>
                 <xs:element name="IsGTIFeedbackEnabled" minOccurs="0" type="xs:int"></xs:element>
                 <xs:element name="ExtraDATVersion" type="xs:string"></xs:element>
                 <xs:element name="MajorEngineVersion" type="xs:int"></xs:element>
                 <xs:element name="MinorDATVersion" type="xs:int"></xs:element>
                 <xs:element name="MinorEngineVersion" type="xs:int"></xs:element>
                 <xs:element name="DATReleaseDay" type="xs:int" minOccurs="0"></xs:element>
                 <xs:element name="DATReleaseMonth" type="xs:int" minOccurs="0"></xs:element>
                 <xs:element name="DATReleaseYear" type="xs:int" minOccurs="0"></xs:element>
                 <xs:element name="DATUpdateTime" type="xs:string" minOccurs="0"></xs:element>
                 <xs:element name="MajorExploitPreventionContentVersion" type="xs:string" minOccurs="0"></xs:element>
                 <xs:element name="MinorExploitPreventionContentVersion" type="xs:string" minOccurs="0"></xs:element>
                 <xs:element name="ExploitPreventionContentReleaseDay" type="xs:string" minOccurs="0"></xs:element>
                 <xs:element name="ExploitPreventionContentReleaseMonth" type="xs:string" minOccurs="0"></xs:element>
                 <xs:element name="ExploitPreventionContentReleaseYear" type="xs:string" minOccurs="0"></xs:element>
                 <xs:element name="MountMonitorInterval" type="xs:int"></xs:element>
                 <xs:element name="QuarantineDirectory" type="xs:string"></xs:element>
                 <xs:element name="EnableDefaultClientUpdateTask" type="xs:int"></xs:element>
                 <xs:element name="MarkerDefaultClientUpdateTask" type="xs:int"></xs:element>
                 <xs:element name="ClientUpdateTaskScheduledTime" type="xs:string"></xs:element>
                 <xs:element name="UpdateTimeoutInterval" type="xs:int"></xs:element>
                 <xs:element name="IncrementalUpdateEngineVersion" type="xs:double"></xs:element>
                 <xs:element name="IsAPEnabled" type="xs:int" minOccurs="0"></xs:element>
                 <xs:element name="IsPostDeploymentStartUp" type="xs:int" minOccurs="0"></xs:element>
                 <xs:element name="IsAutoContentUpdateEnabled" type="xs:int" minOccurs="0"></xs:element>
                 <xs:element name="IsEPEnabled" type="xs:int" minOccurs="0"></xs:element>
               </xs:all>
               <xs:attribute name="type" type="xs:int"></xs:attribute>
         </xs:complexType>
       </xs:element>
   </xs:schema>
''')

valid_DNDClientLogging = StringIO(u'''\
   <xs:schema xmlns:xs="http://www.w3.org/2001/XMLSchema" elementFormDefault="qualified" attributeFormDefault="unqualified">
       <xs:element name="DNDClientLogging">
         <xs:complexType>
               <xs:all>
                 <xs:element name="ActivityLogSize" type="xs:int"></xs:element>
                 <xs:element name="DebugLogSize" type="xs:int"></xs:element>
                 <xs:element name="EnableActivityLogging" type="xs:int"></xs:element>
                 <xs:element name="EnableDebugLoggingForOAS" type="xs:int"></xs:element>
                 <xs:element name="EnableDebugLoggingForODS" type="xs:int"></xs:element>
                 <xs:element name="EnableEventLogging" type="xs:int"></xs:element>
                 <xs:element name="EnableOASActivityLogging" type="xs:int" minOccurs="0"></xs:element>
                 <xs:element name="EnableSendingEventsToePO" type="xs:int"></xs:element>
                 <xs:element name="LimitActivityLogSize" type="xs:int"></xs:element>
                 <xs:element name="LimitDebugLogSize" type="xs:int"></xs:element>
                 <xs:element name="LogLevelForOAS" type="xs:int"></xs:element>
                 <xs:element name="LogLevelForODS" type="xs:int"></xs:element>
                 <xs:element name="LogPath" type="xs:string"></xs:element>
                 <xs:element name="ePOLogLevelForOAS" type="xs:int"></xs:element>
                 <xs:element name="ePOLogLevelForODS" type="xs:int"></xs:element>
                 <xs:element name="EnableDebugLoggingForMsgBus" type="xs:int" minOccurs="0"></xs:element>
                 <xs:element name="EnableODSActivityLogging" type="xs:int" minOccurs="0"></xs:element>
               </xs:all>
               <xs:attribute name="type" type="xs:int"></xs:attribute>
         </xs:complexType>
       </xs:element>
   </xs:schema>
''')

valid_DNDProductInformation = StringIO(u'''\
   <xs:schema xmlns:xs="http://www.w3.org/2001/XMLSchema" elementFormDefault="qualified" attributeFormDefault="unqualified">
       <xs:element name="DNDProductInformation">
         <xs:complexType>
               <xs:all>
                 <xs:element name="ProductBinDir" type="xs:string"></xs:element>
                 <xs:element name="ProductCLIName" type="xs:string"></xs:element>
                 <xs:element name="ProductCopyright" type="xs:string"></xs:element>
                 <xs:element name="ProductInstallDir" type="xs:string"></xs:element>
                 <xs:element name="ProductName" type="xs:string"></xs:element>
                 <xs:element name="ProductNameForEvent" type="xs:string"></xs:element>
                 <xs:element name="ProductODSReportDir" type="xs:string"></xs:element>
                 <xs:element name="ProductVarDir" type="xs:string"></xs:element>
                 <xs:element name="ProductVersion" type="xs:string"></xs:element>
                 <xs:element name="VendorName" type="xs:string"></xs:element>
                 <xs:element name="HotFixes">
                   <xs:complexType>
                     <xs:sequence>
                           <xs:element name="HotFix"></xs:element>
                     </xs:sequence>
                   </xs:complexType>
                 </xs:element>
                 <xs:element name="Patch"></xs:element>
               </xs:all>
               <xs:attribute name="type" type="xs:int"></xs:attribute>
         </xs:complexType>
       </xs:element>
   </xs:schema>
''')

valid_DNDProductInformation1 = StringIO(u'''\
   <xs:schema xmlns:xs="http://www.w3.org/2001/XMLSchema" elementFormDefault="qualified" attributeFormDefault="unqualified">
       <xs:element name="DNDProductInformation">
         <xs:complexType>
               <xs:sequence>
                 <xs:element name="ProductBinDir" type="xs:string"></xs:element>
                 <xs:element name="ProductCLIName" type="xs:string"></xs:element>
                 <xs:element name="ProductCopyright" type="xs:string"></xs:element>
                 <xs:element name="ProductInstallDir" type="xs:string"></xs:element>
                 <xs:element name="ProductName" type="xs:string"></xs:element>
                 <xs:element name="ProductODSReportDir" type="xs:string"></xs:element>
                 <xs:element name="ProductVarDir" type="xs:string"></xs:element>
                 <xs:element name="ProductVersion" type="xs:string"></xs:element>
                 <xs:element name="VendorName" type="xs:string"></xs:element>
                 <xs:element name="HotFixes">
                   <xs:complexType>
                     <xs:sequence>
                           <xs:element name="HotFix" minOccurs="0"></xs:element>
                     </xs:sequence>
                   </xs:complexType>
                 </xs:element>
                 <xs:element name="Patch"></xs:element>
                 <xs:element name="ProductNameForEvent" maxOccurs="unbounded" minOccurs="0" type="xs:string"></xs:element>
               </xs:sequence>
               <xs:attribute name="type" type="xs:int"></xs:attribute>
         </xs:complexType>
       </xs:element>
   </xs:schema>
''')

valid_DNDProductInformation2 = StringIO(u'''\
   <xs:schema xmlns:xs="http://www.w3.org/2001/XMLSchema" elementFormDefault="qualified" attributeFormDefault="unqualified">
       <xs:element name="DNDProductInformation">
         <xs:complexType>
               <xs:all>
                 <xs:element name="ProductBinDir" type="xs:string"></xs:element>
                 <xs:element name="ProductCLIName" type="xs:string"></xs:element>
                 <xs:element name="ProductCopyright" type="xs:string"></xs:element>
                 <xs:element name="ProductInstallDir" type="xs:string"></xs:element>
                 <xs:element name="ProductName" type="xs:string"></xs:element>
                 <xs:element name="ProductNameForEvent" type="xs:string"></xs:element>
                 <xs:element name="ProductODSReportDir" type="xs:string"></xs:element>
                 <xs:element name="ProductVarDir" type="xs:string"></xs:element>
                 <xs:element name="ProductVersion" type="xs:string"></xs:element>
                 <xs:element name="DaysSinceInstallation" type="xs:int"></xs:element>
                 <xs:element name="VendorName" type="xs:string"></xs:element>
                 <xs:element name="GTIHostAddr" type="xs:string"></xs:element>
                 <xs:element name="GTIUserID" type="xs:string"></xs:element>
                 <xs:element name="GTIKey" type="xs:string"></xs:element>
                 <xs:element name="GTICACertFile" type="xs:string"></xs:element>
                 <xs:element name="GTIProductName" type="xs:string"></xs:element>
                 <xs:element name="HotFixes">
                   <xs:complexType>
                     <xs:sequence>
                           <xs:element name="HotFix" maxOccurs="unbounded"></xs:element>
                     </xs:sequence>
                   </xs:complexType>
                 </xs:element>
                 <xs:element name="Patch"></xs:element>
               </xs:all>
               <xs:attribute name="type" type="xs:int"></xs:attribute>
         </xs:complexType>
       </xs:element>
   </xs:schema>
''')

valid_DNDProductInformation3 = StringIO(u'''\
   <xs:schema xmlns:xs="http://www.w3.org/2001/XMLSchema" elementFormDefault="qualified" attributeFormDefault="unqualified">
       <xs:element name="DNDProductInformation">
         <xs:complexType>
               <xs:all>
                 <xs:element name="ProductBinDir" type="xs:string"></xs:element>
                 <xs:element name="ProductCLIName" type="xs:string"></xs:element>
                 <xs:element name="ProductCopyright" type="xs:string"></xs:element>
                 <xs:element name="ProductInstallDir" type="xs:string"></xs:element>
                 <xs:element name="ProductName" type="xs:string"></xs:element>
                 <xs:element name="ProductNameForEvent" type="xs:string"></xs:element>
                 <xs:element name="ProductODSReportDir" type="xs:string"></xs:element>
                 <xs:element name="ProductVarDir" type="xs:string"></xs:element>
                 <xs:element name="ProductVersion" type="xs:string"></xs:element>
                 <xs:element name="DaysSinceInstallation" type="xs:int"></xs:element>
                 <xs:element name="VendorName" type="xs:string"></xs:element>
                 <xs:element name="GTIHostAddr" type="xs:string"></xs:element>
                 <xs:element name="GTIUserID" type="xs:string"></xs:element>
                 <xs:element name="GTIKey" type="xs:string"></xs:element>
                 <xs:element name="GTICACertFile" type="xs:string"></xs:element>
                 <xs:element name="FIPSMode" type="xs:int" minOccurs="0"></xs:element>
                 <xs:element name="GTIFIPSHostAddr" type="xs:string"></xs:element>
                 <xs:element name="GTIFIPSUserID" type="xs:string"></xs:element>
                 <xs:element name="GTIFIPSKey" type="xs:string"></xs:element>
                 <xs:element name="GTIFIPSCACertFile" type="xs:string"></xs:element>
                 <xs:element name="GTIProductName" type="xs:string"></xs:element>
                 <xs:element name="HotFixes">
                   <xs:complexType>
                     <xs:sequence>
                           <xs:element name="HotFix" maxOccurs="unbounded"></xs:element>
                     </xs:sequence>
                   </xs:complexType>
                 </xs:element>
                 <xs:element name="Patch"></xs:element>
               </xs:all>
               <xs:attribute name="type" type="xs:int"></xs:attribute>
         </xs:complexType>
       </xs:element>
   </xs:schema>
''')

valid_DNDProductDefaultSettings = StringIO(u'''\
   <xs:schema xmlns:xs="http://www.w3.org/2001/XMLSchema" elementFormDefault="qualified" attributeFormDefault="unqualified">
       <xs:element name="DNDProductDefaultSettings">
         <xs:complexType>
               <xs:sequence>
                 <xs:element name="DefaultFileTypeToScan">
                   <xs:complexType>
                     <xs:sequence>
                           <xs:element name="FileType" maxOccurs="unbounded" type="xs:string"></xs:element>
                     </xs:sequence>
                     <xs:attribute name="delimiter" type="xs:string"></xs:attribute>
                   </xs:complexType>
                 </xs:element>
                 <xs:element name="ESPSocketPath" type="xs:string"></xs:element>
                 <xs:element name="LogFileForFM" type="xs:string"></xs:element>
                 <xs:element name="LogFileForOASMgr" type="xs:string"></xs:element>
                 <xs:element name="LogFileForODSCollector" type="xs:string"></xs:element>
                 <xs:element name="LogFileForODSMgr" type="xs:string"></xs:element>
                 <xs:element name="LogFileForScanActionMgr" type="xs:string"></xs:element>
                 <xs:element name="LogFileForScanFactory" type="xs:string"></xs:element>
                 <xs:element name="PrefixKeyword">
                   <xs:complexType>
                     <xs:sequence>
                           <xs:element name="KeyName" maxOccurs="unbounded">
                             <xs:complexType>
                            <xs:simpleContent>
                                <xs:extension base="xs:string">
                                    <xs:attribute name="value" type="xs:string"></xs:attribute>
                                </xs:extension>
                            </xs:simpleContent>
                             </xs:complexType>
                           </xs:element>
                     </xs:sequence>
                     <xs:attribute name="delimiter" type="xs:string"></xs:attribute>
                   </xs:complexType>
                 </xs:element>
                 <xs:element name="ProductMaxLogFileSize" type="xs:int"></xs:element>
                 <xs:element name="ProductMaxRotatedLogFileCount" type="xs:int"></xs:element>
               </xs:sequence>
               <xs:attribute name="type" type="xs:int"></xs:attribute>
         </xs:complexType>
       </xs:element>
   </xs:schema>
''')

valid_DNDODSGlobalSettings = StringIO(u'''\
   <xs:schema xmlns:xs="http://www.w3.org/2001/XMLSchema" elementFormDefault="qualified" attributeFormDefault="unqualified">
       <xs:element name="DNDODSGlobalSettings">
         <xs:complexType>
               <xs:sequence>
                 <xs:element name="MaxConcurrentODSLimit" type="xs:int"></xs:element>
               </xs:sequence>
               <xs:attribute name="type" type="xs:int"></xs:attribute>
         </xs:complexType>
       </xs:element>
   </xs:schema>
''')

valid_UpdateProfile = StringIO(u'''\
   <xs:schema xmlns:xs="http://www.w3.org/2001/XMLSchema" elementFormDefault="qualified" attributeFormDefault="unqualified">
       <xs:element name="UpdateProfile">
         <xs:complexType>
               <xs:sequence>
                 <xs:element name="TaskLastRun" type="xs:int"></xs:element>
                 <xs:element name="TaskName" type="xs:string"></xs:element>
                 <xs:element name="TaskOrigin" type="xs:int"></xs:element>
                 <xs:element name="TaskSchedule"></xs:element>
                 <xs:element name="TaskUpdateType" type="xs:int"></xs:element>
               </xs:sequence>
               <xs:attribute name="type" type="xs:string"></xs:attribute>
         </xs:complexType>
       </xs:element>
   </xs:schema>
''')

try:
    # Parse and create XMLSchema for valid entries
    xmlschema_doc_DNDOASProfile = ET.parse(valid_DNDOASProfile)
    xmlschema_doc_ODSProfile = ET.parse(valid_ODSProfile)
    xmlschema_doc_DNDOASGlobalSettings = ET.parse(valid_DNDOASGlobalSettings)
    xmlschema_doc_DNDGlobalSettings = ET.parse(valid_DNDGlobalSettings)
    xmlschema_doc_DNDClientLogging = ET.parse(valid_DNDClientLogging)
    xmlschema_doc_DNDProductInformation = ET.parse(valid_DNDProductInformation)
    xmlschema_doc_DNDProductInformation1 = ET.parse(valid_DNDProductInformation1)
    xmlschema_doc_DNDProductInformation2 = ET.parse(valid_DNDProductInformation2)
    xmlschema_doc_DNDProductInformation3 = ET.parse(valid_DNDProductInformation3)
    xmlschema_doc_DNDProductDefaultSettings = ET.parse(valid_DNDProductDefaultSettings)
    xmlschema_doc_DNDODSGlobalSettings = ET.parse(valid_DNDODSGlobalSettings)
    xmlschema_doc_UpdateProfile = ET.parse(valid_UpdateProfile)

    xmlschema_DNDOASProfile = ET.XMLSchema(xmlschema_doc_DNDOASProfile)
    xmlschema_ODSProfile = ET.XMLSchema(xmlschema_doc_ODSProfile)
    xmlschema_DNDOASGlobalSettings = ET.XMLSchema(xmlschema_doc_DNDOASGlobalSettings)
    xmlschema_DNDGlobalSettings = ET.XMLSchema(xmlschema_doc_DNDGlobalSettings)
    xmlschema_DNDClientLogging = ET.XMLSchema(xmlschema_doc_DNDClientLogging)
    xmlschema_DNDProductInformation = ET.XMLSchema(xmlschema_doc_DNDProductInformation)
    xmlschema_DNDProductInformation1 = ET.XMLSchema(xmlschema_doc_DNDProductInformation1)
    xmlschema_DNDProductInformation2 = ET.XMLSchema(xmlschema_doc_DNDProductInformation2)
    xmlschema_DNDProductInformation3 = ET.XMLSchema(xmlschema_doc_DNDProductInformation3)
    xmlschema_DNDProductDefaultSettings = ET.XMLSchema(xmlschema_doc_DNDProductDefaultSettings)
    xmlschema_DNDODSGlobalSettings = ET.XMLSchema(xmlschema_doc_DNDODSGlobalSettings)
    xmlschema_UpdateProfile = ET.XMLSchema(xmlschema_doc_UpdateProfile)

    # Parse prefs.xml file that was passed for validation
    prefsxml = ET.parse(file_to_validate)
    # Validate all OAS Profiles
    prefsxml_DNDOASProfile_list = prefsxml.findall("DNDOASProfile")
    for prefsxml_DNDOASProfile in prefsxml_DNDOASProfile_list:
        xmlschema_DNDOASProfile.assertValid(prefsxml_DNDOASProfile)
    # Validate all ODS tasks
    prefsxml_ODSProfile_list = prefsxml.findall("ODSProfile")
    for prefsxml_ODSProfile in prefsxml_ODSProfile_list:
        xmlschema_ODSProfile.assertValid(prefsxml_ODSProfile)
    # Validate OAS Global Settings
    prefsxml_DNDOASGlobalSettings = prefsxml.find("DNDOASGlobalSettings")
    xmlschema_DNDOASGlobalSettings.assertValid(prefsxml_DNDOASGlobalSettings)
    # Validate Global Settings
    prefsxml_DNDGlobalSettings = prefsxml.find("DNDGlobalSettings")
    xmlschema_DNDGlobalSettings.assertValid(prefsxml_DNDGlobalSettings)
    # Validate Client Logging
    prefsxml_DNDClientLogging = prefsxml.find("DNDClientLogging")
    xmlschema_DNDClientLogging.assertValid(prefsxml_DNDClientLogging)
    # Validate Product Information
    prefsxml_DNDProductInformation = prefsxml.find("DNDProductInformation")
    try:
        xmlschema_DNDProductInformation.assertValid(prefsxml_DNDProductInformation)
    except:
        try:
            # Allowing schema which has "ProductNameForEvent" tag twice in prefs_xml
            xmlschema_DNDProductInformation1.assertValid(prefsxml_DNDProductInformation)
        except:
            try:
                # Allowing schema which has GTI tags in prefs_xml.
                xmlschema_DNDProductInformation2.assertValid(prefsxml_DNDProductInformation)
            except:
                # Allowing schema which has new FIPS tags in prefs_xml.
                xmlschema_DNDProductInformation3.assertValid(prefsxml_DNDProductInformation)

    # Validate Product Default Settings
    prefsxml_DNDProductDefaultSettings = prefsxml.find("DNDProductDefaultSettings")
    xmlschema_DNDProductDefaultSettings.assertValid(prefsxml_DNDProductDefaultSettings)
    # Validate ODS Global Settings
    prefsxml_DNDODSGlobalSettings = prefsxml.find("DNDODSGlobalSettings")
    xmlschema_DNDODSGlobalSettings.assertValid(prefsxml_DNDODSGlobalSettings)
    # Validate Update Profile
    prefsxml_UpdateProfile_list = prefsxml.findall("UpdateProfile")
    for prefsxml_UpdateProfile in prefsxml_UpdateProfile_list:
        xmlschema_UpdateProfile.assertValid(prefsxml_UpdateProfile)
    # exit with 0 after validation has passed
except:
    # exit with 1 if parsing or validation failed. print str(e) to see exception details
    sys.exit(1)
