#!/usr/bin/python
# Copyright (C) 2016-2022 Musarubra US LLC. All Rights Reserved.

import sys
import os
import logging
import shutil
import re
import subprocess
import binascii
try:
    import xml.etree.ElementTree as ET
except:
    import lxml.etree as ET


configFile = '/var/McAfee/ens/tp/prefs.xml'
datDir = '/var/McAfee/ens/tp/dat'
engineDir = '/opt/McAfee/ens/tp/var/engine/lib/64'
engine32Dir = '/opt/McAfee/ens/tp/var/engine/lib/32/'
epContentDir = '/var/McAfee/ens/tp/ep/content'
epRuleDb = '/var/McAfee/ens/tp/ep/epcombinedrules.db'
apRuleDb = '/var/McAfee/ens/tp/aprulecache.db'
binPath7za = '/opt/McAfee/ens/esp/bin/7za'
logFilePath = '/var/McAfee/ens/log/tp/'
hotFixNumber = '1158046'
patchNumber = '1'
isHotFixPackage = False
isPatchPackage = False
copyRight = 'Copyright (C) 2022 Musarubra US LLC. All Rights Reserved.'
productNameForEvent = 'Trellix Endpoint Security'
majorEngineNumber = '6600'
minorEngineNumber = '9927'
GTIHostAddr='enslinux.rest.gti.trellix.com'
GTIUserID='aabe41b031d87a89a85278b783083ce23bbb5c875c9de90d59f68ecc7007b93afecb1dd125aa03774981dd19915be957'
GTIKey='3d2299d8cecb4717164f0f60b578f098e3378fe14a216087243458b56b59f7e8d09ad4aa385cce0985bd2a6fb1b517fb415eda39bcd42145366a6e9d74648aec'
GTICACertFile='production.crt'
FIPSMode='0'
GTIFIPSHostAddr=''
GTIFIPSUserID=''
GTIFIPSKey=''
GTIFIPSCACertFile='production_gov.crt'
GTIProductName='ENS Linux'
majorEPContentVersion = '10.7.0'
minorEPContentVersion = '00079'
# This is the maximum bucket count that can be for the serialised EP or AP DB file
maxDBBucketCount = 1000

# Use the following constants as argument to updateConfig() function.

# Use for updating value of existing tag, it is not recommended to use this option if you are not 100 % sure if the tag is present in xml
update_Tag_If_Present = 0
# Use for adding new tag, kindly do not use this option as it causes duplicate tag addition on multiple upgrades, use option "update_tag_if_present_else_add_tag" instead
add_New_Tag = 1
# Use for clearing the value of a key without deleting it
clear_Tag = 2
# Use for deleting the key from xml
delete_Tag = 3
# Use to update the value of key if its present in xml else adds the key to xml
update_Tag_If_Present_Else_Add_Tag = 4
# Use to add the new tag and value if the tag name and associated value is not present in the xml
add_New_Tag_If_Key_Value_Not_Present = 5
# Use to add the tag if it is not present in the xml, do nothing if the key is present
add_New_Tag_With_Attributes_If_Not_Present = 6
# Use for deleting the key value pair from xml
delete_Tag_With_Specific_Value = 7
# Used to update the attributes of the tag as well as the value if the tag is present, otherwise adds the new tag with the value and attributes
update_Tag_With_Attributes_If_Present_Else_Add_Tag = 8

def executeCmd(args, shellFlag=False, waitForOutput=True, env=None):
    """
    Method to execute system command and return response.
    This method returns a tuple on execution.
    If the command execution is successful, returns a tuple (0,result, error)
    If the command execution is unsuccessful, returns a tuple (resultCode, result, error )
    """
    try:
        output = subprocess.Popen(args, stdout=subprocess.PIPE, stderr=subprocess.PIPE, shell=shellFlag, env=env)
        if not waitForOutput:
            return (0, "", "")
        (res, err) = output.communicate()
        rescode = output.poll()
        if rescode != 0:
            logging.debug("Response code of execution of command is %d" % rescode)
            logging.debug("Response string is %s" % err.strip())
        return (rescode, res, err)
    except:
        _e = sys.exc_info()[1]
        logging.exception("Exception occurred while executing %s" % _e)
        return (1, "Failed to execute", "Fail")

def usage():
    '''
    Usage/Help
    '''
    print(sys.argv[0] +" <post> </absolute/path/to/backup-dir> <RPM/DEB>")

def updateProxySettingTags():
    updateConfig("DNDProxySettings", "0", add_New_Tag_With_Attributes_If_Not_Present, 'ProxyCustomIsEnabled')
    updateConfig("DNDProxySettings", "", add_New_Tag_With_Attributes_If_Not_Present, 'ProxyCustomPort')
    updateConfig("DNDProxySettings", "", add_New_Tag_With_Attributes_If_Not_Present, 'ProxyCustomServer')
    updateConfig("DNDProxySettings", "", add_New_Tag_With_Attributes_If_Not_Present, 'ProxyExclusions', "delimiter",',')
    updateConfig("DNDProxySettings/ProxyExclusions", "", add_New_Tag_With_Attributes_If_Not_Present, "ProxyExclusion")
    updateConfig("DNDProxySettings", "0", add_New_Tag_With_Attributes_If_Not_Present, 'ProxyHttpAuthIsEnabled')
    updateConfig("DNDProxySettings", "", add_New_Tag_With_Attributes_If_Not_Present, 'ProxyHttpAuthPassword')
    updateConfig("DNDProxySettings", "", add_New_Tag_With_Attributes_If_Not_Present, 'ProxyHttpAuthUserName')
    updateConfig("DNDProxySettings", "0", add_New_Tag_With_Attributes_If_Not_Present, 'ProxyIsEnabled')

def updateAPGlobalExclusionsTags():
    updateConfig("DNDAPGlobalExclusions", "", add_New_Tag_With_Attributes_If_Not_Present, 'APExclusions', "delimiter",',')
    updateConfig("DNDAPGlobalExclusions/APExclusions", "/opt/McAfee/cma/bin/cmdagent", add_New_Tag_If_Key_Value_Not_Present, 'APExclusion')
    updateConfig("DNDAPGlobalExclusions/APExclusions", "/opt/McAfee/cma/bin/masvc", add_New_Tag_If_Key_Value_Not_Present, 'APExclusion')
    updateConfig("DNDAPGlobalExclusions/APExclusions", "/opt/McAfee/cma/bin/macmnsvc", add_New_Tag_If_Key_Value_Not_Present, 'APExclusion')
    updateConfig("DNDAPGlobalExclusions/APExclusions", "/opt/McAfee/cma/bin/macompatsvc", add_New_Tag_If_Key_Value_Not_Present, 'APExclusion')
    updateConfig("DNDAPGlobalExclusions/APExclusions", "/opt/McAfee/cma/bin/maconfig", add_New_Tag_If_Key_Value_Not_Present, 'APExclusion')
    updateConfig("DNDAPGlobalExclusions/APExclusions", "/opt/McAfee/cma/bin/msaconfig", add_New_Tag_If_Key_Value_Not_Present, 'APExclusion')
    updateConfig("DNDAPGlobalExclusions/APExclusions", "/opt/McAfee/cma/bin/Mue", add_New_Tag_If_Key_Value_Not_Present, 'APExclusion')
    updateConfig("DNDAPGlobalExclusions/APExclusions", "/opt/McAfee/cma/bin/Mue_InUse", add_New_Tag_If_Key_Value_Not_Present, 'APExclusion')
    updateConfig("DNDAPGlobalExclusions/APExclusions", "/opt/McAfee/agent/bin/Mue", add_New_Tag_If_Key_Value_Not_Present, 'APExclusion')
    updateConfig("DNDAPGlobalExclusions/APExclusions", "/opt/McAfee/agent/bin/Mue_InUse", add_New_Tag_If_Key_Value_Not_Present, 'APExclusion')
    updateConfig("DNDAPGlobalExclusions/APExclusions", "/opt/McAfee/agent/bin/cmdagent", add_New_Tag_If_Key_Value_Not_Present, 'APExclusion')
    updateConfig("DNDAPGlobalExclusions/APExclusions", "/opt/McAfee/agent/bin/ma_msgbus_auth.sig", add_New_Tag_If_Key_Value_Not_Present, 'APExclusion')
    updateConfig("DNDAPGlobalExclusions/APExclusions", "/opt/McAfee/agent/bin/macmnsvc", add_New_Tag_If_Key_Value_Not_Present, 'APExclusion')
    updateConfig("DNDAPGlobalExclusions/APExclusions", "/opt/McAfee/agent/bin/macompatsvc", add_New_Tag_If_Key_Value_Not_Present, 'APExclusion')
    updateConfig("DNDAPGlobalExclusions/APExclusions", "/opt/McAfee/agent/bin/maconfig", add_New_Tag_If_Key_Value_Not_Present, 'APExclusion')
    updateConfig("DNDAPGlobalExclusions/APExclusions", "/opt/McAfee/agent/bin/masvc", add_New_Tag_If_Key_Value_Not_Present, 'APExclusion')
    updateConfig("DNDAPGlobalExclusions/APExclusions", "/opt/McAfee/agent/bin/msaconfig", add_New_Tag_If_Key_Value_Not_Present, 'APExclusion')
    updateConfig("DNDAPGlobalExclusions/APExclusions", "/opt/McAfee/agent/bin/policy_upgrade", add_New_Tag_If_Key_Value_Not_Present, 'APExclusion')
    updateConfig("DNDAPGlobalExclusions/APExclusions", "/opt/McAfee/ens/tp/bin/mfetpd", add_New_Tag_If_Key_Value_Not_Present, 'APExclusion')
    updateConfig("DNDAPGlobalExclusions/APExclusions", "/opt/McAfee/ens/tp/bin/mfetpcli", add_New_Tag_If_Key_Value_Not_Present, 'APExclusion')
    updateConfig("DNDAPGlobalExclusions/APExclusions", "/opt/McAfee/ens/esp/bin/mfeespd", add_New_Tag_If_Key_Value_Not_Present, 'APExclusion')
    updateConfig("DNDAPGlobalExclusions/APExclusions", "/opt/McAfee/ens/fw/bin/mfefwd", add_New_Tag_If_Key_Value_Not_Present, 'APExclusion')
    updateConfig("DNDAPGlobalExclusions/APExclusions", "/opt/McAfee/ens/fw/bin/mfefwcli", add_New_Tag_If_Key_Value_Not_Present, 'APExclusion')
    updateConfig("DNDAPGlobalExclusions/APExclusions", "/opt/McAfee/ens/esp/bin/7za", add_New_Tag_If_Key_Value_Not_Present, 'APExclusion')
    updateConfig("DNDAPGlobalExclusions/APExclusions", "/usr/local/mcafee/solidcore/bin/scsrvc", add_New_Tag_If_Key_Value_Not_Present, 'APExclusion')
    updateConfig("DNDAPGlobalExclusions/APExclusions", "/usr/local/mcafee/solidcore/bin/scmapl", add_New_Tag_If_Key_Value_Not_Present, 'APExclusion')
    updateConfig("DNDAPGlobalExclusions/APExclusions", "/usr/local/mcafee/solidcore/bin/sadmin", add_New_Tag_If_Key_Value_Not_Present, 'APExclusion')
    updateConfig("DNDAPGlobalExclusions/APExclusions", "/opt/McAfee/mvedr/mfemvedr", add_New_Tag_If_Key_Value_Not_Present, 'APExclusion')
    updateConfig("DNDAPGlobalExclusions/APExclusions", "/opt/McAfee/mvedr/trace/bin/mvedrtrace", add_New_Tag_If_Key_Value_Not_Present, 'APExclusion')
    updateConfig("DNDAPGlobalExclusions/APExclusions", "/opt/McAfee/Mar/MarService", add_New_Tag_If_Key_Value_Not_Present, 'APExclusion')
    updateConfig("DNDAPGlobalExclusions/APExclusions", "/opt/isec/ens/threatprevention/bin/isectpd", delete_Tag_With_Specific_Value, 'APExclusion')
    updateConfig("DNDAPGlobalExclusions/APExclusions", "/opt/isec/ens/threatprevention/bin/isecav", delete_Tag_With_Specific_Value, 'APExclusion')
    updateConfig("DNDAPGlobalExclusions/APExclusions", "/opt/isec/ens/esp/bin/isecespd", delete_Tag_With_Specific_Value, 'APExclusion')
    updateConfig("DNDAPGlobalExclusions/APExclusions", "/opt/isec/ens/esp/bin/7za", delete_Tag_With_Specific_Value, 'APExclusion')
    updateConfig("DNDAPGlobalExclusions/APExclusions", "/opt/McAfee/mfw/bin/mfefirewall", delete_Tag_With_Specific_Value, 'APExclusion')
    updateConfig("DNDAPGlobalExclusions/APExclusions", "/opt/McAfee/mfw/bin/mfw", delete_Tag_With_Specific_Value, 'APExclusion')

def updateGTISelectionCriteriaTags():
    # Add tag "IsGTIEnabled"
    updateConfig("DNDGTISelectionCriteria", "1", add_New_Tag_With_Attributes_If_Not_Present, 'IsGTIEnabled')
    updateConfig("DNDGTISelectionCriteria", "3", add_New_Tag_With_Attributes_If_Not_Present, 'GTISensitivityLevel')
    # Add tag "HighLevelCriteria" tag and its subtags
    updateConfig("DNDGTISelectionCriteria", "", add_New_Tag_With_Attributes_If_Not_Present, 'HighLevelCriteria', "delimiter",',')
    updateConfig("DNDGTISelectionCriteria/HighLevelCriteria", "application/x-sharedlib", add_New_Tag_If_Key_Value_Not_Present, "FileType")
    updateConfig("DNDGTISelectionCriteria/HighLevelCriteria", "application/x-executable", add_New_Tag_If_Key_Value_Not_Present, "FileType")
    updateConfig("DNDGTISelectionCriteria/HighLevelCriteria", "application/x-object", add_New_Tag_If_Key_Value_Not_Present, "FileType")
    updateConfig("DNDGTISelectionCriteria/HighLevelCriteria", "application/x-rpm", add_New_Tag_If_Key_Value_Not_Present, "FileType")
    updateConfig("DNDGTISelectionCriteria/HighLevelCriteria", "application/vnd.debian.binary-package", add_New_Tag_If_Key_Value_Not_Present, "FileType")
    updateConfig("DNDGTISelectionCriteria/HighLevelCriteria", "application/gzip", add_New_Tag_If_Key_Value_Not_Present, "FileType")
    updateConfig("DNDGTISelectionCriteria/HighLevelCriteria", "application/gzip", delete_Tag, "FileType")
    updateConfig("DNDGTISelectionCriteria/HighLevelCriteria", "application/x-gzip", add_New_Tag_If_Key_Value_Not_Present, "FileType")
    updateConfig("DNDGTISelectionCriteria/HighLevelCriteria", "application/x-tar", add_New_Tag_If_Key_Value_Not_Present, "FileType")
    updateConfig("DNDGTISelectionCriteria/HighLevelCriteria", "application/zip", add_New_Tag_If_Key_Value_Not_Present, "FileType")
    updateConfig("DNDGTISelectionCriteria/HighLevelCriteria", "text/x-shellscript", add_New_Tag_If_Key_Value_Not_Present, "FileType")
    updateConfig("DNDGTISelectionCriteria/HighLevelCriteria", "text/x-python", add_New_Tag_If_Key_Value_Not_Present, "FileType")
    updateConfig("DNDGTISelectionCriteria/HighLevelCriteria", "text/x-ruby", add_New_Tag_If_Key_Value_Not_Present, "FileType")
    updateConfig("DNDGTISelectionCriteria/HighLevelCriteria", "text/x-perl", add_New_Tag_If_Key_Value_Not_Present, "FileType")
    updateConfig("DNDGTISelectionCriteria/HighLevelCriteria", "text/x-php", add_New_Tag_If_Key_Value_Not_Present, "FileType")
    # Add tag "SelCriteria2" tag and its subtags
    updateConfig("DNDGTISelectionCriteria", "", add_New_Tag_With_Attributes_If_Not_Present, 'SelCriteria2', "delimiter",',')
    updateConfig("DNDGTISelectionCriteria/SelCriteria2", "/var/www/html", add_New_Tag_If_Key_Value_Not_Present, "Folder")
    updateConfig("DNDGTISelectionCriteria/SelCriteria2", "/usr/share/nginx/www", add_New_Tag_If_Key_Value_Not_Present, "Folder")
    # Add tag "SelCriteria3" tag and its subtags
    updateConfig("DNDGTISelectionCriteria", "", add_New_Tag_With_Attributes_If_Not_Present, 'SelCriteria3', "delimiter",',')
    updateConfig("DNDGTISelectionCriteria/SelCriteria3", "tmpfs", add_New_Tag_If_Key_Value_Not_Present, "FileSystem")
    updateConfig("DNDGTISelectionCriteria/SelCriteria3", "ramfs", add_New_Tag_If_Key_Value_Not_Present, "FileSystem")
    # Add tag "SelCriteria4" tag and its subtags
    updateConfig("DNDGTISelectionCriteria", "", add_New_Tag_With_Attributes_If_Not_Present, 'SelCriteria4', "delimiter",',')
    updateConfig("DNDGTISelectionCriteria/SelCriteria4", "/tmp", add_New_Tag_If_Key_Value_Not_Present, "Folder")
    updateConfig("DNDGTISelectionCriteria/SelCriteria4", "/var/tmp", add_New_Tag_If_Key_Value_Not_Present, "Folder")
    # Add tag "SelCriteria8" tag and its subtags
    updateConfig("DNDGTISelectionCriteria", "", add_New_Tag_With_Attributes_If_Not_Present, 'SelCriteria8', "delimiter",',')
    updateConfig("DNDGTISelectionCriteria/SelCriteria8", "doc", add_New_Tag_If_Key_Value_Not_Present, "FileType")
    updateConfig("DNDGTISelectionCriteria/SelCriteria8", "docx", add_New_Tag_If_Key_Value_Not_Present, "FileType")
    updateConfig("DNDGTISelectionCriteria/SelCriteria8", "xls", add_New_Tag_If_Key_Value_Not_Present, "FileType")
    updateConfig("DNDGTISelectionCriteria/SelCriteria8", "xlsx", add_New_Tag_If_Key_Value_Not_Present, "FileType")
    updateConfig("DNDGTISelectionCriteria/SelCriteria8", "pdf", add_New_Tag_If_Key_Value_Not_Present, "FileType")
    updateConfig("DNDGTISelectionCriteria/SelCriteria8", "zip", add_New_Tag_If_Key_Value_Not_Present, "FileType")
    updateConfig("DNDGTISelectionCriteria/SelCriteria8", "scr", add_New_Tag_If_Key_Value_Not_Present, "FileType")
    # Add tag "ExecutingTypes" tag and its subtags
    updateConfig("DNDGTISelectionCriteria", "", add_New_Tag_With_Attributes_If_Not_Present, 'ExecutingTypes', "delimiter",',')
    updateConfig("DNDGTISelectionCriteria/ExecutingTypes", "application/x-sharedlib", add_New_Tag_If_Key_Value_Not_Present, "FileType")
    updateConfig("DNDGTISelectionCriteria/ExecutingTypes", "application/x-executable", add_New_Tag_If_Key_Value_Not_Present, "FileType")
    updateConfig("DNDGTISelectionCriteria/ExecutingTypes", "application/x-object", add_New_Tag_If_Key_Value_Not_Present, "FileType")
    updateConfig("DNDGTISelectionCriteria/ExecutingTypes", "text/x-shellscript", add_New_Tag_If_Key_Value_Not_Present, "FileType")
    updateConfig("DNDGTISelectionCriteria/ExecutingTypes", "text/x-python", add_New_Tag_If_Key_Value_Not_Present, "FileType")
    updateConfig("DNDGTISelectionCriteria/ExecutingTypes", "text/x-ruby", add_New_Tag_If_Key_Value_Not_Present, "FileType")
    updateConfig("DNDGTISelectionCriteria/ExecutingTypes", "text/x-perl", add_New_Tag_If_Key_Value_Not_Present, "FileType")
    # Add tag "ELFTypes" tag and its subtags
    updateConfig("DNDGTISelectionCriteria", "", add_New_Tag_With_Attributes_If_Not_Present, 'ELFTypes', "delimiter",',')
    updateConfig("DNDGTISelectionCriteria/ELFTypes", "application/x-sharedlib", add_New_Tag_If_Key_Value_Not_Present, "FileType")
    updateConfig("DNDGTISelectionCriteria/ELFTypes", "application/x-executable", add_New_Tag_If_Key_Value_Not_Present, "FileType")
    updateConfig("DNDGTISelectionCriteria/ELFTypes", "application/x-object", add_New_Tag_If_Key_Value_Not_Present, "FileType")
    # Add tag "GTIThrottling" tag and its subtags
    updateConfig("DNDGTISelectionCriteria", "", add_New_Tag_With_Attributes_If_Not_Present, 'GTIThrottling')
    updateConfig("GTIThrottling", "", delete_Tag, "IsThrottlingEnabled")
    updateConfig("DNDGTISelectionCriteria/GTIThrottling", "200", add_New_Tag_With_Attributes_If_Not_Present, "Threshold")
    updateConfig("DNDGTISelectionCriteria/GTIThrottling", "0", add_New_Tag_With_Attributes_If_Not_Present, "NumHitsToday")
    updateConfig("DNDGTISelectionCriteria/GTIThrottling", "0", add_New_Tag_With_Attributes_If_Not_Present, "NumConsumedQuota")
    updateConfig("GTIThrottling", "", delete_Tag, "AverageHits")
    # Add tag "LastGTIParamsUpdate"
    updateConfig("DNDGTISelectionCriteria", "0", add_New_Tag_With_Attributes_If_Not_Present, 'LastGTIParamsUpdate')
    # Add tag "GTIQueryConfigurations" tag and its subtags
    updateConfig("DNDGTISelectionCriteria", "", add_New_Tag_With_Attributes_If_Not_Present, 'GTIQueryConfigurations')
    updateConfig("DNDGTISelectionCriteria/GTIQueryConfigurations", "5", add_New_Tag_With_Attributes_If_Not_Present, "NumberOfRequestsInAQuery")
    updateConfig("DNDGTISelectionCriteria/GTIQueryConfigurations", "0", add_New_Tag_With_Attributes_If_Not_Present, "AverageSSLHandShakeTimeForLast24Hours")
    updateConfig("DNDGTISelectionCriteria/GTIQueryConfigurations", "0", add_New_Tag_With_Attributes_If_Not_Present, "AverageRetriesForAQueryForLast24Hours")
    updateConfig("DNDGTISelectionCriteria/GTIQueryConfigurations", "0", add_New_Tag_With_Attributes_If_Not_Present, "AverageRTTForLast24Hours")
    updateConfig("DNDGTISelectionCriteria/GTIQueryConfigurations", "0", add_New_Tag_With_Attributes_If_Not_Present, "NumberOfGTIQueriesForLast24Hours")

# getDirList function return list of directory in the path passed as argument
def getDirList(dirName):
    logging.debug("Checking directory : " + dirName)
    arrayDirList = []
    dirs = os.listdir(dirName)
    logging.debug("ArrayDir List contain:")
    for fileList in dirs:
        absolutePath = os.path.join(dirName, fileList)
        if os.path.isdir(absolutePath):
            if fileList[0].isdigit():
                # Store folder name in array
                arrayDirList.append(fileList)
                logging.debug(fileList)

    return arrayDirList


# getValidVersion function return latest valid DAT/Engine version.
# It do check of fileCount and Size in directory passed.
def getValidVersion(arrayDirList, dirName, validFileCount = 3):
    # Sort array of DAT/Engine list in descending order
    arrayDirList.sort(key=float, reverse=True)
    dirNeedtoReplace = 0

    # Iterate arrayDirList and validate size and count
    for i in range(len(arrayDirList)):
        isSizeZero = 0
        fileCount = 0
        folderFiles = os.listdir(os.path.join(dirName, arrayDirList[i]))
        logging.debug("Checking if folder contains files: " + arrayDirList[i])
        logging.debug(folderFiles)

        # Iterate DAT/Engine folder
        for dirFile in folderFiles:
            absolutePath = os.path.join(dirName, arrayDirList[i], dirFile)
            sizeFile = os.path.getsize(os.path.abspath(absolutePath))

            if sizeFile == 0:
                isSizeZero = 1

            fileCount = fileCount + 1

        if isSizeZero == 1:
            continue
        elif fileCount < validFileCount:
            continue
        else:
            dirNeedtoReplace = arrayDirList[i]
            break

    return dirNeedtoReplace

# This method is used to fix the engine symlink path which might be broken during upgrade
# from versions with older engine paths like pre 10.6.6 or versions between 10.6.6 to 10.6.9.
def fixEngineSymlink(engine64Ver, engine32Ver):
    _cmd = 'ln -sf ' + engineDir + '/' + engine64Ver + '/liblnxfv.so.4 ' + engineDir + '/' + engine64Ver + '/liblnxfv.so'
    shellFlag = True
    _ret, _output, _err = executeCmd(_cmd, shellFlag)
    _cmd = 'ln -sf ' + engine32Dir + '/' + engine32Ver + '/liblnxfv.so.4 ' + engine32Dir + '/' + engine32Ver + '/liblnxfv.so'
    _ret, _output, _err = executeCmd(_cmd, shellFlag)

# This method can be used to add a new tag in prefs.xml.
# @param newTagName Tag which needs to be added in prefs.xml
# @value value for the new tag.
# @param attributeName this would represent the attribute for that tag
# @param attributeValue this would represent the attribute value for that tag
def addNewTag(newTagName, value='', attributeName='', attributeValue=''):
    try:
        tree = ET.parse(configFile)
        root = tree.getroot()
        checkTag = root.find(newTagName)
        if checkTag is None:
            newtag = ET.SubElement(root, newTagName)
            newtag.set(attributeName, attributeValue)
            newtag.text = str(value)
            tree.write(configFile)
        else:
            logging.debug(newTagName + " tag is already present in prefs.xml, hence not adding it again")
    except:
        _e = sys.exc_info()[1]
        logging.exception("%s" %str(_e))
        logging.error("Exception occurred while adding new tag in prefs.xml")
        sys.exit(0)

# This method can be used to add a new tag recursively in all the occurrences of a parent tag in prefs.xml.
# @param parentTagName Tag for which a child tag needs to be added in prefs.xml
# @param newTagName Tag which needs to be added in prefs.xml
# @value value for the new tag.
# @param attributeName this would represent the attribute for that tag
# @param attributeValue this would represent the attribute value for that tag
def addNewTagRecursive(parentTagName, newTagName, value='', attributeName='', attributeValue=''):
    tree = ET.parse(configFile)
    root = tree.getroot()
    tag = root.find(parentTagName)
    checkTag = tag.find(newTagName)
    if checkTag is None:
        try:
            for parent in root.iter(parentTagName):
                if attributeName == "":
                    attribute = {}
                else:
                    attribute = {attributeName:attributeValue}
                newtag = ET.SubElement(parent, newTagName, attrib=attribute)
                newtag.text = str(value)
                # This is required so that new tags are added in a new line.
                newtag.tail = "\n "
        except:
            for parent in root.getiterator(parentTagName):
                if attributeName == "":
                    attribute = {}
                else:
                    attribute = {attributeName:attributeValue}
                newtag = ET.SubElement(parent, newTagName, attrib=attribute)
                newtag.text = str(value)
                # This is required so that new tags are added in a new line.
                newtag.tail = "\n "
        tree.write(configFile)
    else:
        logging.debug( str(newTagName) + " already present as a child tag of " + str(parentTagName) + " hence not adding it in prefs.xml")

# This method can be used to update pref file configuration settings. Currently it does supporting of
# updating value of a existing tag.
# @param replaceTag Tag which value needs to be modified
# @value value to be updated
# @param updateType this would represent as the type of update like updation of value of a existing tag,
# addition of new tag, deletion of any tag. Pass 0 for updating value of existing tag, 1 for adding new tag
# 2 for clearing the value of a key without deleting it, 3 for deleting the key from xml
# 4 to update the value of key if its present in xml else adds the key to xml
# 5 to add the new tag and value if the key value pair do not match
# 6 to add the key if it is not present, do nothing if the key is present
# This has kept as default value to avoid passing in case of not required.
# @param newTag new tags to be added, this has kept as default parameter to avoid passing in case of not required
# @param attributeName this would represent the attribute for that tag
# @param attributeValue this would represent the attribute value for that tag
# Important - Please never use the updateType = 1 ,always use updateType = 6.
# Using updateType = 1 leads to addition of duplicate tags, hence using option 6 is a better and a safer option.
def updateConfig(replaceTag, value, updateType=0, newTag='', attributeName='', attributeValue=''):
    try:
        if updateType == 0:
            tree = ET.parse(configFile)
            root = tree.getroot()
            tag = root.find(replaceTag)
            tag.text = str(value)
            tree.write(configFile)

        '''writes to xml in a flat structure, the indentation is taken care by boost property tree
        when the process starts and something is written to the xml '''
        if updateType == 1:
            tree = ET.parse(configFile)
            root = tree.getroot()
            tag = root.find(replaceTag)
            subTag = ET.SubElement(tag, newTag, attrib={})
            subTag.text = str(value)
            tree.write(configFile)
        #in this section it clears all the sub elements under replaceTag value
        if updateType == 2:
            tree = ET.parse(configFile)
            root = tree.getroot()
            tag = root.find(replaceTag)
            tag.clear()
            tree.write(configFile)
        # in this section it iterates to replaceTag value and remove newTag value
        #remove key Patch Usage :updateConfig("DNDProductInformation", "", delete_Tag, 'Patch')
        if updateType == 3:
            tree = ET.parse(configFile)
            root = tree.getroot()
            try:
                for parent in root.iter(replaceTag):
                    for child in parent.iter(newTag):
                        parent.remove(child)
            except:
                for parent in root.getiterator(replaceTag):
                    for child in parent.getiterator(newTag):
                        parent.remove(child)
            tree.write(configFile)
        #update the value of key if its present in xml else adds the key to xml
        if updateType == 4:
            tree = ET.parse(configFile)
            root = tree.getroot()
            tag = root.find(replaceTag)
            checkTag = tag.find(newTag)
            if checkTag is None:
                if attributeName == "":
                    attribute = {}
                else:
                    attribute = {attributeName:attributeValue}
                subTag = ET.SubElement(tag, newTag, attrib=attribute)
                subTag.text = str(value)
            else:
                checkTag.text = str(value)
            tree.write(configFile)
        #Add the key and value if key-value pair does not match
        if updateType == 5:
            tree = ET.parse(configFile)
            root = tree.getroot()
            tag = root.find(replaceTag)
            tagList = [i.text for i in tag.findall(newTag)]
            if not value in tagList:
                subTag = ET.SubElement(tag, newTag, attrib={})
                subTag.text = str(value)
            tree.write(configFile)
        #Add the key if it is not present, do nothing if the key is present
        if updateType == 6:
            tree = ET.parse(configFile)
            root = tree.getroot()
            tag = root.find(replaceTag)
            checkTag = tag.find(newTag)
            if checkTag is None:
                if attributeName == "":
                    attribute = {}
                else:
                    attribute = {attributeName:attributeValue}
                subTag = ET.SubElement(tag, newTag, attrib=attribute)
                subTag.text = str(value)
                tree.write(configFile)
        # Remove a key value pair from the xml
        # Remove key "FileSystem" with value "tmpfs"
        # Usage : updateConfig("DNDOASGlobalSettings/OASExcludedFileSystems", "tmpfs", delete_Tag_With_Specific_Value, "FileSystem")
        if updateType == 7:
            tree = ET.parse(configFile)
            root = tree.getroot()
            for parent in root.findall(replaceTag):
                for child in parent.findall(newTag):
                    if child.text == str(value):
                        parent.remove(child)
            tree.write(configFile)
        # Update the tag attributes as well as its value if present, else add new tag with the attributes
        # Usage : updateConfig("ProductDefaultSettings/PrefixKeyword", "%DEFLOGDIR%", update_Tag_With_Attributes_If_Present_Else_Add_Tag, 'KeyName', '/var/McAfee/ens/esp/log')
        if updateType == 8:
            tree = ET.parse(configFile)
            root = tree.getroot()
            tag = root.find(replaceTag)
            for checkTag in tag.findall(newTag):
                if checkTag is None:
                    if attributeName == "":
                        attribute = {}
                    else:
                        attribute = {attributeName:attributeValue}
                    subTag = ET.SubElement(tag, newTag, attrib=attribute)
                    subTag.text = str(value)
                    break
                elif checkTag.text == str(value):
                    if attributeName == "":
                        attribute = {}
                    else:
                        attribute = {attributeName: attributeValue}
                    tag.remove(checkTag)
                    subTag = ET.SubElement(tag, newTag, attrib=attribute)
                    subTag.text = str(value)
                    break
            else:
                if attributeName == "":
                    attribute = {}
                else:
                    attribute = {attributeName:attributeValue}
                subTag = ET.SubElement(tag, newTag, attrib=attribute)
                subTag.text = str(value)
            tree.write(configFile)
    except:
        _e = sys.exc_info()[1]
        logging.exception("%s" %str(_e))
        logging.error("Exception occurred while updating config prefs.xml")
        sys.exit(0)

#deleteUnusedDir function will delete old unused DAT/Engine files
def deleteUnusedDir(arrayDirList, dirNeedtoReplace, dirName):
    try:
        for listFile in arrayDirList:
            if listFile != dirNeedtoReplace:
                if listFile:
                    logging.info("Deleting file : "+listFile)
                    shutil.rmtree(os.path.join(dirName, listFile))
    except:
        _e = sys.exc_info()[1]
        logging.exception("%s" %str(_e))
        logging.error("Exception occurred while deleting unused DAT/Engine files")

# This method is used to copy the old log files to the respective new mfe log archive directory
# These old log files are copied to upgrade Tmp dir during preinst stage.
# Here before copying the files to the new location we rename the files as per the name change convention
def copyLogFiles(oldName, newName, upgradeTmpDir, rename=True):
    try:
        for item in os.listdir(upgradeTmpDir + "/" + oldName):
            s = os.path.join(upgradeTmpDir + "/" + oldName, item)
            if rename:
                item = item.replace(oldName, newName)
            d = os.path.join("/var/McAfee/ens/log/tp/" + newName, item)
            shutil.copyfile(s, d)

    except:
            _e = sys.exc_info()[1]
            logging.exception(" %s" %str(_e))
            logging.error("Exception occurred while copying log files")

# From 10.7.x onwards, ODS reports will be stored in archive directory under ODS report directory.
# During upgrade copy *log files from ODS report directory to archive directory before compressing it
def copyODSReportsToArchiveDirectory(oldPath, newPath):
    try:
        source = os.listdir(oldPath)
        for files in source:
            if files.endswith(".log"):
                shutil.move(oldPath+files, newPath)
    except:
            _e = sys.exc_info()[1]
            logging.exception(" %s" %str(_e))
            logging.error("Exception occurred while copying ODS report log files")

# This method will do compression of the log files to one archive file based on the
# directory information and archive name provided
def compressLogFilesDuringUpgrade(dirPath, archiveName):
    _cmd = binPath7za + ' a -sdel ' + dirPath + '/' + archiveName + ' ' + dirPath + '/*log ' + '\'-x!' + dirPath + '/*zip' + '\''
    shellFlag = True
    _ret, _output, _err = executeCmd(_cmd, shellFlag)

def updateLogFileNameTags():
    updateConfig("DNDProductDefaultSettings", "mfetpd", update_Tag_If_Present_Else_Add_Tag, 'LogFileForFM')
    updateConfig("DNDProductDefaultSettings", "mfeoasmgr", update_Tag_If_Present_Else_Add_Tag, 'LogFileForOASMgr')
    updateConfig("DNDProductDefaultSettings", "mfeodscollector", update_Tag_If_Present_Else_Add_Tag, 'LogFileForODSCollector')
    updateConfig("DNDProductDefaultSettings", "mfeodsmanager", update_Tag_If_Present_Else_Add_Tag, 'LogFileForODSMgr')
    updateConfig("DNDProductDefaultSettings", "mfescanactionmgr", update_Tag_If_Present_Else_Add_Tag, 'LogFileForScanActionMgr')
    updateConfig("DNDProductDefaultSettings", "mfescanfactory", update_Tag_If_Present_Else_Add_Tag, 'LogFileForScanFactory')

# This method checks if a given file belongs to McAfeeTP rpm package
def checkFilePackageForRPM(fileName, fileDir):
    logging.debug("Checking if : " + str(fileName) + " belongs to McAfeeTP rpm package")
    match = False

    _cmd = 'rpm -qf ' + fileDir + '/' + fileName
    shellFlag = True
    _ret, _output, _err = executeCmd(_cmd, shellFlag)
    pattern = re.compile('(McAfeeTP-[0-9]{1,}.[0-9]{1,}.[0-9]{1,}-[0-9]{1,})')
    packageName = _output.decode('utf-8')
    match = pattern.search(packageName)

    return match

# This method removes a serialised file for map that has a bucket count more than maxDBBucketCount
# Loading a serialised file with a high bucket count takes memory
def removeInvalidSerialisedDB(fileName):
    if not os.path.exists(fileName):
        logging.info("Serialised DB file "+ str(fileName) + " does not exist")
        return
    fo = open(fileName, 'rb')
    # Read upto 60 bytes of the serialised map file, where bucket count will be present
    chunk = fo.read(60)
    fo.close()
    if len(chunk) != 60:
        logging.info("Unable to determine bucket count in the serialised DB file "+str(fileName) + " as size of the file is less than 60 bytes")
        return
    else:
        hexVal = binascii.hexlify(chunk)
        # After hex encoding, bucket count is expected to be from location 106 to 114
        hexCount = hexVal[106:-6]
        # Fix the endianness of the hex value of the bucket count by reversing it. So, hexCount of 4bf0cb36 becomes a bucket count of 36CBF04B
        hexBucketCount = hexCount[6:] + hexCount[4:-2] + hexCount[2:-4] + hexCount[0:-6]
        # Convert bucket count in hexadecimal to integer
        intBucketCount = int(hexBucketCount,16)
        if intBucketCount > maxDBBucketCount :
            try:
                os.remove(fileName)
                logging.info("Deleting serialised DB file "+ str(fileName) + " as bucket count is higher than " + str(maxDBBucketCount) + " and may be corrupted.")
            except:
                logging.error("Unable to delete the serialised DB file "+ str(fileName) + " where bucket count is higher than " + str(maxDBBucketCount) + " and may be corrupted.")

# Main function
def main():
    '''
    Main Loop
    '''
    #Configure the logger for migration script
    log_format = "%(asctime)s  %(levelname)6s:  %(message)s"
    log_date_format = '%a, %d %b %Y %H:%M:%S'

    if len(sys.argv) != 4:
        usage()
        sys.exit(0)
    else:
        try:
            if not os.path.exists(logFilePath):
                print ("Exiting.. Because log file path does not exist")
                sys.exit(0)

            logging.basicConfig(level=logging.DEBUG,
                                format=log_format,
                                datefmt=log_date_format,
                                filename=logFilePath+"migrate-mfetp.log")

            #Setting log file permission to (0600/-rw-------)
            #0600 is an octal value and conversion to decimal system is 384
            os.chmod(logFilePath+"migrate-mfetp.log", 384)
            logging.debug("Checking if temporary backup directory exists...")

            if sys.argv[1] != 'post':
                logging.error("Argument required  as : post ")
                sys.exit(0)

            backup_dir = sys.argv[2]
            if not os.path.exists(backup_dir):
                logging.error("Exiting the backup process because backup directory %s does not exist" %backup_dir)
                sys.exit(0)
            logging.debug("backup dir is : %s"  %backup_dir)

            # Remove EP serialised DB file if it has a bucket count more than maxDBBucketCount
            removeInvalidSerialisedDB(epRuleDb)
            # Remove AP serialised DB file if it has a bucket count more than maxDBBucketCount
            removeInvalidSerialisedDB(apRuleDb)

            if not isHotFixPackage:
                arrayDatList = getDirList(datDir)
                # Retain the existing DATs only if it is an upgrade from post 10.7.0 as pre 10.7.0 TP used V2 DATs, whereas 10.7.0 onwards TP uses MED DAT.
                if os.path.exists(backup_dir + '/pre1066ISecTPInstalled') or os.path.exists(backup_dir + '/pre1070mfeTPInstalled'):
                    updateConfig("DNDGlobalSettings/MajorDATVersion", 999)
                    deleteUnusedDir(arrayDatList, '999', datDir)
                else:
                    datNeedtoReplace = getValidVersion(arrayDatList, datDir)
                    logging.info("DAT replace:" +str(datNeedtoReplace))
                    if datNeedtoReplace != 0:
                        # In case of rpm machines, if the DAT was not updated since the installation of any previous version, then in that case the DAT file is the one shipped
                        # with that version and it belongs to that rpm package. This DAT will be deleted when the older version is removed after upgrade by the rpm.
                        # In order to prevent that, check if the datNeedtoReplace belongs to any McAfeeTP rpm packages. If it matches then the datNeedtoReplace will be changed to the 999
                        if sys.argv[3] == 'RPM' :
                            match = checkFilePackageForRPM(datNeedtoReplace, datDir)
                            if not match:
                                updateConfig("DNDGlobalSettings/MajorDATVersion", datNeedtoReplace)
                                deleteUnusedDir(arrayDatList, datNeedtoReplace, datDir)
                            else:
                                updateConfig("DNDGlobalSettings/MajorDATVersion", 999)
                        else:
                            updateConfig("DNDGlobalSettings/MajorDATVersion", datNeedtoReplace)
                            deleteUnusedDir(arrayDatList, datNeedtoReplace, datDir)
                    else:
                        logging.error("Value to DAT need to replace is " + str(datNeedtoReplace) + ".Therefore not updating config")
                arrayEngineList = getDirList(engineDir)
                engineNeedtoReplace = getValidVersion(arrayEngineList, engineDir)
                logging.info("Engine replace:" +str(engineNeedtoReplace))
                if engineNeedtoReplace != 0:
                    updateConfig("DNDGlobalSettings/IncrementalUpdateEngineVersion", engineNeedtoReplace)
                    updateConfig("DNDGlobalSettings/MajorEngineVersion", majorEngineNumber)
                    updateConfig("DNDGlobalSettings/MinorEngineVersion", minorEngineNumber)
                    deleteUnusedDir(arrayEngineList, engineNeedtoReplace, engineDir)
                else:
                    logging.error("Value of Engine needed to replace is " + str(engineNeedtoReplace) + ".Therefore not updating config")

                #replace 32 bit engine libraries
                arrayEngine32List = getDirList(engine32Dir)
                engine32NeedtoReplace = getValidVersion(arrayEngine32List, engine32Dir)
                logging.info("Replace 32 bit engine with :" +str(engine32NeedtoReplace))
                if engine32NeedtoReplace != 0:
                    deleteUnusedDir(arrayEngine32List, engine32NeedtoReplace, engine32Dir)
                else:
                    logging.error("Value of 32 bit Engine needed to replace is " + str(engine32NeedtoReplace) + ".Therefore not updating config")

                fixEngineSymlink(engineNeedtoReplace, engine32NeedtoReplace)
                # replace exploit prevention content
                arrayEpContentList = getDirList(epContentDir)
                epContentNeedToReplace = getValidVersion(arrayEpContentList, epContentDir, 1)
                logging.info("Exploit Prevention Content replace : " +str(epContentNeedToReplace))
                updateConfig("DNDGlobalSettings", majorEPContentVersion, add_New_Tag_With_Attributes_If_Not_Present, 'MajorExploitPreventionContentVersion')
                updateConfig("DNDGlobalSettings", minorEPContentVersion, add_New_Tag_With_Attributes_If_Not_Present, 'MinorExploitPreventionContentVersion')
                if epContentNeedToReplace != 0:
                    if sys.argv[3] == 'RPM' :
                        match = checkFilePackageForRPM(epContentNeedToReplace, epContentDir)
                        if not match:
                            updateConfig("DNDGlobalSettings/MinorExploitPreventionContentVersion", epContentNeedToReplace)
                            deleteUnusedDir(arrayEpContentList, epContentNeedToReplace, epContentDir)
                        else:
                            updateConfig("DNDGlobalSettings/MajorExploitPreventionContentVersion", majorEPContentVersion)
                            updateConfig("DNDGlobalSettings/MinorExploitPreventionContentVersion", minorEPContentVersion)
                    else:
                        updateConfig("DNDGlobalSettings/MinorExploitPreventionContentVersion", epContentNeedToReplace)
                        deleteUnusedDir(arrayEpContentList, epContentNeedToReplace, epContentDir)
                else:
                    logging.error("Value of Exploit Prevention Content needed to replace is " + str(epContentNeedToReplace) + ".Therefore not updating config")
            # Default UseFanotify value is disable, Do not edit the value if the tag is already present.
            updateConfig("DNDOASGlobalSettings", 0, add_New_Tag_With_Attributes_If_Not_Present, 'UseFanotify')
            # Default EnableOASActivityLogging value is disable,, Do not edit the value if the tag is already present.
            updateConfig("DNDClientLogging", 0, add_New_Tag_With_Attributes_If_Not_Present, 'EnableOASActivityLogging')
            # Default EnableDebugLoggingForMsgBus value is disable,, Do not edit the value if the tag is already present.
            updateConfig("DNDClientLogging", 0, add_New_Tag_With_Attributes_If_Not_Present, 'EnableDebugLoggingForMsgBus')
            updateConfig("DNDOASGlobalSettings/OASExcludedFileSystems", "tmpfs", delete_Tag_With_Specific_Value, 'FileSystem')
            updateConfig("DNDOASGlobalSettings/OASExcludedFileSystems", "secfs2", delete_Tag_With_Specific_Value, 'FileSystem')
            # Default EnableODSActivityLogging value is disable. Do not edit the value if the tag is already present.
            updateConfig("DNDClientLogging", 0, add_New_Tag_With_Attributes_If_Not_Present, 'EnableODSActivityLogging')
            # Update syscall table address for 32 bit. Do not edit the value if the tag is already present
            updateConfig("DNDOASGlobalSettings", 0, add_New_Tag_With_Attributes_If_Not_Present, 'SyscallTableAddress32Bit')
            # Update IsFanotifySupported. Do not edit the value if the tag is already present
            updateConfig("DNDOASGlobalSettings", 1, add_New_Tag_With_Attributes_If_Not_Present, 'IsFanotifySupported') 
            # Update UseDeferredScan. Do not edit the value if the tag is already present
            updateConfig("DNDOASGlobalSettings", 0, add_New_Tag_With_Attributes_If_Not_Present, 'UseDeferredScan') 
            if os.path.isdir(backup_dir + "/isectpd"):
                copyLogFiles("isectpd", "mfetpd", backup_dir)
            if os.path.isdir(backup_dir + "/isecav"):
                copyLogFiles("isecav", "mfetpcli", backup_dir)
            if os.path.isdir(backup_dir + "/isecscanfactory"):
                copyLogFiles("isecscanfactory", "mfescanfactory", backup_dir)
            if os.path.isdir(backup_dir + "/isecscanactionmgr"):
                copyLogFiles("isecscanactionmgr", "mfescanactionmgr", backup_dir)
            if os.path.isdir(backup_dir + "/isecoasmgr"):
                copyLogFiles("isecoasmgr", "mfeoasmgr", backup_dir)
            if os.path.isdir(backup_dir + "/isecodscollector"):
                copyLogFiles("isecodscollector", "mfeodscollector", backup_dir)
            # Retain the names in case of odsreport  and vsel migration dir
            if os.path.isdir(backup_dir + "/odsreport"):
                copyLogFiles("odsreport", "odsreport", backup_dir, False)
            if os.path.isdir(backup_dir + "/vsel"):
                copyLogFiles("vsel", "vsel", backup_dir, False)

            copyODSReportsToArchiveDirectory("/var/McAfee/ens/log/tp/odsreport/", "/var/McAfee/ens/log/tp/odsreport/archive/")
            compressLogFilesDuringUpgrade("/var/McAfee/ens/log/tp/mfetpd", "mfetpd.zip")
            if os.listdir('/var/McAfee/ens/log/tp/mfeodscollector'):
                compressLogFilesDuringUpgrade("/var/McAfee/ens/log/tp/mfeodscollector", "mfeodscollector.zip")
            if os.listdir('/var/McAfee/ens/log/tp/odsreport/archive'):
                compressLogFilesDuringUpgrade("/var/McAfee/ens/log/tp/odsreport/archive", "odsreport.zip")
            if os.listdir('/var/McAfee/ens/log/tp/mfeoasmgr'):
                compressLogFilesDuringUpgrade("/var/McAfee/ens/log/tp/mfeoasmgr", "mfeoasmgr.zip")

            updateLogFileNameTags()

            if isHotFixPackage:
                tree = ET.parse(configFile)
                root = tree.getroot()
                tag = root.find("DNDProductInformation/HotFixes/HotFix")
                if not tag.text:
                    updateConfig("DNDProductInformation/HotFixes", hotFixNumber, update_Tag_If_Present_Else_Add_Tag, 'HotFix')
                else:
                    updateConfig("DNDProductInformation/HotFixes", hotFixNumber, add_New_Tag, 'HotFix')
            if isPatchPackage:
                #delete the HotFix information and add Patch details
                updateConfig("DNDProductInformation/HotFixes", "", 2)
                updateConfig("DNDProductInformation/HotFixes", "", add_New_Tag, 'HotFix')
                updateConfig("DNDProductInformation", "", delete_Tag, 'ProductNameForEvent')
                updateConfig("DNDProductInformation", productNameForEvent, add_New_Tag_With_Attributes_If_Not_Present, 'ProductNameForEvent')
                updateConfig("DNDProductInformation", "1", add_New_Tag_With_Attributes_If_Not_Present, 'DaysSinceInstallation')
                updateConfig("DNDProductInformation", GTIHostAddr, add_New_Tag_With_Attributes_If_Not_Present, 'GTIHostAddr')
                updateConfig("DNDProductInformation", GTIUserID, add_New_Tag_With_Attributes_If_Not_Present, 'GTIUserID')
                updateConfig("DNDProductInformation", GTIKey, add_New_Tag_With_Attributes_If_Not_Present, 'GTIKey')
                updateConfig("DNDProductInformation", GTICACertFile, add_New_Tag_With_Attributes_If_Not_Present, 'GTICACertFile')
                updateConfig("DNDProductInformation", FIPSMode, add_New_Tag_With_Attributes_If_Not_Present, 'FIPSMode')
                updateConfig("DNDProductInformation", GTIFIPSHostAddr, add_New_Tag_With_Attributes_If_Not_Present, 'GTIFIPSHostAddr')
                updateConfig("DNDProductInformation", GTIFIPSUserID, add_New_Tag_With_Attributes_If_Not_Present, 'GTIFIPSUserID')
                updateConfig("DNDProductInformation", GTIFIPSKey, add_New_Tag_With_Attributes_If_Not_Present, 'GTIFIPSKey')
                updateConfig("DNDProductInformation", GTIFIPSCACertFile, add_New_Tag_With_Attributes_If_Not_Present, 'GTIFIPSCACertFile')
                updateConfig("DNDProductInformation", GTIProductName, add_New_Tag_With_Attributes_If_Not_Present, 'GTIProductName')
                updateConfig("DNDProductInformation/Patch", patchNumber)
                updateConfig("DNDProductInformation/ProductCopyright", copyRight)
                updateConfig("DNDGlobalSettings", "0", add_New_Tag_With_Attributes_If_Not_Present, 'DATReleaseDay')
                updateConfig("DNDGlobalSettings", "0", add_New_Tag_With_Attributes_If_Not_Present, 'DATReleaseMonth')
                updateConfig("DNDGlobalSettings", "0", add_New_Tag_With_Attributes_If_Not_Present, 'DATReleaseYear')
                updateConfig("DNDGlobalSettings", "0", add_New_Tag_With_Attributes_If_Not_Present, 'DATUpdateTime')
                updateConfig("DNDGlobalSettings", "0", add_New_Tag_With_Attributes_If_Not_Present, 'ExploitPreventionContentReleaseDay')
                updateConfig("DNDGlobalSettings", "0", add_New_Tag_With_Attributes_If_Not_Present, 'ExploitPreventionContentReleaseMonth')
                updateConfig("DNDGlobalSettings", "0", add_New_Tag_With_Attributes_If_Not_Present, 'ExploitPreventionContentReleaseYear')
                updateConfig("DNDGlobalSettings", "0", add_New_Tag_With_Attributes_If_Not_Present, 'IsGTIFeedbackEnabled')
                updateConfig("DNDOASGlobalSettings", "1", add_New_Tag_With_Attributes_If_Not_Present, 'IsOASSupported')
                updateConfig("DNDOASGlobalSettings", "100", add_New_Tag_With_Attributes_If_Not_Present, 'MaxOASCPULimit')
                updateConfig("DNDOASGlobalSettings/OASExcludedFileSystems", "vxclonefs", add_New_Tag, 'FileSystem')
                updateConfig("DNDOASGlobalSettings/OASExcludedFileSystems", "novfs", add_New_Tag, 'FileSystem')
                updateConfig("DNDOASGlobalSettings/OASExcludedFileSystems", "configfs", add_New_Tag, 'FileSystem')
                addNewTag("DNDGTISelectionCriteria", "", "type", "57")
                addNewTag("DNDProxySettings", "", "type", "58")
                addNewTag("DNDAPGlobalExclusions", "", "type", "59")
                addNewTagRecursive("ODSProfile/LastRunSummary", "NoOfCleanGTIFiles", "0")
                addNewTagRecursive("ODSProfile", "MaxCPULimit", "100")
                addNewTagRecursive("ODSProfile", "IsGTIEnabled", "1")
                addNewTagRecursive("ODSProfile", "GTISensitivityLevel", "3")
                updateGTISelectionCriteriaTags()
                updateProxySettingTags()
                updateAPGlobalExclusionsTags()

            if (not isHotFixPackage and not isPatchPackage):
                #delete the HotFix information and Patch Information
                updateConfig("DNDProductInformation/HotFixes", "", clear_Tag)
                updateConfig("DNDProductInformation/HotFixes", "", add_New_Tag, 'HotFix')
                updateConfig("DNDProductInformation", "", add_New_Tag_With_Attributes_If_Not_Present, 'Patch')
                updateConfig("DNDProductInformation", "", delete_Tag, 'ProductNameForEvent')
                updateConfig("DNDProductInformation", productNameForEvent, add_New_Tag_With_Attributes_If_Not_Present, 'ProductNameForEvent')
                updateConfig("DNDProductInformation", "1", add_New_Tag_With_Attributes_If_Not_Present, 'DaysSinceInstallation')
                updateConfig("DNDProductInformation", GTIHostAddr, update_Tag_If_Present_Else_Add_Tag, 'GTIHostAddr')
                updateConfig("DNDProductInformation", GTIUserID, add_New_Tag_With_Attributes_If_Not_Present, 'GTIUserID')
                updateConfig("DNDProductInformation", GTIKey, add_New_Tag_With_Attributes_If_Not_Present, 'GTIKey')
                updateConfig("DNDProductInformation", GTICACertFile, add_New_Tag_With_Attributes_If_Not_Present, 'GTICACertFile')
                updateConfig("DNDProductInformation", GTICACertFile, add_New_Tag_With_Attributes_If_Not_Present, 'GTICACertFile')
                updateConfig("DNDProductInformation", FIPSMode, add_New_Tag_With_Attributes_If_Not_Present, 'FIPSMode')
                updateConfig("DNDProductInformation", GTIFIPSHostAddr, add_New_Tag_With_Attributes_If_Not_Present, 'GTIFIPSHostAddr')
                updateConfig("DNDProductInformation", GTIFIPSUserID, add_New_Tag_With_Attributes_If_Not_Present, 'GTIFIPSUserID')
                updateConfig("DNDProductInformation", GTIFIPSKey, add_New_Tag_With_Attributes_If_Not_Present, 'GTIFIPSKey')
                updateConfig("DNDProductInformation", GTIFIPSCACertFile, add_New_Tag_With_Attributes_If_Not_Present, 'GTIFIPSCACertFile')
                updateConfig("DNDProductInformation", GTIProductName, add_New_Tag_With_Attributes_If_Not_Present, 'GTIProductName')
                updateConfig("DNDProductInformation/ProductCopyright", copyRight)
                updateConfig("DNDGlobalSettings", "0", add_New_Tag_With_Attributes_If_Not_Present, 'DATReleaseDay')
                updateConfig("DNDGlobalSettings", "0", add_New_Tag_With_Attributes_If_Not_Present, 'DATReleaseMonth')
                updateConfig("DNDGlobalSettings", "0", add_New_Tag_With_Attributes_If_Not_Present, 'DATReleaseYear')
                updateConfig("DNDGlobalSettings", "0", add_New_Tag_With_Attributes_If_Not_Present, 'DATUpdateTime')
                updateConfig("DNDGlobalSettings", "0", add_New_Tag_With_Attributes_If_Not_Present, 'ExploitPreventionContentReleaseDay')
                updateConfig("DNDGlobalSettings", "0", add_New_Tag_With_Attributes_If_Not_Present, 'ExploitPreventionContentReleaseMonth')
                updateConfig("DNDGlobalSettings", "0", add_New_Tag_With_Attributes_If_Not_Present, 'ExploitPreventionContentReleaseYear')
                updateConfig("DNDGlobalSettings", "0", add_New_Tag_With_Attributes_If_Not_Present, 'IsAPEnabled')
                updateConfig("DNDGlobalSettings", "0", add_New_Tag_With_Attributes_If_Not_Present, 'IsEPEnabled')
                # Update the prefs IsPostDeploymentStartUp and IsAutoContentUpdateEnabled so that update task will be run after upgrade.
                updateConfig("DNDGlobalSettings", "1", update_Tag_If_Present_Else_Add_Tag, 'IsPostDeploymentStartUp')
                updateConfig("DNDGlobalSettings", "1", update_Tag_If_Present_Else_Add_Tag, 'IsAutoContentUpdateEnabled')
                updateConfig("DNDGlobalSettings", "0", add_New_Tag_With_Attributes_If_Not_Present, 'IsGTIFeedbackEnabled')
                updateConfig("DNDOASGlobalSettings", "1", add_New_Tag_With_Attributes_If_Not_Present, 'IsOASSupported')
                updateConfig("DNDOASGlobalSettings", "100", add_New_Tag_With_Attributes_If_Not_Present, 'MaxOASCPULimit')
                updateConfig("DNDOASGlobalSettings/OASExcludedFileSystems", "vxclonefs", add_New_Tag_If_Key_Value_Not_Present, 'FileSystem')
                updateConfig("DNDOASGlobalSettings/OASExcludedFileSystems", "novfs", add_New_Tag_If_Key_Value_Not_Present, 'FileSystem')
                updateConfig("DNDOASGlobalSettings/OASExcludedFileSystems", "configfs", add_New_Tag_If_Key_Value_Not_Present, 'FileSystem')
                updateConfig("DNDOASGlobalSettings/OASExcludedFileSystems", "fuse.s3fs", add_New_Tag_If_Key_Value_Not_Present, 'FileSystem')
                updateConfig("DNDOASGlobalSettings/OASExcludedFileSystems", "efivarfs", add_New_Tag_If_Key_Value_Not_Present, 'FileSystem')
                updateConfig("DNDOASGlobalSettings/OASExcludedFileSystems", "tracefs", add_New_Tag_If_Key_Value_Not_Present, 'FileSystem')
                updateConfig("DNDOASGlobalSettings/OASExcludedFileSystems", "selinuxfs", add_New_Tag_If_Key_Value_Not_Present, 'FileSystem')
                updateConfig("DNDOASGlobalSettings/OASExcludedFileSystems", "gpfs", add_New_Tag_If_Key_Value_Not_Present, 'FileSystem')
                updateConfig("DNDOASGlobalSettings", "0", add_New_Tag_With_Attributes_If_Not_Present, 'KallsymsLookUpNameAddress')
                addNewTag("DNDGTISelectionCriteria", "", "type", "57")
                addNewTag("DNDProxySettings", "", "type", "58")
                addNewTag("DNDAPGlobalExclusions", "", "type", "59")
                addNewTagRecursive("ODSProfile/LastRunSummary", "NoOfCleanGTIFiles", "0")
                addNewTagRecursive("ODSProfile", "MaxCPULimit", "100")
                addNewTagRecursive("ODSProfile", "IsGTIEnabled", "1")
                addNewTagRecursive("ODSProfile", "GTISensitivityLevel", "3")
                updateGTISelectionCriteriaTags()
                updateProxySettingTags()
                updateAPGlobalExclusionsTags()
                updateConfig("DNDProductInformation", "/var/McAfee/ens/tp", update_Tag_If_Present_Else_Add_Tag, 'ProductVarDir')
                updateConfig("DNDProductInformation", "/opt/McAfee/ens/tp", update_Tag_If_Present_Else_Add_Tag, 'ProductInstallDir')
                updateConfig("DNDProductInformation", "mfetpcli", update_Tag_If_Present_Else_Add_Tag, 'ProductCLIName')
                updateConfig("DNDProductDefaultSettings", "/var/McAfee/ens/esp/espsocket", update_Tag_If_Present_Else_Add_Tag, 'ESPSocketPath')
                updateConfig("DNDProductDefaultSettings/PrefixKeyword", "%DEFLOGDIR%", update_Tag_With_Attributes_If_Present_Else_Add_Tag, 'KeyName', 'value', '/var/McAfee/ens/log/tp')
        except:
            _e = sys.exc_info()[1]
            logging.exception(" %s" %str(_e))
            logging.error("Exception occurred while upgrading build")
            sys.exit(0)

# Start Here
if __name__ == "__main__":
    main()
