#!/bin/bash
# Copyright (C) 2016-2021 McAfee, LLC. All Rights Reserved.
# Script to do uninstallation of ISecRt, ESP for Linux and Threat Prevention for Linux
#   for both silent (ePO) and prompt type of uninstallation
# TBD: Add support for force uninstallation

# Exit codes on failure in this script are
# 1 - This script can be run by bash shell only.
# 5 - Must be a root user to run this script.
# 21 - Invalid command line option passed during uninstallation.
# 14 - Uninstallation failed.
# 0 - Uninstallation was successful
# NOTE: Exit codes are kept uniform for all installation and uninstallation scripts

# Use only bash for executing the script
ps -o args= -p "$$" | grep bash > /dev/null
if [ $? -ne 0 ]
then
    echo "Error: This script can be run by bash shell only"
    echo "Usage: $0 [uninstalltype] [path of log file]"
    echo "Uninstall type can be 'silent' or 'prompt' (default)"
    echo "  In 'silent' install, product will be uninstalled immediately"
    echo "  In 'prompt' install, user will be prompted to confirm product uninstallation"
    echo "Any file path provided will be used directly"
    exit 1
fi

usage() {
    echo "Usage: $0 [uninstalltype] [path of log file]"
    echo "Uninstall type can be 'silent' or 'prompt' (default)"
    echo "  In 'silent' install, product will be uninstalled immediately"
    echo "  In 'prompt' install, user will be prompted to confirm product uninstallation"
    echo "Any file path provided will be used directly"
    exit 21
}

# Echo's the message on the console as well as logs the message in syslog file
logMessage()
{
    message=$1
    echo "${message}"
    logger -t "${me}" "${message}"
}

# Function to check if a package is installed
# 1st param - name of the package to be checked
checkInstalled() {
    PACKAGE_NAME=$1
    ${CHECK_CMD} ${PACKAGE_NAME} > /dev/null 2>&1
    if [ $? -ne 0 ]
    then
        return 1
    fi
    return 0
}

# Function to uninstall a file
# 1st param - Name of package to be uninstalled
# 2nd param - Description of package being uninstalled
uninstall() {
    PACKAGE_NAME=$1
    PACKAGE_DESC=$2
    checkInstalled ${PACKAGE_NAME}
    if [ $? -ne 0 ]
    then
        logMessage "${PACKAGE_DESC} is currently not installed."
        return 0
    fi
    ${UNINSTALL_CMD} ${PACKAGE_NAME}
    if [ $? -ne 0 ]
    then
        logMessage "Failed to uninstall ${PACKAGE_DESC}."
        exit 14
    else
        logMessage "Successfully uninstalled ${PACKAGE_DESC}."
    fi
    return 0
}

# Function to cleanup temporary files
cleanup()
{
    exit 0
}

# Function to read or quit from the user input
readOrQuit()
{
    read
    if [ $? -ne 0 ]
    then
        cleanup
    fi
}

function removeENSLCgroupDirectoryIfExists()
{
    if [ -d /sys/fs/cgroup/cpu,cpuacct/ensl_tp_ods ]; then
        rmdir /sys/fs/cgroup/cpu,cpuacct/ensl_tp_ods 2>/dev/null
    elif [ -d /cgroup/cpu/ensl_tp_ods ]; then
        rmdir /cgroup/cpu/ensl_tp_ods 2>/dev/null
    fi
    if [ -d /sys/fs/cgroup/cpu,cpuacct/ensl_tp_oas ]; then
        rmdir /sys/fs/cgroup/cpu,cpuacct/ensl_tp_oas 2>/dev/null
    elif [ -d /cgroup/cpu/ensl_tp_oas ]; then
        rmdir /cgroup/cpu/ensl_tp_oas 2>/dev/null
    fi
}

# This will be used to automatically determine the name of the output log file
declare -r me=${0##*/}
# Set default permissions of files to allow read / write only for owner
umask 077
# Used to track if this is a rpm or deb based system
DEB_VARIANT="no"

# Check uninstaller is executed as root
ID="$(id -u)"
if [ $? -ne 0 -o "$ID" -ne 0 ]
then
    logMessage "Must be root / sudo user to uninstall this product"
    exit 5
fi

# There are two types of uninstallation - silent and prompt
# Default is prompt which prompts the user to confirm uninstallation of product
# For 'silent', directly uninstall the product
uninstallType=$1
if [ ! -n "${uninstallType}" ]
then
    uninstallType="prompt"
else
    if [ "${uninstallType}" != "silent" -a "${uninstallType}" != "prompt" ]
    then
        logMessage "${uninstallType} - Unsupported uninstallation type provided"
        usage
    fi
fi

# Set log file, default log file path is /tmp/name_of_script.log
log=$2
if [ ! -n "${log}" ]
then
    log="/tmp/${me}.log"
else
    # Use default log file path if provided log file is not writable
    touch $log 2>/dev/null
    if [ $? -ne 0 ]
    then
        log="/tmp/${me}.log"
    fi
fi

# Setup file descriptors for silent uninstall
if [ "${uninstallType}" = "silent" ]
then
    if [ -f /etc/os-release ]; then
        . /etc/os-release
        OS=$NAME
        VER=$VERSION_ID
    fi
    # Do not close stdin on Ubuntu 22.04
    if [ "${OS}" = "Ubuntu" -a "${VER}" = "22.04" ]; then
        logMessage "${OS} ${VER}"
    else
        exec 0<&-   # Close stdin
    fi
    exec 1>$log # Redirects stdout to log file
    exec 2>&1   # Redirect stderr to stdout
    logMessage "Script execution time: $(date)"
fi

# Confirm uninstallation for prompt option
if [ "${uninstallType}" = "prompt" ]
then
        MORE=1
        UNINSTALL_PRODUCT='yes'
        while [ ${MORE} -eq 1 ]
        do
            echo -n "Uninstall Trellix Endpoint Security for Linux Threat Prevention ? (yes or no) : "
            readOrQuit
            if [ "${REPLY}" = "yes" ]
            then
                MORE=0
            else
                if [ "${REPLY}" = "no" ]
                then
                    UNINSTALL_PRODUCT='no'
                    MORE=0
                fi
            fi
        done
    # Abort if uninstallation is not accepted
    if [ "${UNINSTALL_PRODUCT}" = 'no' ]
    then
        logMessage "Aborting uninstallation"
        cleanup
    fi
    # cleanup on interrupt and termination (only in prompt mode)
    trap cleanup 2 15
fi

# Name of packages to be uninstalled
MFE_TP_PACKAGE_NAME="McAfeeTP"
MFE_ESP_FILEACCESS_PACKAGE_NAME="McAfeeESPFileAccess"
MFE_ESP_AAC_PACKAGE_NAME="McAfeeESPAac"

# All known rpm based systems either have /etc/redhat-release (Redhat, Fedora, CentOS) or /etc/SuSE-release (SuSE, openSuSE)
if [ -f /etc/redhat-release -o -f /etc/SuSE-release ]
then
    DEB_VARIANT="no"
elif [ -f /etc/system-release ]
then
    # Amazon Linux AMI is rpm based, has /etc/system-release and its content starts with Amazon
    distribRel=$(cat /etc/system-release)
    amazonSearchPattern="^Amazon"
    if [[ $distribRel =~ $amazonSearchPattern ]]
    then
        DEB_VARIANT="no"
    fi
elif [ -f /etc/os-release ]
then
    # SuSE 15 and above does not ship with /etc/SuSE-release; check /etc/os-release instead
    distribId=$(cat /etc/os-release | grep ^ID=)
    slesSearchPattern="sles|sled"
    opensuseSearchPattern="opensuse|openSUSE"
    if [[ $distribId =~ $slesSearchPattern ]]
    then
        DEB_VARIANT="no"
    elif [[ $distribId =~ $opensuseSearchPattern ]]
    then
        DEB_VARIANT="no"
    else
        DEB_VARIANT="yes"
    fi
else
    DEB_VARIANT="yes"
fi
if [ "${DEB_VARIANT}" = "no" ]
then
    logMessage "Detected rpm based distribution"
    UNINSTALL_CMD="rpm -e "
    CHECK_CMD="rpm -q "
else
    logMessage "Detected deb based distribution"
    UNINSTALL_CMD="dpkg --purge "
    CHECK_CMD="dpkg -s "
fi

# Uninstall McAfeeTP for Linux
uninstall "${MFE_TP_PACKAGE_NAME}" "Trellix Endpoint Security for Linux Threat Prevention"
uninstall "${MFE_ESP_FILEACCESS_PACKAGE_NAME}" "Trellix Endpoint Security Kernel Modules for Linux"
uninstall "${MFE_ESP_AAC_PACKAGE_NAME}" "Trellix Endpoint Security Kernel Modules for Linux"

removeENSLCgroupDirectoryIfExists
# CWD may not exist, change directory as a fail-safe
cd /tmp/
# Run ESP uninstaller
/opt/McAfee/ens/esp/scripts/uninstall-mfeesp.sh
# Exit the script, but since product was successfully uninstalled, ignore return code of ESP Uninstallation
exit 0
