#!/bin/sh
#
# This script is responsible for installing the Serial driver
#

if [ `whoami` != root ]; then
	echo "You must be root to install the device driver!"
	exit 1
fi

# Install a copy of the driver in the version independent location
if [ ! -d /lib/modules ]; then 
	mkdir -p /lib/modules
fi

if test -d /lib/modules/misc ; then
    cp serial.o /lib/modules/misc
    rm -f /lib/modules/serial.o
else
    cp serial.o /lib/modules
fi

# Install a copy of the driver in a kernel version specific location
if [ -f .kver ]; then
    KVER=`sed -e 's/-SMP//' -e 's/-MOD//' .kver`
    KVERSION=`cat .kver` 
else
    KVER=`grep UTS_RELEASE /usr/include/linux/version.h | \
	awk '{print $3}' | tr -d \"`
    KVERSION=$KVER
    if grep -q "#define CONFIG_MODVERSIONS" \
			$LINUX_SRC/include/linux/autoconf.h ; then 
	KVERSION=${KVERSION}-MOD
    fi
    if grep -q smp_ $LINUX_SRC/include/linux/modules/ksyms.ver ; then
	KVERSION=${KVERSION}-SMP
    fi
fi

if grep -q smp_ /proc/ksyms ; then 
	RSMP='-SMP'
fi
if grep -q "kfree" /proc/ksyms &&
   ! grep -q "kfree$$" /proc/ksyms ; then
	RMOD='-MOD'
fi
RVERSION=`uname -r`$RMOD$RSMP
echo " "
echo "Installing serial driver for Linux $KVERSION"
echo " "
if [ $KVERSION != $RVERSION ]; then
    echo "WARNING: The current kernel is actually version $RVERSION."
    echo " "
fi

#
# Actually install the kernel module
#
if [ ! -d /lib/modules/$KVER/misc ]; then 
	mkdir -p /lib/modules/$KVER/misc
fi

cp serial.o /lib/modules/$KVER/misc

#
# We're not playing the modules game any more, so remove the old 
# mod_serial.o file.
# 
rm -f /lib/modules/mod_serial.o /lib/modules/$KVER/misc/mod_serial.o

#
# Now that we're done, run depmod -a so that modprobe knows where to fund
# stuff
#
depmod -a

#
# Since the serial driver exports symbols, update the serial.ver and 
# modversions file.
#
if test -z "$LINUX_SRC" ; then
    echo "Defaulting linux sources to /usr/src/linux"
    LINUX_SRC=/usr/src/linux
fi
VERFILE=$LINUX_SRC/include/linux/modules/serial.ver
MODVERFILE=$LINUX_SRC/include/linux/modversions.h

if ! test -f $VERFILE || ! cmp -s serial.ver $VERFILE ; then 
	echo "Updating $VERFILE"
	cp serial.ver $VERFILE
fi
if ! grep -q /serial.ver $MODVERFILE ; then \
	echo "Updating $MODVERFILE to include serial.ver"
	sed -f modversions.sed < $MODVERFILE > ${MODVERFILE}.new
	mv $MODVERFILE ${MODVERFILE}.old
	mv ${MODVERFILE}.new $MODVERFILE
fi


#
# OK, now we install the installation script 
#
if [ -d /etc/rc.d/init.d ]; then 
	# This is a RedHat or other system using a System V init scheme
	RC_DEST=/etc/rc.d/init.d/serial
	RC_DIR=/etc/rc.d
	RC_START=S83serial
	RC_STOP=K18serial
elif [ -d /etc/init.d ]; then 
	# This is a Debian system

	RC_DEST=/etc/init.d/serial
	RC_DIR=/etc
	RC_START=S83serial
	RC_STOP=K20serial
else
        # This must be a Slackware or other non-SysV init system
	if [ ! -f /etc/rc.d/rc.serial ]; then
		echo "The initialization script will be installed in "
		echo "/etc/rc.d/rc.serial.  You will need to edit your "
		echo "/etc/rc files to run /etc/rc.d/rc.serial"

		if [ ! -d /etc/rc.d ]; then 
			mkdir -p /etc/rc.d
		fi
	fi

	RC_DEST=/etc/rc.d/rc.serial
	RC_DIR=""
fi
		
#
# Determine the version numbers of the installed script (if any) and the
# rc.serial script which we have.
#
SRC_VERS=`grep "^# FILE_VERSION: " rc.serial | awk '{print $3}'`
DEST_VERS=0
if test -f $RC_DEST ; then
    DEST_VERS=`grep "^# FILE_VERSION: " $RC_DEST | awk '{print $3}'`
    if test -z "$DEST_VERS" ; then
    	DEST_VERS=0
    fi
fi

#
# Only update the destination file if we have a newer version.
#
if test $SRC_VERS -gt $DEST_VERS ; then
	if test -f $RC_DEST ; then
		echo "Updating $RC_DEST"
		NEW_INSTALL=no
	else
		echo "Installing $RC_DEST"
		NEW_INSTALL=yes
	fi
	cp rc.serial $RC_DEST
fi

if test -n "$RC_DIR" ; then
	rm -f $RC_DIR/rc?.d/[SK]??rc.serial
	if test "$NEW_INSTALL" = "yes" ; then
	        echo " "
		echo "You are using a system which uses the System V init"
		echo "scheme.  The initialization script has been installed"
		echo "in $RC_DEST and it has been automatically "
		echo "set up to be run when you reboot"

		ln -sf ../init.d/serial $RC_DIR/rc2.d/${RC_START}
		ln -sf ../init.d/serial $RC_DIR/rc3.d/${RC_START}
		ln -sf ../init.d/serial $RC_DIR/rc5.d/${RC_START}
		ln -sf ../init.d/serial $RC_DIR/rc0.d/${RC_STOP}
		ln -sf ../init.d/serial $RC_DIR/rc1.d/${RC_STOP}
		ln -sf ../init.d/serial $RC_DIR/rc6.d/${RC_STOP}
	fi

	if [ -f /etc/rc.d/rc.serial ]; then
		echo "You have an old /etc/rc.d/rc.serial script"
		echo "After checking to make sure it is no longer being used"
		echo "you should delete it."
	fi
fi
