/*
 * ntp_unixclock.c - routines for reading and adjusting a 4BSD-style
 *		     system clock.  Emacs has hide-ifdef-mode ...
 */

#ifdef HAVE_CONFIG_H
# include <config.h>
#endif

#include <stdio.h>
#include <sys/types.h>
#ifdef HAVE_UNISTD_H
# include <unistd.h>
#endif /* HAVE_UNISTD_H */
#ifndef SYS_WINNT
# ifdef NLIST_STRUCT
#  include <nlist.h>
#  ifdef NLIST_NAME_UNION
#   define N_NAME n_un.n_name
#  else /* not NLIST_NAME_UNION */
#   define N_NAME n_name
#  endif /* not NLIST_NAME_UNION */
# endif /* NLIST_STRUCT */
# include <sys/stat.h>
# include <sys/time.h>
#endif /* SYS_WINNT */

#ifdef HAVE_SYS_PARAM_H
# include <sys/param.h>
#endif
#ifdef HAVE_UTMP_H
# include <utmp.h>
#endif

#if defined(HAVE_GETBOOTFILE)
# include <paths.h>
#endif

#include "ntpd.h"
#include "ntp_io.h"
#include "ntp_unixtime.h"
#include "ntp_stdlib.h"

#if defined(HAVE_LIBKVM)
# ifdef HAVE_SYS_PROC_H
#  include <sys/proc.h>
# endif
# include <kvm.h>
# include <limits.h>

# ifndef _POSIX2_LINE_MAX
#  define _POSIX2_LINE_MAX 2048
# endif
#endif	/* HAVE_LIBKVM */

#ifdef HAVE_SYS_TIMEX_H		/* Linux - also HAVE___ADJTIMEX */
# include <sys/timex.h>
#endif

#ifdef hz /* Was: RS6000 */
# undef hz
#endif /* hz */

extern int debug;
/*
 * These routines (init_systime, get_systime, step_systime, adj_systime)
 * implement an interface between the (more or less) system independent
 * bits of NTP and the peculiarities of dealing with the Unix system
 * clock.  These routines will run with good precision fairly independently
 * of your kernel's value of tickadj.  I couldn't tell the difference
 * between tickadj==40 and tickadj==5 on a microvax, though I prefer
 * to set tickadj == 500/hz when in doubt.  At your option you
 * may compile this so that your system's clock is always slewed to the
 * correct time even for large corrections.  Of course, all of this takes
 * a lot of code which wouldn't be needed with a reasonable tickadj and
 * a willingness to let the clock be stepped occasionally.  Oh well.
 */
extern	long adj_precision;	/* adj precision in usec (tickadj) */
extern	long tvu_maxslew;	/* maximum adjust doable in 1<<CLOCK_ADJ sec (usec) */

#ifdef SYS_WINNT
extern  long units_per_tick;    /* imported from lib/systime.c module */
#endif /* SYS_WINNT */

extern	u_long tsf_maxslew;	/* same as above, as long format */

extern	l_fp sys_clock_offset;	/* correction for current system time */

#if defined(GDT_SURVEYING)
extern l_fp gdt_rsadj;		/* running sum of adjustments to time */
#endif

/*
 * Import sys_clock (it is updated in get_systime)
 */
extern long sys_clock;

static	void	clock_parms	P((u_long *, u_long *));

/*
 * init_systime - initialize the system clock support code, return
 *		  clock precision.
 *
 * Note that this code obtains to kernel variables related to the local
 * clock, tickadj and tick.  The code knows how the Berkeley adjtime
 * call works, and assumes these two variables are obtainable and are
 * used in the same manner.  Tick is supposed to be the number of
 * microseconds which are added to the system clock at clock interrupt
 * time when the time isn't being slewed.  Tickadj is supposed to be
 * the number of microseconds which are added or subtracted from tick when
 * the time is being slewed.
 *
 * If either of these two variables is missing, or is there but is used
 * for a purpose different than that described, you are SOL and may have
 * to do some custom kludging.
 *
 * This really shouldn't be in here.
 */
void
init_systime()
{
	u_long tickadj;
	u_long tick;
	u_long hz;

	/*
	 * Obtain the values
	 */
#if !defined(VMS)
	clock_parms(&tickadj, &tick);
#else
	{
		extern int get_tickfreq(int *FREQ, int *SYSTICK);
		int freq,systick,sts;

		sts = get_tickfreq(&freq,&systick);
		if(!(sts & 1)) lib$stop(sts);

		tick = 1000000/freq;	/* (usec) */
	}
	tickadj = tick/10;	/* VMS_{ADJTIME,TICTIME}.C will do up to 10% */
#endif /* VMS */
#ifdef	DEBUG
	if (debug)
		printf("kernel vars: tickadj = %ld, tick = %ld\n", tickadj, tick);
#endif

	/*
	 * If tickadj or hz wasn't found, we're doomed.  If hz is
	 * unreasonably small, forget it.
	 */
	if (tickadj == 0 || tick == 0) {
		syslog(LOG_ERR, "tickadj or tick unknown, exiting");
		exit(3);
	}
	if (tick > 65535) {
		syslog(LOG_ERR, "tick value of %lu is unreasonably large",
		    tick);
		exit(3);
	}

	/*
	 * Estimate hz from tick
	 */
	hz = 1000000L / tick;

#ifdef SYS_WINNT
	if ((1000000L % tick) >	tick/2)
		hz +=1;
#endif /* SYS_WINNT */

	/*
	 * Set adj_precision and the maximum slew based on this.  Note
	 * that maxslew is set slightly shorter than it needs to be as
	 * insurance that all slews requested will complete in 1<<CLOCK_ADJ
	 * seconds.
	 */
#ifdef ADJTIME_IS_ACCURATE
	adj_precision = 1;
#else
# ifndef SYS_WINNT
	adj_precision = tickadj;
# else
	/* minimum adjustment is one 100 nanosecond unit at each clock tick */
	adj_precision = (long) ((1<<CLOCK_ADJ) * hz * 0.1);
# endif /* SYS_WINNT */
#endif /* ADJTIME_IS_ACCURATE */
#if defined(SYS_HPUX) && (SYS_HPUX < 10)
	/*
	 * when using adjtimed daemon, need to allow more time
	 * because daemon may not run right away
	 */
	tvu_maxslew = tickadj * (hz-3) * (1<<CLOCK_ADJ);
#else
#if defined(SLEWALWAYS) && !defined(ADJTIME_IS_ACCURATE)
	/*
	 * give us more time if we are always slewing... just in case
	 */
	tvu_maxslew = tickadj * (hz-3) * (1<<CLOCK_ADJ);
#else
# ifndef SYS_WINNT
	tvu_maxslew = tickadj * (hz-1) * (1<<CLOCK_ADJ);
# else
	tvu_maxslew = tickadj * hz * (1<<CLOCK_ADJ);
# endif /* SYS_WINTNT */
#endif /* SLEWALWAYS */
#endif /* SYS_HPUX */
	if (tvu_maxslew > 999990) {
		/*
		 * Don't let the maximum slew exceed 1 second in 4.  This
		 * simplifies calculations a lot since we can then deal
		 * with less-than-one-second fractions.
		 */
		tvu_maxslew = (999990/adj_precision) * adj_precision;
	}
	TVUTOTSF(tvu_maxslew, tsf_maxslew);
	NLOG(NLOG_SYSINFO) /* conditional if clause for conditional syslog */
	  syslog(LOG_NOTICE, "tickadj = %d, tick = %d, tvu_maxslew = %d",
	    tickadj, tick, tvu_maxslew);
#ifdef DEBUG
	if (debug)
		printf(
	"adj_precision = %ld, tvu_maxslew = %ld, tsf_maxslew = 0.%08lx\n",
		    adj_precision, tvu_maxslew, tsf_maxslew);
#endif

	/*
	 * Set the current offset to 0
	 */
	L_CLR(&sys_clock_offset);
#if defined(GDT_SURVEYING)
	/*
	 * Set the current increment
	 */

	L_CLR(&gdt_rsadj);
#endif /* GDT_SURVEYING */
}


#if !defined(VMS)

# ifndef HZ
#  define HZ	60
# endif

# ifdef NLIST_STRUCT
#  ifdef NLIST_NAME_UNION
#   define NL_B {{
#   define NL_E }}
#  else
#   define NL_B {
#   define NL_E }
#  endif
# endif

#if !defined(K_TICKADJ_NAME)
# define K_TICKADJ_NAME	"_tickadj"
#endif
#if !defined(K_TICK_NAME)
# define K_TICK_NAME	"_tick"
#endif

/*
 * clock_parms - return the local clock tickadj and tick parameters
 *
 */
static void
clock_parms(ptickadj, ptick)
	u_long *ptickadj;
	u_long *ptick;
{
	u_long tick;
#ifdef SYS_WINNT
	DWORD add, every;
	BOOL noslew;
#else
#if defined(HAVE_SYSCONF) && defined(_SC_CLK_TCK)
	int hz;
#endif
#ifndef NOKMEM
	static struct nlist nl[] = {
# define N_TICKADJ 0
		NL_B K_TICKADJ_NAME NL_E,
# define N_TICK 1
		NL_B K_TICK_NAME NL_E,
		NL_B "" NL_E,
	};
# ifdef HAVE_K_OPEN
# else /* not HAVE_K_OPEN */
#  ifdef HAVE_KVM_OPEN
	register kvm_t *kd;
#  else /* not HAVE_KVM_OPEN */
	register int i;
	int kmem;
	struct stat stbuf;
	int vars[2];
	off_t where;
#   ifdef HAVE_BOOTFILE
	const char *kernelname;
#   else /* not HAVE_BOOTFILE */
	static char *kernelnames[] = {
		"/kernel/unix",
		"/kernel",
		"/vmunix",
		"/unix",
		"/mach",
		"/hp-ux",
		"/386bsd",
		"/netbsd",
#    ifdef KERNELFILE
		KERNELFILE,
#    endif
		NULL
	};
#   endif /* not HAVE_BOOTFILE */
#  endif /* not HAVE_KVM_OPEN */
# endif /* not HAVE_K_OPEN */
#endif /* not NOKMEM */
#endif /* not SYS_WINNT */

#ifdef SYS_WINNT
	if (!GetSystemTimeAdjustment(&add, &every, &noslew)) {
		*ptick = 0;
		*ptickadj = 0;
		return;
	}
	units_per_tick = add;
#else /* not SYS_WINNT */
#if defined(HAVE_SYSCONF) && defined(_SC_CLK_TCK)
	hz = (int) sysconf (_SC_CLK_TCK);
#endif

    {
#ifndef NOKMEM
# ifdef HAVE_K_OPEN /* { */
	if (K_open((char *)0, O_RDONLY, "/vmunix")!=0) {
		syslog(LOG_ERR, "K_open failed");
		exit(3);
	}
	kusenlist(1);
	if (knlist(nl) != 0) {
		syslog(LOG_ERR, "knlist failed");
		exit(3);
	}
# else  /* } not HAVE_K_OPEN { */
#  ifdef HAVE_KVM_OPEN /* { */
	if ((kd = kvm_open(NULL, NULL, NULL, O_RDONLY, NULL)) == NULL) {
		syslog(LOG_ERR, "kvm_open failed");
		exit(3);
	}
	if (kvm_nlist(kd, nl) != 0) {
		syslog(LOG_ERR, "kvm_nlist failed");
		exit(3);
	}
#  else /* } not HAVE_KVM_OPEN { */
#   ifdef HAVE_GETBOOTFILE
	kernelname = getbootfile();
	if (!kernelname)
	  {
	    syslog(LOG_ERR, "getbootfile() didn't find kernel name!");
	    exit(3);
	  }
	else if (stat(kernelname, &stbuf) == -1)
	  {
	    syslog(LOG_ERR, "stat(%s) failed: %m", kernelname);
	    exit(3);
	  }
	else if (nlist(kernelname, nl) < 0)
	  {
	    syslog(LOG_ERR, "nlist(%s) failed: %m", kernelname);
	    exit(3);
	  }  
#   else /* not HAVE_GETBOOTFILE */
	for (i = 0; kernelnames[i] != NULL; i++) {
		if (stat(kernelnames[i], &stbuf) == -1)
			continue;
		if (nlist(kernelnames[i], nl) >= 0)
			break;
	}
	if (kernelnames[i] == NULL) {
		syslog(LOG_ERR, "Clock init couldn't find kernel object file");
		exit(3);
	}
#   endif /* not HAVE_GETBOOTFILE */
	kmem = open("/dev/kmem", O_RDONLY);
	if (kmem < 0) {
		syslog(LOG_ERR, "Can't open /dev/kmem for reading: %m");
#ifdef	DEBUG
		if (debug)
			perror("/dev/kmem");
#endif
		exit(3);
	}
#  endif /* } not HAVE_KVM_OPEN */
# endif /* } not HAVE_K_OPEN */
#endif /* not NOKMEM */
#endif /* not SYS_WINNT */
    }

    /* Skippy says we need to know TICK before handling TICKADJ */
    {
#if defined(HAVE_SYS_TIMEX_H) && (HAVE___ADJTIMEX)
	struct timex txc;
#endif /* HAVE_SYS_TIMEX_H && HAVE___ADJTIMEX */

#ifdef PRESET_TICK
# if defined(HAVE_SYS_TIMEX_H) && defined(HAVE___ADJTIMEX)
#  ifdef MOD_OFFSET
	txc.modes = 0;
#  else
	txc.mode = 0;
#  endif
	__adjtimex(&txc);
# endif /* HAVE_SYS_TIMEX_H && HAVE___ADJTIMEX */
	*ptick = (u_long) PRESET_TICK;
#else  /* PRESET_TICK */
# ifdef NOKMEM
#  include "Bletch: NOKMEM and no PRESET_TICK"
# endif /* NOKMEM */
# ifdef HAVE_K_OPEN
	if (K_read(ptick,sizeof(*ptick),nl[N_TICK].n_value) !=
	    sizeof(*ptick)) {
		syslog(LOG_ERR, "K_read tick failed");
		exit(3);
	}
# else /* not HAVE_K_OPEN */
#  ifdef HAVE_KVM_OPEN
	if (kvm_read(kd, nl[N_TICK].n_value, (char *)ptick, sizeof(*ptick)) !=
	    sizeof(*ptick)) {
		syslog(LOG_ERR, "kvm_read tick failed");
		exit(3);
	}
#  else /* not HAVE_KVM_OPEN */
	if ((where = nl[N_TICK].n_value) == 0) {
		syslog(LOG_ERR, "Unknown kernel var %s",
		       nl[N_TICK].N_NAME);
		exit(3);
	}
	if (lseek(kmem, where, SEEK_SET) == -1) {
		syslog(LOG_ERR, "lseek for %s fails: %m",
		       nl[N_TICK].N_NAME);
		exit(3);
	}
	if (read(kmem, ptick, sizeof(*ptick)) != sizeof(*ptick)) {
		syslog(LOG_ERR, "read for %s fails: %m",
		       nl[N_TICK].N_NAME);
		exit(3);
	}
#   ifdef NLIST_EXTRA_INDIRECTION
	/*
         * Aix requires one more round of indirection
	 * if storage class a pointer.
         */
	if ( nl[N_TICK].n_sclass == 0x6b) {
		if (lseek(kmem, *ptick, SEEK_SET) == -1) {
			syslog(LOG_ERR, "lseek2 for %s fails: %m",
			    nl[N_TICK].N_NAME);
			exit(3);
		}
		if (read(kmem, ptick, sizeof(*ptick)) != sizeof(*ptick)) {
			syslog(LOG_ERR, "read2 for %s fails: %m",
			    nl[N_TICK].N_NAME);
			exit(3);
		}
	}
#   endif
#  endif /* not HAVE_KVM_OPEN */
# endif /* not HAVE_K_OPEN */
#endif /* not PRESET_TICK */
	
	tick = *ptick;		/* This might be used by PRESET_TICKADJ */
    }

    /* Skippy says we need to know TICK before handling TICKADJ */
    {
#ifdef NOKMEM
# ifdef PRESET_TICKADJ
	*ptickadj = (u_long) PRESET_TICKADJ;
# else  /* PRESET_TICKADJ */
#  include "Bletch: NOKMEM and no PRESET_TICKADJ"
# endif /* PRESET_TICKADJ */
#else  /* not NOKMEM */
	if (nl[N_TICKADJ].n_value == 0)
	  {
	    syslog(LOG_ERR, "nlist didn't find a value for <%s>",
		   nl[N_TICKADJ].N_NAME);
	    exit(3);
	  }
# ifdef HAVE_K_OPEN
	if (K_read(ptickadj,sizeof(*ptickadj),nl[N_TICKADJ].n_value) !=
	    sizeof(*ptickadj)) {
		syslog(LOG_ERR, "K_read tickadj failed");
		exit(3);
	}
# else /* not HAVE_K_OPEN */
#  ifdef HAVE_KVM_OPEN
	if (kvm_read(kd, nl[N_TICKADJ].n_value, (char *)ptickadj, sizeof(*ptickadj)) !=
	    sizeof(*ptickadj)) {
		syslog(LOG_ERR, "kvm_read tickadj failed");
		exit(3);
	}
#  else /* not HAVE_KVM_OPEN */
	if ((where = nl[N_TICKADJ].n_value) == 0) {
		syslog(LOG_ERR, "Unknown kernel var %s",
		       nl[N_TICKADJ].N_NAME);
		exit(3);
	}
	if (lseek(kmem, where, SEEK_SET) == -1) {
		syslog(LOG_ERR, "lseek for %s fails: %m",
		       nl[N_TICKADJ].N_NAME);
		exit(3);
	}
	if (read(kmem, ptickadj, sizeof(*ptickadj)) != sizeof(*ptickadj)) {
		syslog(LOG_ERR, "read for %s fails: %m",
		       nl[N_TICKADJ].N_NAME);
		exit(3);
	}
#   ifdef NLIST_EXTRA_INDIRECTION
	/*
         * Aix requires one more round of indirection
	 * if storage class a pointer.
         */
	if ( nl[N_TICKADJ].n_sclass == 0x6b) {
		if (lseek(kmem, *ptickadj, SEEK_SET) == -1) {
			syslog(LOG_ERR, "lseek2 for %s fails: %m",
			    nl[N_TICKADJ].N_NAME);
			exit(3);
		}
		if (read(kmem, ptickadj, sizeof(*ptickadj)) != sizeof(*ptickadj)) {
			syslog(LOG_ERR, "read2 for %s fails: %m",
			    nl[N_TICKADJ].N_NAME);
			exit(3);
		}
	}
#   endif
#  endif /* not HAVE_KVM_OPEN */
# endif /* not HAVE_K_OPEN */
#endif /* not NOKMEM */
    }

#ifndef NOKMEM
# ifdef HAVE_K_OPEN
	(void) K_close();
# else /* not HAVE_K_OPEN */
#  ifdef HAVE_KVM_OPEN
	if (kvm_close(kd) < 0) {
		syslog(LOG_ERR, "kvm_close failed");
		exit(3);
	}
#  else /* not HAVE_KVM_OPEN */
	close(kmem);
#  endif /* not HAVE_KVM_OPEN */
# endif /* not HAVE_K_OPEN */
#endif /* not NOKMEM */

#ifdef	DEBUG
	if (debug)
		printf("tick = %d, tickadj = %d\n", *ptick, *ptickadj);
#endif
}
#endif /* not VMS */
