/***************************************************************************
 * LPRng - An Extended Print Spooler System
 *
 * Copyright 1988-1995 Patrick Powell, San Diego State University
 *     papowell@sdsu.edu
 * See LICENSE for conditions of use.
 *
 ***************************************************************************
 * MODULE: fixcontrol.c
 * PURPOSE: fix order of lines in control file
 **************************************************************************/

static char *const _id =
"$Id: fixcontrol.c,v 3.0 1996/05/19 04:05:58 papowell Exp $";
/********************************************************************
 * int Fix_control( struct control_file *cf, char *order )
 *   fix the order of lines in the control file so that they
 *   are in the order of the letters in the order string.
 * Lines are checked for metacharacters and other trashy stuff
 *   that might have crept in by user efforts
 *
 * cf - control file area in memory
 * order - order of options
 *
 *  order string: Letter - relative position in file
 *                * matches any character not in string
 *                  can have only one wildcard in string
 *   Berkeley-            HPJCLIMWT1234
 *   PLP-                 HPJCLIMWT1234*
 *
 * RETURNS: 0 if fixed correctly
 *          non-zero if there is something wrong with this file and it should
 *          be rejected out of hand
 ********************************************************************/

#include "lp.h"
#include "printcap.h"

static char *orderstr;

/********************************************************************
 * BSD and LPRng order
 * We use these values to determine the order of jobs in the file
 * The order of the characters determines the order of the options
 *  in the control file.  A * puts all unspecified options there
 ********************************************************************/

static char Bsd_order[]
  = "HPJCLIMWT1234"
;

static char LPRng_order[]
  = "HPJCLIMWT1234*"
;


int ordercomp( const void *left, const void *right )
{
	char *pos;
	int l, r, c;
	l = r = 0;

	c = **((char **)left);
	pos = strchr( orderstr, c );
	if( pos == 0 ) pos = strchr( orderstr, '*' );
	if( pos ) l = pos - orderstr;
	c = **((char **)right);
	pos = strchr( orderstr, c );
	if( pos == 0 ) pos = strchr( orderstr, '*' );
	if( pos ) r = pos - orderstr;
	DEBUG8("ordercomp '%s' to '%s', l %d, r %d -> %d",
		*(char **)left,*(char **)right,l,r,l-r);
	return( l - r );
}

int Fix_control( struct control_file *cf, char **bad )
{
	int i, c;		/* ACME Integers and Counters, Inc */
	char *line;				/* line in file */
	char *s;				/* ACME Pointers and Chalkboards, Inc */
	char *last_option = 0;	/* last option in order string */
	int reorder = 0;		/* reorder */
	char *order;
	char **lines;			/* lines in file */

	/* if no order supplied, don't check */

    if( Backwards_compatible ){
        order = Bsd_order;
		/* remove any non-listed options */
		for( i = 'A'; i <= 'Z'; ++i ){
			if( !strchr("CHIJLNMPSTUW", i ) && cf->capoptions[i-'A'] ){
			DEBUG3("Fix_control: removing option '%c'='%s'",
				i, cf->capoptions[i-'A'] );
				cf->capoptions[i-'A'][0] = 0;
				cf->capoptions[i-'A'] = 0;
			}
		}
    } else {
        order = LPRng_order;
    }


	DEBUG6("Fix_control: order '%s', line_count %d, control_info %d",
		order, cf->info_lines.count, cf->control_info );
	if( Debug > 6) dump_control_file( "Fix_control: before", cf );

	orderstr = order;
	*bad = 0;

	DEBUG6("Fix_control: Use_queuename %d, Queuename '%s', Printer %s",
		Use_queuename, cf->QUEUENAME, Printer );

	/* check to see if we need to insert the Q entry */
	/* if we do, we insert this at the head of the list */
	if( !Backwards_compatible && Use_queuename
		&& cf->QUEUENAME == 0 && Printer ){
		char buffer[M_QUEUENAME];
		if( order && ( strchr( order, 'Q' ) || strchr( order, '*' )) ){
			plp_snprintf(buffer, sizeof(buffer)-1, "Q%s", Printer );
			line = Prefix_job_line( cf, buffer );
			cf->QUEUENAME = line;
			++cf->control_info;
		} else {
			log( LOG_INFO,
		"Fix_control: '%s' needs Q entry, forbidden by order restrictions",
					cf->name );
		}
	}

	/*
	 * we check to see if there is a metacharacter embedded on
	 * any line of the file.
	 */
	lines = (void *)cf->info_lines.list;
	for( i = 0; i < cf->info_lines.count; ++i ){
		/* get line and first character on line */
		line = lines[i];
		c = *line;
		if( (s = Find_meta( line+1 )) ){
			log( LOG_INFO,
				"Fix_control: '%s' has metacharacter 0x%02x '%c' in line '%s'",
					cf->name, *s, isprint(*s)?*s:'?', line );
			Clean_meta( s );
		}
	}

	/*
	 * we check to see if order is correct - we need to check to
	 * see if allowed options in file first.
	 */

	for( i = 0; order && reorder == 0 && i < cf->control_info; ++i ){
		/* get line and first character on line */
		line = lines[i];
		c = *line;
		if( c == 'U' || c == 'N' ) continue;
		/* look for letter in order string */
		s = strchr( order, c );
		if( s == 0 ){
			/* not found in the file, check for * */
			if( last_option ){
				s = strchr( last_option, '*' );
			} else {
				s = strchr( order, '*' );
			}
		}
		/* if not found, you have a problem */
		if( s == 0 ){
			logerr( LOG_INFO,
				"control file '%s' has forbidden option '%s'",
				cf->name, line );
			*bad = line;
			return(1);
		}
		if( last_option && last_option > s ){
			DEBUG6( "out of order '%s'", line );
			reorder = 1;
		}
		last_option = s;
	}
	DEBUG6( "control file '%s' reorder %d, order '%s'", cf->name,
		reorder, order );
	/* found the 'N' or data file */

	if( reorder ){
		qsort( lines, cf->control_info, sizeof( char *), ordercomp );
	}

	if( Debug > 6) dump_control_file( "Fix_control: after", cf );
	return( 0 );
}
