// -*- C++ -*-
//
// Files in egfs
//
// This program is distributed under the terms of the
// Free Software Foundation General Public Licence.
// See COPYING for details
// Copyright Jeremy Fitzhardinge <jeremy@sw.oz.au> 1994,1995

#pragma implementation

#include "egfile.h"

#include <unistd.h>
#include <sys/stat.h>

// Create a file with some initial content
egfile::egfile(Filesystem &fs, Handle h, char *b, int sz)
       : SimpleInode(fs, h)
{
	// Use our invoker's ID rather than the person doing the syscall.
	uid = getuid();
	gid = getgid();
	mode = S_IFREG | 0644;
	if (sz == -1)
		sz = strlen(b);
	buf = new char [sz];
	size = sz;
	memcpy(buf, b, sz);
}

// Set the length of the file.  Very naïf: it just frees the old buffer
// and allocates the new one, without trying to optimise at all.
void
egfile::truncate(off_t off)
{
	char *nb = new char[off];

	memset(nb, 0, off);
	memcpy(nb, buf, off > size ? size : off);
	delete buf;
	buf = nb;
	size = off;
}

// Not much to reading
int
egfile::do_read(const up_preamble &, upp_repl &,
		const upp_read_s &arg, upp_read_r &ret)
{
	size_t sz = size - arg.off;

	// Reads beyond EOF always get 0 bytes
	if (sz <= 0)
	{
		ret.data.nelem = 0;
		return 0;
	}

	// Read either the amount wanted or to the end of the file.
	sz = sz > arg.size ? arg.size : sz;

	// Just point the return to the data
	ret.data.nelem = sz;
	ret.data.elems = &buf[arg.off];
	return 0;
}

// Writing files could be much more efficient.  As it is, each write
// which grows the file will cause truncate() to alloc, copy and free
// the buffer with the file contents.
int
egfile::do_write(const up_preamble &, upp_repl &,
		 const upp_write_s &arg, upp_write_r &ret)
{
	// Writing off the end of the file means we grow it
	if (arg.off + arg.data.nelem > size)
		truncate(arg.off + arg.data.nelem);

	// Copy the data into place
	memcpy(buf+arg.off, arg.data.elems, arg.data.nelem);

	// We can never do partial writes
	ret.wrote = arg.data.nelem;
	return 0;
}

// Set the size of the file (truncate is historical; it can also grow
// files)
int
egfile::do_truncate(const up_preamble &, upp_repl &,
		    const upp_truncate_s &arg)
{
	truncate(arg.size);
	return 0;
}
