// -*- C++ -*-

// Handle FTP connections

// (C) Copyright 1994 Jeremy Fitzhardinge <jeremy@sw.oz.au>
// This code is distributed under the terms of the
// GNU General Public Licence.  See COPYING for more details.

#pragma implementation

#include "ftpconn.h"
#include "io.h"
#include "LWP.h"
#include "ConfFile.h"
#include "serv_port.h"

// Host types
#include "unix_host.h"
#include "dos_host.h"

#include <netdb.h>
#include <stdlib.h>
#include <unistd.h>
#include <sys/socket.h>
#include <errno.h>
#include <stdio.h>
#include <string.h>
#include <ctype.h>
#include <arpa/telnet.h>
#include <arpa/inet.h>
#include <assert.h>

#include <String.h>
#include <Obstack.h>
#include <fstream.h>

static const serv_port ftp_port("ftp", "tcp");

class semlock
{
	Semaphore &sem;
public:
	semlock(Semaphore &sem):sem(sem)	{ sem.wait(); }
	~semlock()				{ sem.signal(); }
};

ftpconn::ftpconn(const Path &conffile, const String &host)
	:hostname(host), conf(conffile), sem(1)
{
	struct hostent *he = gethostbyname((const char *)hostname);

	if (he == NULL)
	{
		cerr << "Hostname lookup for " << host << " failed: " << flush;
		herror("");
	}
	
	init(he);
}

ftpconn::ftpconn(const Path &conffile, const String &host,
		 const struct hostent *he)
	:hostname(host), conf(conffile), sem(1)
{
	init(he);
}

void
ftpconn::init(const struct hostent *he)
{
	ref = 0;
	need_tz = 1;
	need_siteroot = 1;
	timezone = 0;
	ctlsock = datasock = -1;
	inbuf = NULL;
	memset(&addr, 0, sizeof(addr));
	memset(&myaddr, 0, sizeof(myaddr));
	state = Unresolved;

	getparams();

	if (state == Unresolved)
	{
		if (resolve_name(he) != 0)
			return;

		state = Resolved;
	}
}

static unix_host unix_host_type;
static dos_host dos_host_type;

host *host_types[] =
{
	&unix_host_type,
	&dos_host_type,
};
		
// Various parameters for this site/connection
void
ftpconn::getparams()
{
	if (!conf.get("username", username))
		username="anonymous";
	if (!conf.get("password", passwd))
		passwd="ftpfs";
	if (!conf.getnum("update_timeout", upd_timeout))
		upd_timeout = 60*60;
	if (!conf.getnum("fail_timeout", fail_timeout))
		fail_timeout = 120;
	if (conf.get("site_root", site_root))
		need_siteroot = 0;
	
	if (conf.getnum("timezone", timezone))
		need_tz = 0;

	String hosttype;
	site_host = NULL;
	if (conf.get("hosttype", hosttype))
	{
		for (unsigned i = 0;
		     i < sizeof(host_types)/sizeof(*host_types);
		     i++)
			if (hosttype == host_types[i]->hosttype())
				site_host = host_types[i];
	}
	if (site_host == NULL)
		site_host = &unix_host_type;
	
	String dotaddr;
	if (conf.get("ip_address", dotaddr))
		if ((addr.sin_addr.s_addr = inet_addr(dotaddr)) != -1)
		{
			addr.sin_port = ftp_port.port;
			addr.sin_family = AF_INET;
			state = Resolved;
		}
}

void
ftpconn::setparams()
{
	if (state >= Resolved)
	{
		String dotaddr(inet_ntoa(addr.sin_addr));

		conf.set("ip_address", dotaddr);
	}
	if (!need_tz)
		conf.setnum("timezone", timezone);
	if (!need_siteroot)
		conf.set("site_root", site_root);
	
	conf.set("hosttype", site_host->hosttype());
}

ftpconn::~ftpconn()
{
	setparams();
	if (state > Resolved)
		goto_state(Resolved);
	if (inbuf)
		delete inbuf;
	cout << "FTP connection to " << hostname << " closed\n";
}

ftpstate_t
ftpconn::resolve(ftpstate_t)
{
	assert(state == Unresolved);
	
	struct hostent *he = gethostbyname(hostname);

	if (resolve_name(he))
		return Bad;
	return Resolved;
}

int
ftpconn::resolve_name(const struct hostent *he)
{
	if (he == NULL)
		return -1;

	memset(&addr, 0, sizeof(addr));
	addr.sin_family = he->h_addrtype;
	memcpy(&addr.sin_addr, he->h_addr_list[0], sizeof(he->h_length));
	addr.sin_port = ftp_port.port;

	return 0;
}

ftpstate_t
ftpconn::connect(ftpstate_t)
{
	if (need_tz)
	{
		time_t now = time(0);		// UTC
		time_t there = site_host->localtime(addr);
		time_t diff = (now-there)/60;

		if (there != (time_t)-1)
		{
			time_t d = diff;
			cout << "Time diff = "<<d/60<<"hr "<<d%60<<"min\n";
			timezone = diff;
			need_tz = 0;
		}
	}
		
	if ((ctlsock = socket(AF_INET, SOCK_STREAM, 0)) == -1)
	{
		perror("ftpconn::connect socket() failed");
		return Bad;
	}

	if (::connect(ctlsock, (struct sockaddr *)&addr, sizeof(addr)) == -1)
	{
		perror("ftpconn::connect connect() failed");
		::close(ctlsock);
		return Bad;
	}

	int len = sizeof (myaddr);
	if (getsockname(ctlsock, (struct sockaddr *)&myaddr, &len) < 0)
	{
		perror("ftpconn::connect getsockname() failed");
		::close(ctlsock);
		return Bad;
	}

	inbuf = new ifstream(ctlsock);
	inbuf->unsetf(ios::skipws);
	
	if (getreply() != 220)
	{
		cerr << "Failed to connect properly\n";
		::close(ctlsock);
		delete inbuf;
		inbuf = NULL;
		return Bad;
	}

	cout << "Connected OK to " << hostname << '\n';
	
	return Connected;
}

ftpstate_t
ftpconn::close(ftpstate_t)
{
	::close(ctlsock);
	::close(datasock);

	ctlsock = datasock = -1;

	return Resolved;
}

ftpstate_t
ftpconn::login(ftpstate_t)
{
	int ret = 0;
	
	if (sendcmd("USER "+username) != 331)
		return Bad;

	if (sendcmd("PASS "+passwd) != 230)
		return Bad;

	if (need_siteroot && ftpcwd(site_root)/100 != 2)
	{
		cout << "Failed to get CWD\n";
		site_root = ".";
	}
	else
		need_siteroot = 0;
	
	if (ftpsetcwd(site_root) != 250)
	{
		goto_state(Resolved);
		return Bad;
	}
	
	if (ret != 0)
		cout << "Login as " << username <<
			", passwd " << passwd << " failed\n";
	else
	{
		sendcmd("SYST");
		sendcmd("STAT");
	}
	
	return LoggedIn;
}

ftpstate_t
ftpconn::logout(ftpstate_t)
{
	sendcmd("QUIT");
	return Connected;
}

ftpstate_t
ftpconn::listen(ftpstate_t)
{
	dataaddr = myaddr;		// Our address
	dataaddr.sin_port = 0;		// System port

	closedata(LoggedIn);

	if ((datasock = socket(AF_INET, SOCK_STREAM, 0)) == -1)
	{
		perror("creation of data socket failed");
		return Bad;
	}

	if (::bind(datasock, (struct sockaddr *)&dataaddr, sizeof(dataaddr)) == -1)
	{
		perror("bind of data socket");
		closedata(LoggedIn);
		return Bad;
	}

	int len = sizeof (dataaddr);
	if (getsockname(datasock, (struct sockaddr *)&dataaddr, &len) == -1)
	{
		perror("data socket getsockname");
		closedata(LoggedIn);
		return Bad;
	}

	if (::listen(datasock, 1) == -1)
	{
		perror("listen on data socket");
		closedata(LoggedIn);
		return Bad;
	}

	char *addr = (char *)&dataaddr.sin_addr;
	char *port = (char *)&dataaddr.sin_port;
	char buf[128];
	
#define	UC(b)	(((int)b)&0xff)
	sprintf(buf, "PORT %d,%d,%d,%d,%d,%d",
		UC(addr[0]), UC(addr[1]), UC(addr[2]), UC(addr[3]),
		UC(port[0]), UC(port[1]));
#undef UC
	if (sendcmd(buf)/100 != 2)
	{
		cerr << "Send of \"" << buf << "\" failed\n";
		return Bad;
	}

	return Listening;
}

ftpstate_t
ftpconn::closedata(ftpstate_t nst)
{
	::close(datasock);
	datasock = -1;
	return nst;
}


ftpstate_t
ftpconn::abort(ftpstate_t nst)
{
	if (state == Transfer)
		sendcmd("ABOR");
	return nst;
}

static const char *st_name[] =
{
	"Unresolved",
	"Resolved",
	"Connected",
	"LoggedIn",
	"Listening",
	"Transfer",
	"Bad"
};

int
ftpconn::goto_state(ftpstate_t newstate)
{
	typedef ftpstate_t (ftpconn::*trfunc)(ftpstate_t);

	static trfunc trtab[6][6] = 
	{
		{ NULL, &resolve, &resolve, &resolve, &resolve, NULL },
		{ NULL, NULL, &connect, &connect, &connect, NULL },
		{ NULL, &close, NULL, &login, &login, NULL },
		{ NULL, &logout, &logout, NULL, &listen, NULL },
		{ NULL, &closedata, &closedata, &closedata, NULL, &closedata },
		{ NULL, &abort, &abort, &abort, &abort, NULL }
	};

	while(state != newstate)
	{
		assert(state < Bad);
		assert(newstate < Bad);
		
		trfunc tr = trtab[state][newstate];
		
		if (tr == NULL)
		{
			cerr << "No transition from " << st_name[state]
				<< " to " << st_name[newstate] << '\n';
			return -1;
		}

		ftpstate_t next = (tr)(newstate);

		if (next == Bad)
		{
			cerr << "Transition from " << st_name[state]
				<< " to " << st_name[newstate] << " failed\n";
			return -1;
		}
		state = next;
	}

	return 0;
}

/*
 * Send a command to the ftp server.
 *
 * All we do is pack the command off to the other server, and call getreply()
 * to deal with the response.  We just return the code from getreply().
 *
 * It returns either the code from getreply() or   return with -1 and errno
 * set if there is a syscall failure.
 */
int
ftpconn::sendcmd(String command, String *repl)
{
	int	ret;
	int	retry = 0;

	cout << "Sending command \"" << command << "\"\n";
	command += "\r\n";
	
	if (fullwrite(ctlsock, (const unsigned char *)command,
		      command.length()) != command.length())
	{
		perror("fullwrite of command failed");
		return -1;
	}
		
	ret = getreply(repl);

	return ret;
}

/*
 * Get a reply from the ftp daemon, and deal with it.
 *
 * This is all dealing with the FTP protocol - see Internet RFC 959
 * to find out what all this really means.
 *
 * These are the RFC 959 return codes, attached to the beginning of each
 * reply from the remote machine.
 *
 *	FTP Code		Meaning
 *	1xx			success (positive preliminary reply)
 *	2xx			success (positive completion reply)
 *	3xx			success (positive intermediate reply)
 *	4xx			failure	(transiant negative completion reply)
 *	5xx			failure	(permanent negative completion reply)
 *
 *
 * The code returned is either '-1' if some system call failure happens,
 * or the code number attached to the reply.
 */
int
ftpconn::getreply(String *string)
{
	unsigned char	ch;
	int	digit;		// Which digit we're processing
	int	code;		// Numeric code
	Obstack	reply;		// Reply in string form
	
	digit = 0;
	code = 0;

	while((*inbuf >> ch) && ch != '\n')
	{
		if (ch == IAC)
		{
			char telcode[4];
			
			/*
			 * Deal with telnet protocol commands by ignoring them
			 */
			*inbuf >> ch;
			switch (ch)
			{
			case WILL:
			case WONT:
				*inbuf >> ch;
				sprintf(telcode, "%c%c%c", IAC, DONT, ch);
				fullwrite(ctlsock, (unsigned char *)telcode, 3);
				break;
			case DO:
			case DONT:
				*inbuf >> ch;
				sprintf(telcode, "%c%c%c", IAC, WONT, ch);
				fullwrite(ctlsock, (unsigned char *)telcode, 3);
				break;
			default:
				break;
			}
			continue;
		}

		if (ch == '\r')
			continue;

		reply.grow(ch);

		if (digit < 3 && isdigit(ch))
			code = code*10 + ch-'0';

		/*
		 * Handle multi-line messages
		 * These have the form:
		 * xyz-message
		 *  message contents
		 *  multiple lines
		 * xyz end message
		 */
		if (digit == 3 && ch == '-')
		{
			String	line;
			char	codebuf[5];	/* String at end of message */
			int	ret;
			
			sprintf(codebuf, "%3.03d ", code);
			do
			{
				ret = readline(*inbuf, line);
				if (line.contains('\r'))
					line = line.before('\r');
				reply.grow((const char *)line, line.length());
				reply.grow('\n');
			} while(inbuf->good() && codebuf != line.at(0,4));
			
			if (!inbuf->good())
			{
				perror("inbuf->readline failed");
				state = Resolved;
				close(Resolved);
				return -1;
			}
			break;
		}
		else if (digit == 3 && ch != ' ')
			fprintf(stderr, "RFC 959 violation: Unexpected character after digits - 0x%02x, not ' '\n", ch);
		digit++;
	}

	if (inbuf->eof() || inbuf->bad())
	{
		// If we get an EOF, then just go straight to Resolved state
		close(Resolved);
		state = Resolved;
		return -1;
	}
	char *buf = (char *)reply.finish(0);
	if (string != NULL)
		*string = String(buf);

	cout << "Message \"" << buf << "\"\n" <<
		"getreply() returning " << code << '\n';
	
	return code;
}

class ftp_return
{
	const String &p;
	ftpconn &ftp;
		
 public:
	ftp_return(ftpconn &ftp, const String &p)
		:p(p),ftp(ftp)
		{
		}
	~ftp_return()
	{
		ftp.ftpsetcwd(p, 1);
	}
};

// Get a file list from an ftp server
// Calls host-specific code to parse the directory listing
ftp_filelist *
ftpconn::getfilelist(const String &dname)
{
	String dirname = dname;
	String path = dirname.before('/', -1);
	String dir = dirname.after('/', -1);
	const String &wildcard = site_host->wildcard();

	if (dir == "")
		dir = dname;
	
	semlock lock(sem);

	cout << "ftpconn::getfilelist getting dir on host " << hostname
		<< ", dirname=" << dirname << ", basedir=" << path
		<< ", subdir=" << dir << "\n";

	if (goto_state(LoggedIn))
	{
		fprintf(stderr, "Failed to log in\n");
		return NULL;
	}

	if (sendcmd("TYPE A") != 200)
	{
		fprintf(stderr, "Failed to set TYPE A\n");
		return NULL;
	}

	ftp_return ftp_here(*this, site_root);
	if (path != "" && ftpsetcwd(path) != 250)
	{
		cerr << "Failed to change to \"" << path << "\"\n";
		return NULL;
	}
	
	if (goto_state(Listening))
	{
		fprintf(stderr, "Failed to listen\n");
		return NULL;
	}

	if (sendcmd("LIST "+dir+wildcard)/100 != 1)
	{
		fprintf(stderr, "Failed to send LIST\n");
		return NULL;
	}

	struct sockaddr_in from;
	int fromlen = sizeof(from);
	int sfd;

	if ((sfd = ::accept(datasock, (struct sockaddr *)&from, &fromlen)) == -1)
	{
		perror("Failed to accept()");
		goto_state(LoggedIn);
		return NULL;
	}
	closedata(LoggedIn);
	
	ifstream in(sfd);
	String line;
	ftp_filelist *fl = NULL;

	int ret;
	while((ret = readline(in, line)) > 0)
	{
		if (line.contains('\r'))
			line = line.before('\r');
		
		ftp_filelist *nfl = site_host->parse_filelist(line, timezone);

		if (nfl == NULL)
			continue;
		nfl->next = fl;
		fl = nfl;
	}

	::close(sfd);
	getreply();
	goto_state(LoggedIn);
	
	return fl;
}

int
ftpconn::getfile(const String &fname, int fd, ftpxfer *xfer)
{
	String filename = fname;
	String path = filename.before('/', -1);
	String file = filename.after('/', -1);
	if (file == "")
		file = fname;
	
	semlock lock(sem);

	cout << "ftpconn::getfile getting file on host " << hostname
		<< ", dir " << path << ", file " << file << "\n";

	if (goto_state(LoggedIn))
	{
		fprintf(stderr, "Failed to log in\n");
		if (xfer != NULL)
			xfer->error(EPERM);
		return -1;
	}

	ftp_return ftp_here(*this, site_root);
	if (path != "" && ftpsetcwd(path) != 250)
	{
		cerr << "Failed to change to \"" << path << "\n";
		return -1;
	}

	
	if (sendcmd("TYPE I") != 200)
	{
		fprintf(stderr, "Failed to set TYPE I\n");
		if (xfer != NULL)
			xfer->error(EIO);
		return -1;
	}
	
	if (goto_state(Listening))
	{
		fprintf(stderr, "Failed to listen\n");
		if (xfer != NULL)
			xfer->error(EIO);
		return -1;
	}

	if (sendcmd("RETR "+file)/100 != 1)
	{
		fprintf(stderr, "Failed to send RETR\n");
		if (xfer != NULL)
			xfer->error(EIO);
		return -1;
	}

	struct sockaddr_in from;
	int fromlen = sizeof(from);
	int sfd;

	if ((sfd = ::accept(datasock, (struct sockaddr *)&from, &fromlen)) == -1)
	{
		perror("Failed to accept()");
		if (xfer != NULL)
			xfer->error(errno);
		goto_state(LoggedIn);
		return -1;
	}

	goto_state(Transfer);
	
	unsigned char buf[2048];
	int ret;

	cout << "Fetching " << file << " from " << hostname << ':' << path << '\n';
	while((ret = fullread(sfd, buf, sizeof(buf))) > 0)
	{
		putchar('#'); fflush(stdout);
		if (fullwrite(fd, buf, ret) == -1)
		{
			ret = -1;
			break;
		}

		if (xfer)
			xfer->notify(sfd);
	}
	if (ret == -1)
	{
		perror("file xfer failed");
		if (xfer != NULL)
			xfer->error(errno);
		::close(sfd);
		goto_state(LoggedIn);
		return -1;
	}

	cout << file << " done\n";
	
	getreply();
	::close(sfd);
	
	state = LoggedIn;	// Go directly, avoid unneeded ABOR

	return 0;
}

int
ftpconn::ftpcwd(String &cwd)
{
	String repl;
	int ret;
	
	if ((ret = sendcmd("PWD", &repl)) != 257)
		return ret;

	static const Regex bqrx("^[^\"]*\"");
	static const Regex aqrx("\"[^\"]*$");
	static const Regex rmqrx("\"\"");
	static const String nil;
	
	repl.gsub(bqrx, nil);
	repl.gsub(aqrx, nil);
	repl.gsub(rmqrx, nil);

	cwd = repl;
	return ret;
}

int
ftpconn::ftpsetcwd(const String &dir, int nostep)
{
	int ret;
	static const String cwd("CWD ");
	
	if (nostep)
		ret = sendcmd(cwd+dir);
	else
	{
		String *steps;
		int nelem = site_host->pathsteps(dir, &steps);
		int i;

		ret = 250;
		
		for(i = 0; i < nelem; i++)
		{
			ret = sendcmd(cwd+steps[i]);
			if (ret / 100 != 2)
				break;
		}
	
		delete [] steps;
	}
	return ret;
}

int
ftpconn::makedir(const String &dname)
{
	String dirname = dname;
	String path = dirname.before('/', -1);
	String dir = dirname.after('/', -1);
	if (dir == "")
		dir = dname;
	
	semlock lock(sem);

	cout << "ftpconn::makedir making dir on host " << hostname
		<< ", dirname=" << dirname << ", basedir=" << path
		<< ", subdir=" << dir << "\n";

	if (goto_state(LoggedIn))
	{
		fprintf(stderr, "Failed to log in\n");
		return -1;
	}

	ftp_return ftp_here(*this, site_root);
	if (path != "" && ftpsetcwd(path) != 250)
	{
		cerr << "Failed to change to \"" << path << "\"\n";
		return -1;
	}

	int ret = sendcmd("MKD "+dir);

	if (ret/100 != 2)
		return -1;
	return 0;
}
