// -*- C++ -*-

// Directories cached from an ftp site

// (C) Copyright 1994 Jeremy Fitzhardinge <jeremy@sw.oz.au>
// This code is distributed under the terms of the
// GNU General Public Licence.  See COPYING for more details.

#pragma implementation

#include <sys/types.h>
#include <sys/stat.h>
#include <unistd.h>

#include <errno.h>

#include <String.h>
#include <iostream.h>
#include <fstream.h>

#include "ftpdir.h"
#include "ftpconn.h"
#include "ftplink.h"
#include "ftpfile.h"
#include "pushd.h"
#include "ConfFile.h"

static const Regex ignore_pat("^\\.ftpfs_.*$");

ftpdir::ftpdir(ftpfs &fs, const String &elem, dir *up,
	       ftpdir *ftpup, const String &cfname)
       : dir(fs, elem, up), ftpup(ftpup)
{
	conn = new ftpconn(Path(cfname, &name), elem);
	construct();
}

ftpdir::ftpdir(ftpfs &fs, const String &name, dir *up,
	       ftpdir *ftpup, ftpconn *conn)
       : dir(fs, name, up), conn(conn), ftpup(ftpup)
{
	construct();
}

void
ftpdir::construct()
{
	int fd = parent->open();

	conf = NULL;
	
	if (fd != -1)
	{
		pushd here(fd);
		String dir=name.path();
		mkdir(dir, mode|0700);
		conf = new ConfFile(Path(".ftpfs_dir", &name));
	}
	else
		cout << "name=" << name.elem() <<
			" failed to open parent " <<
				parent->getname().path() << "\n";

	mode |= 0777;
	
	getparams();
	init_done = 0;
	conn->incref();
}

void
ftpdir::init()
{
	dir::init();

	int dfd = open();
	pushd here(dfd);

	ifstream dinfo(".ftpfs_dirinfo");

	if (!dinfo.bad())
	{
		mode_t mode;
		size_t size;
		time_t time;
		String name;
		String symlink;
		
		for(;;)
		{
			dinfo >> mode;
			dinfo >> size;
			dinfo >> time;
			dinfo >> name;
			if (S_ISLNK(mode))
				dinfo >> symlink;
			
			if (dinfo.eof() || dinfo.fail())
				break;
			
			if (lookup(name) != NULL)
				continue;

			Inode *ino = NULL;
			
			if (S_ISREG(mode))
			{
				ftpfile *f = new ftpfile(filesys, name,
							 this, conn);
				f->setstats(size, time);
				ino = f;
			}
			else if (S_ISDIR(mode))
			{
				ftpdir *d = new ftpdir(filesys, name, this,
						       this, conn);
				d->mtime = time;
				ino = d;
			}
			else if (S_ISLNK(mode))
				ino = new ftplink(filesys, symlink, this, name);
			
			if (ino == NULL)
				continue;
			link(name, ino);
		}
	}
}

void
ftpdir::cleanup()
{
	int fd = open();
	pushd here(fd);

	::unlink(".ftpfs_dirinfo");
	ofstream dinfo(".ftpfs_dirinfo", ios::out|ios::trunc, 0444);

	if (!dinfo.bad())
	{
		DirEntry *de=NULL;

		for(de = scan(de); de != NULL; scan(de))
		{
			String name = de->getname();

			if (name == "." || name == ".." ||
			    name.matches(ignore_pat))
				continue;

			up_inode ino;

			((SimpleInode *)de->geti())->iread(ino); // XXX

			dinfo << oct << ' ' << ino.mode << ' '
			      << dec << ' ' << ino.size << ' '
			      << ino.mtime << ' ' << name;
			if (S_ISLNK(ino.mode))
				dinfo << ' ' <<
					((ftplink *)de->geti())->getlink(); // XXX
			dinfo << '\n';
		}
	}
}

ftpdir::~ftpdir()
{
	setparams();
	conn->decref();
	if (conf)
		delete conf;
}

void
ftpdir::getparams()
{
	if (conf == NULL)
		return;
	
	if (!conf->getnum("last_update", last_upd))
		last_upd = 0;
	if (!conf->getnum("last_failure", last_failure))
		last_failure = 0;
}

void
ftpdir::setparams()
{
	if (conf == NULL)
		return;
	conf->setnum("last_update", last_upd);
	conf->setnum("last_failure", last_failure);
}

Inode *
ftpdir::newinode(dir *d, const struct stat *st, const String &name)
{
	Inode *ino = NULL;
	
	if (S_ISDIR(st->st_mode))
	{
		if (name == "." || name == "..")
			return NULL;
		ino = new ftpdir(filesys, name, this, this, conn);
	}
	else if (S_ISLNK(st->st_mode))
	{
		char buf[1024];
		int len = readlink(name, buf, sizeof(buf));
		if (len == -1)
			return NULL;
			
		ino = new ftplink(filesys, String(buf, len), this, name);
	}
	else if (S_ISREG(st->st_mode) && !name.matches(ignore_pat))
		ino = new ftpfile(filesys, name, this, conn);
	
	return ino;
}

int
ftpdir::updatedir(int force)
{
	time_t now = time(0);
	if (!init_done)
	{
		init_done = 1;
		init();
	}
	
	if (!force)
	{
		if (last_upd > (now - conn->get_upd_timeout()))
			return 0;
		if ((last_failure > last_upd) &&
		    last_failure > (now - conn->get_fail_timeout()))
			return 0;
	}	
	String p = ftppath();

	cout << "Updating directory at path \"" << p << "\"\n";
	
	ftp_filelist *fl, *flist;

	fl = flist = conn->getfilelist(p);

	if (fl == NULL)
	{
		last_failure = now;
		return ENOENT;
	}

	// Check to see if a file has changed type
	for(; fl != NULL; fl = fl->next)
	{
		DirEntry *de;
		
		if ((de = lookup(fl->name)) != NULL)
		{
			int bad = 0;
			struct up_inode i;

			((SimpleInode *)de->geti())->iread(i); // XXX
			
			switch(fl->type)
			{
			case FT_Directory:
				if (!S_ISDIR(i.mode))
					bad = 1;
				break;
			case FT_File:
				if (!S_ISREG(i.mode))
					bad = 1;
				else
					((ftpfile *)de->geti())->setstats(fl->size, fl->time);	// XXX
				break;
			case FT_Link:
				if (!S_ISLNK(i.mode))
					bad = 1;
				break;
			default:
				break;
			}
			
			if (!bad)
				continue;
			unlink(fl->name);
		}	

		Inode *ino = NULL;

		switch(fl->type)
		{
		case FT_Directory:
			ino = new ftpdir(filesys, fl->name, this, this, conn);
			break;
		case FT_File:
		{
			ftpfile *file = new ftpfile(filesys, fl->name, this, conn);
			file->setstats(fl->size, fl->time);
			ino = file;
			break;
		}
		case FT_Link:
			ino = new ftplink(filesys, fl->link, this, fl->name);
			break;
		default:
			break;
		}
		
		if (ino != NULL && link(fl->name, ino))
			delete ino;
	}

	DirEntry *de=NULL;
	pushd here(open());
	
	for(de = scan(de); de != NULL; scan(de))
	{
		String name = de->getname();
		
		if (name == "." || name == ".." ||
		    name.matches(ignore_pat))
			continue;
		
		int match = 0;
		for(fl = flist; fl != NULL; fl = fl->next)
			if (fl->name == name)
			{
				match = 1;
				break;
			}
		if (!match)
		{
			unlink(name);
			::unlink(name);
		}
	}

	for(; flist != NULL; flist = fl)
	{
		fl = flist->next;
		delete flist;
	}
	
	last_upd = now;
	return 0;
}

int
ftpdir::do_readdir(const up_preamble &pre, upp_repl &rep,
		   const upp_readdir_s &arg, upp_readdir_r &ret)
{
	updatedir();

	return dir::do_readdir(pre, rep, arg, ret);
}

int
ftpdir::do_multireaddir(const up_preamble &pre, upp_repl &rep,
			const upp_multireaddir_s &arg, upp_multireaddir_r &ret)
{
	updatedir();

	return dir::do_multireaddir(pre, rep, arg, ret);
}

int
ftpdir::do_lookup(const up_preamble &pre, upp_repl &rep,
		  const upp_lookup_s &arg, upp_lookup_r &ret)
{
	String name((char *)arg.name.elems, (int)arg.name.nelem);
	if (name == ".ftpfs_update")
	{
		updatedir(1);
		return ENOENT;
	}
	
	int r = dir::do_lookup(pre, rep, arg, ret);

	if (r == ENOENT)
	{
		updatedir();
		r = dir::do_lookup(pre, rep, arg, ret);
	}

	return r;
}

int 
ftpdir::do_create(const up_preamble &, upp_repl &,
		  const upp_create_s &arg, upp_create_r &ret)
{
	if (!S_ISDIR(arg.mode))
		return EPERM;
	
	String name((char *)arg.name.elems, (int)arg.name.nelem);
	String fullname;
	cat(ftppath(), "/", name, fullname);
	
	if (conn->makedir(fullname))
		return EPERM;

	Inode *ino = new ftpdir(filesys, name, this, this, conn);
	link(name, ino);
	
	return 0;
}

String
ftpdir::ftppath() const
{
	String p;

	if (ftpup)
		cat(ftpup->ftppath(), "/", name.elem(), p);
	else
		p = ".";
	return p;
}
