// -*- C++ -*-

// Non-blocking threaded IO
// This sets the fd for the operation into nonblocking mode.
// If the IO would have blocked, it's put onto the dispatch
// list and another thread is scheduled.  When the fd is ready
// for IO, this thread is rescheduled and the IO is completed.

// (C) Copyright 1994 Jeremy Fitzhardinge <jeremy@sw.oz.au>
// This code is distributed under the terms of the
// GNU General Public Licence.  See COPYING for more details.

#include <sys/types.h>
#include <sys/time.h>
#include <fcntl.h>
#include <errno.h>
#include <unistd.h>
#include <assert.h>

#include <ThreadComm.h>
#include <LWP.h>

CommBase *selector = NULL;
int do_nonblock = 0;

class reio : public DispatchFD
{
	int dispatch(int, int);

	Semaphore sem;

	CommBase *comm;
	
public:
	reio();

	void wait(int fd, int what);
	
};

reio::reio()
     : sem(0), comm(selector)
{
	assert(selector != NULL);
}

void
reio::wait(int fd, int what)
{
	comm->addDispatch(fd, this, what);
	sem.wait();
}

int
reio::dispatch(int, int)
{
	sem.signal();
	return 0;
}

extern "C" 
ssize_t read(int fd, void *buf, size_t sz)
{
	int flags;
	ssize_t ret;
	
	if ((flags = fcntl(fd, F_GETFL, 0)) == -1)
		return -1;

 again:
	if (do_nonblock && fcntl(fd, F_SETFL, flags|O_NONBLOCK) == -1)
		return -1;

	if ((ret = __read(fd, buf, sz)) == -1)
	{
		int err = errno;
		
		fcntl(fd, F_SETFL, flags);
	
		if (err == EINTR)
			goto again;
		
		if (err == EAGAIN)
		{
			reio *rio = new reio();

			rio->wait(fd, DISP_R);
			goto again;
		}
		errno = err;
		return ret;
	}
	fcntl(fd, F_SETFL, flags);

	return ret;
}

extern "C"
ssize_t write(int fd, const void *buf, size_t sz)
{
	int flags;
	ssize_t ret;
	
	if ((flags = fcntl(fd, F_GETFL, 0)) == -1)
		return -1;

 again:
	if (do_nonblock && fcntl(fd, F_SETFL, flags|O_NONBLOCK) == -1)
		return -1;

	if ((ret = __write(fd, buf, sz)) == -1)
	{
		int err = errno;
		
		fcntl(fd, F_SETFL, flags);
	
		if (err == EINTR)
			goto again;
		
		if (err == EAGAIN)
		{
			reio *rio = new reio();
			rio->wait(fd, DISP_W);
			goto again;
		}
		errno = err;
		return ret;
	}
	fcntl(fd, F_SETFL, flags);

	return ret;
}

#if 0
extern "C"
int select(int width, fd_set *read_fd, fd_set *write_fd, fd_set *except_fd,
	   struct timeval *timeout)
{
	int i;
	
	for(i = 0; i < width; i++)
	{
		int what = 0;
		if (FD_ISSET(i, read_fd))
			what |= DISP_R;
		if (FD_ISSET(i, write_fd))
			what |= DISP_W;
		if (FD_ISSET(i, execpt_fd))
			what |= DISP_E;
		if (what != 0)
		{
		}
	}
}
#endif

#include "netsys.h"

extern "C"
int send(int fd, const void *buf, int len, unsigned sndfl)
{
	int flags;
	int ret;
	
	if ((flags = fcntl(fd, F_GETFL, 0)) == -1)
		return -1;

 again:
	if (do_nonblock && fcntl(fd, F_SETFL, flags|O_NONBLOCK) == -1)
		return -1;

	if ((ret = __send(fd, buf, len, sndfl)) == -1)
	{
		int err = errno;
		fcntl(fd, F_SETFL, flags);
	
		if (err == EAGAIN)
		{
			reio *rio = new reio();

			rio->wait(fd, DISP_W);
			goto again;
		}
		errno = err;
		return ret;
	}
	fcntl(fd, F_SETFL, flags);

	return ret;
}

extern "C"
int sendto(int fd, const void *buf, int len, unsigned sndfl,
	   const struct sockaddr *to, int tolen)
{
	int flags;
	int ret;

	if ((flags = fcntl(fd, F_GETFL, 0)) == -1)
		return -1;

 again:
	if (do_nonblock && fcntl(fd, F_SETFL, flags|O_NONBLOCK) == -1)
		return -1;

	if ((ret = __sendto(fd, buf, len, sndfl, to, tolen)) == -1)
	{
		int err = errno;
		
		fcntl(fd, F_SETFL, flags);
	
		if (err == EAGAIN)
		{
			reio *rio = new reio();

			rio->wait(fd, DISP_W);
			goto again;
		}
		errno = err;
		return ret;
	}
	fcntl(fd, F_SETFL, flags);

	return ret;
}

extern "C" 
int recv(int fd, void *buf, int len, unsigned rcvfl)
{
	int flags;
	ssize_t ret;
	
	if ((flags = fcntl(fd, F_GETFL, 0)) == -1)
		return -1;

 again:
	if (do_nonblock && fcntl(fd, F_SETFL, flags|O_NONBLOCK) == -1)
		return -1;

	if ((ret = __recv(fd, buf, len, rcvfl)) == -1)
	{
		int err = errno;
		
		fcntl(fd, F_SETFL, flags);
	
		if (err == EAGAIN)
		{
			reio *rio = new reio();

			rio->wait(fd, DISP_R);
			goto again;
		}
		errno = err;
		return ret;
	}
	fcntl(fd, F_SETFL, flags);
	return ret;
}

extern "C" 
int recvfrom(int fd, void *buf, int len, unsigned rcvfl,
	     struct sockaddr *from, int *fromlen)
{
	int flags;
	ssize_t ret;
	
	if ((flags = fcntl(fd, F_GETFL, 0)) == -1)
		return -1;

 again:
	if (do_nonblock && fcntl(fd, F_SETFL, flags|O_NONBLOCK) == -1)
		return -1;

	if ((ret = __recvfrom(fd, buf, len, rcvfl, from, fromlen)) == -1)
	{
		int err = errno;
		
		fcntl(fd, F_SETFL, flags);
	
		if (err == EAGAIN)
		{
			reio *rio = new reio();

			rio->wait(fd, DISP_R);
			goto again;
		}
		errno = err;
		return ret;
	}
	fcntl(fd, F_SETFL, flags);

	return ret;
}

extern "C" 
int connect(int fd, struct sockaddr *addr, int addrlen)
{
	int flags;
	ssize_t ret;
	
	if ((flags = fcntl(fd, F_GETFL, 0)) == -1)
		return -1;

 again:
	if (do_nonblock && fcntl(fd, F_SETFL, flags|O_NONBLOCK) == -1)
		return -1;

	if ((ret = __connect(fd, addr, addrlen)) == -1)
	{
		int err = errno;
		
		fcntl(fd, F_SETFL, flags);

#if 0
		// Hack around Linux kernel bug
		// If we get EALREADY back, it means we have connected,
		// are in the process of disconnecting again.  There is
		// received data ready for us to read
		if (err == EALREADY)
			return 0;
#endif
		if (err == EALREADY || err == EINPROGRESS)
		{
			reio *rio = new reio();

			rio->wait(fd, DISP_R);
			goto again;
		}
		errno = err;
		return ret;
	}
	fcntl(fd, F_SETFL, flags);
	
	return ret;
}

extern "C" 
int accept(int fd, struct sockaddr *addr, int *addrlen)
{
	int flags;
	ssize_t ret;
	
	if ((flags = fcntl(fd, F_GETFL, 0)) == -1)
		return -1;

 again:
	if (do_nonblock && fcntl(fd, F_SETFL, flags|O_NONBLOCK) == -1)
		return -1;

	if ((ret = __accept(fd, addr, addrlen)) == -1)
	{
		int err = errno;
		
		fcntl(fd, F_SETFL, flags);
	
		if (err == EAGAIN)
		{
			reio *rio = new reio();

			rio->wait(fd, DISP_R);
			goto again;
		}
		errno = err;
		return ret;
	}

	fcntl(fd, F_SETFL, flags);
	return ret;
}
