// -*- C++ -*-

// (C) Copyright 1994 Jeremy Fitzhardinge <jeremy@sw.oz.au>
// This code is distributed under the terms of the
// GNU General Public Licence.  See COPYING for more details.

#pragma implementation

#include "unix_host.h"
#include "serv_port.h"
#include "ftpconn.h"
#include "getdate.h"
#include <io.h>

#include <unistd.h>
#include <time.h>
#include <sys/types.h>
#include <sys/socket.h>
#include <String.h>
#include <stdio.h>
#include <netinet/in.h>
#include <stdlib.h>

static const serv_port daytime_port("daytime", "tcp");

int
unix_host::pathsteps(const String &path, String *ret[])
{
	static const Regex deldot("\\(^\\./\\)\\|\\(/\\.\\/\\)\\|\\(/\\.$\\)\\|\\(//\\)\\|\\(^\\.$\\)");
	static const Regex deldotdot("\\(^|/\\)[^/]+/\\.\\./");
	
	*ret = new String[1];
	(*ret)[0] = path;
	(*ret)[0].gsub(deldot, "");
	(*ret)[0].gsub(deldotdot, "/");

	if ((*ret)[0] == "")
		return 0;
	
	return 1;
}

// Get local time at remote machine with the "time" service
time_t
unix_host::localtime(struct sockaddr_in addr)
{
	int fd;
	unsigned char rawtime[512];
	int size;
	
	addr.sin_port = daytime_port.port;
	if ((fd = socket(AF_INET, SOCK_STREAM, 0)) == -1)
	{
		perror("Time service socket");
		return -1;
	}

	if (connect(fd, (struct sockaddr *)&addr, sizeof(addr)) == -1)
	{
		perror("Time service connect");
		close(fd);
		return -1;
	}

	if ((size = fullread(fd, rawtime, sizeof(rawtime)-1)) == -1)
	{
		perror("Time service read");
		close(fd);
		return -1;
	}
	close(fd);

	rawtime[size] = 0;

	time_t rtime = get_date(rawtime, NULL);

	struct tm *tm = ::gmtime(&rtime);
	cout << "Remote time "<<asctime(tm);
	return rtime;
}

// Parse a date from "ls" - its about the only thing getdate won't handle
static time_t
parsedate(const String &date, time_t timezone)
{
	String parts[3];

	if (split(date, parts, 3, RXwhite) != 3)
		return -1;
	
	static const String months[13] = { "xxx",
		"Jan", "Feb", "Mar",
		"Apr", "May", "Jun",
		"Jul", "Aug", "Sep",
		"Oct", "Nov", "Dec"
		};

	static const Regex rx_time("[012]?[0-9]:[0-5][0-9]");
	static const Regex rx_year("[0-9][0-9][0-9][0-9]");
	static const Regex rx_day("[0-9]?[0-9]");

	int month = -1;
	int day = -1;
	int year = -1;
	int hours = -1;
	int mins = -1;
	
	for(int i = 1; i < 13; i++)
		if (parts[0] == months[i])
		{
			month = i;
			break;
		}
	if (month == -1)
		return 0;
	
	if (!parts[1].matches(rx_day))
		return 0;
	day = atoi(parts[1]);
	
	if (parts[2].matches(rx_year))
		year = atoi(parts[2]);
	else
	{
		String hm[2];
		if (!parts[2].matches(rx_time))
			return 0;
		if (split(parts[2], hm, 2, String(":")) != 2)
			return 0;
		hours = atoi(hm[0]);
		mins = atoi(hm[1]);
	}

	time_t now = time(0);

	struct tm *tm = gmtime(&now);
	
	return Convert(month, day,
		       year == -1 ? tm->tm_year : year,
		       hours == -1 ? 0 : hours,
		       mins == -1 ? 0 : mins, 0,
		       MER24, DSToff, timezone);
}

struct ftp_filelist *
unix_host::parse_filelist(const String &ro_line, time_t timezone)
{
	String line = ro_line;
	
	// Match a line of "ls" output
	static const Regex ls_line(
		"^\\([-dl][r-][w-][xsS-][r-][w-][xs-][r-][w-][x-]\\) +"	// perms
		"[0-9]+ +"						// links
		"[0-9a-zA-Z_-]+ +"					// uid
		"[0-9a-zA-Z_-]+ +"					// gid
		"\\([0-9]+\\) +"					// size
		"\\(\\(Jan\\|Feb\\|Mar\\|Apr\\|May\\|Jun\\|Jul\\|Aug\\|Sep\\|Oct\\|Nov\\|Dec\\) \\( \\|[0-9]\\)[0-9] \\([012][0-9]:[0-5][0-9]\\| [0-9][0-9][0-9][0-9]\\)\\) "	// date
		"\\(.+\\)$"						// name
	);
	static const Regex bsd_ls_line(
		"^\\([-dl][r-][w-][xsS-][r-][w-][xs-][r-][w-][x-]\\) +"	// perms
		"[0-9]+ +"						// links
		"[0-9a-zA-Z_-]+ +"					// uid
		"\\([0-9]+\\) +"					// size
		"\\(\\(Jan\\|Feb\\|Mar\\|Apr\\|May\\|Jun\\|Jul\\|Aug\\|Sep\\|Oct\\|Nov\\|Dec\\) \\( \\|[0-9]\\)[0-9] \\([012][0-9]:[0-5][0-9]\\| [0-9][0-9][0-9][0-9]\\)\\) "	// date
		"\\(.+\\)$"						// name
	);

	const Regex *ls_rx = &ls_line;
	
	if (!line.matches(ls_line))
	{
		if (!line.matches(bsd_ls_line))
		{
			cout << "Failed to match ls line " << line << '\n';
			return NULL;
		}
		ls_rx = &bsd_ls_line;
	}

	int start, len;

	ftp_filelist *nf = new ftp_filelist;
	ls_rx->match_info(start, len, 7);
	nf->name = String(line.at(start,len));
	
	switch(line[0])
	{
	case 'd':
		nf->type = FT_Directory;
		break;
	case '-':
		nf->type = FT_File;
		break;
	case 'l':
		{
			String name = nf->name;
			static const Regex sym_rx("^\\([^ ]+\\) -> \\(.*\\)$");
			if (!name.matches(sym_rx))
			{
				nf->type = FT_Unknown;
				break;
			}

			nf->type = FT_Link;
			sym_rx.match_info(start, len, 1);
			nf->name = String(name.at(start, len));
			sym_rx.match_info(start, len, 2);
			nf->link = String(name.at(start, len));
		}
		break;
	default:
		nf->type = FT_Unknown;
	}

	ls_rx->match_info(start, len, 2);
	String sz_str(line.at(start, len));
	nf->size = strtol(sz_str, NULL, 10);

	ls_rx->match_info(start, len, 3);
	String date_str(line.at(start, len));
	nf->time = parsedate(date_str, timezone);
	nf->next = NULL;

	struct tm *tm = gmtime(&nf->time);
	cout << "Name " << nf->name << " size " << nf->size <<
		" date " << asctime(tm);

	return nf;
}

const String &
unix_host::hosttype() const
{
	static const String ux("unix");

	return ux;
}
