// -*- C++ -*-
// Find home
// A simple user process filesystem.  This one sets up a dir full
// of symlinks pointing to everyone in /etc/passwd's home dirs.
//
// This program is distributed under the terms of the
// Free Software Foundation General Public Licence.
// Copyright Jeremy Fitzhardinge 1993

#include <Comm.h>
#include <Filesystem.h>
#include <DirInode.h>
#include <SimpleInode.h>
#include "homer.h"
#include "password.h"
#include <sys/stat.h>
#include <getopt.h>
#include <signal.h>
#include "dbg.h"

// Choose an inode number for the filesystem.  We use the line
// number in /etc/passwd as the inode, starting from zero, so have
// something high to avoid it.  This can be bigger, but it gets
// unwieldy.
const Handle DIRINO = 65534;

// This is all the operations the filesystem as a whole will accept
// from the kernel.  This is done this way so that if the kernel
// asks about completely new operations we can still say "no".
int
homer::Enquire(up_ops op)
{
	LOG("enq");
	switch(op)
	{
	case up_mount:
	case up_iread:
	case up_readdir:
	case up_multireaddir:
	case up_lookup:
	case up_readlink:
//	case up_unlink:
		return 0;

	default:
		// Rather than just failing, ask our base class
		return Filesystem::Enquire(op);
	}
}

// Just another constructor for the generic directory inode
// After construction everything is handled by the base class
class dirino:public DirInode
{
public:
	dirino::dirino(Filesystem &, Handle, password *);
};

// A symlink to the home dir
// This has most of the work in it, but there isn't very much
// to do.
class nameino:public SimpleInode
{
	char *dir;
	int dirlen;
	
protected:
	int do_readlink(const up_preamble &, upp_repl &,
			const upp_readlink_s &, upp_readlink_r &);
public:
	nameino(Filesystem &, Handle, uid_t, gid_t, char *);
	virtual ~nameino();
};

nameino::nameino(Filesystem &fs, Handle hand,
		 uid_t u, gid_t g, char *d)
	: SimpleInode(fs, hand)
{
	uid = u;
	gid = g;
	dir = strdup(d);
	size = dirlen = strlen(d);
	mode = S_IFLNK | 0777;
	atime = ctime = mtime = time(0);
}

nameino::~nameino()
{
	free(dir);
}

// Make a dir entry for each line in the passwd file, and init our
// own inode values.
dirino::dirino(Filesystem &fs, Handle h, password *pw)
       :DirInode(fs, h, this)
{
	const pwent *pwe;

	pwe = pw->getlist();

	for (; pwe != NULL; pwe = pwe->next)
	{
		Inode *ino = findino(pwe->ino);
		link(pwe->name, ino);
	}

	uid = getuid();
	gid = getgid();
	mode = S_IFDIR | 0555;
	mtime = ctime = atime = time(0);
}

// Constructor for the filesystem.
// This creates all the filesystem's inodes from the password file,
// then creates a single directory to give them names.
homer::homer(const char *mpoint, password *pw) : Filesystem(mpoint)
{
	const pwent *pwe = pw->getlist();
	
	for(; pwe != NULL; pwe = pwe->next)
		new nameino(*this, pwe->ino, pwe->uid, pwe->gid, pwe->dir);

	new dirino(*this, DIRINO, pw);
}

int
homer::do_mount(const up_preamble &pre, upp_repl &repl,
		upp_mount_r &ret)
{
	ret.root.handle = DIRINO;
	return 0;
}

int
nameino::do_readlink(const up_preamble &pre, upp_repl &repl,
		     const upp_readlink_s &arg, upp_readlink_r &ret)
{
	LOG("readlink");

	atime = time(0);
	ret.name.elems = dir;
	ret.name.nelem = dirlen;

	return 0;
}

const char *mpoint = NULL;

int
main(int argc, char *argv[])
{
	int infd = 0;
	int outfd = 1;
	int c;
	
	LOG("Starting");

#ifdef DEBUG
	printf("pid=%d\n", getpid());
	getchar();
#endif

	// Parse the command line options from muserfs
	while((c = getopt(argc, argv, "i:o:m:")) != EOF)
		switch(c)
		{
		case 'i':
			infd = atoi(optarg);
			break;

		case 'o':
			outfd = atoi(optarg);
			break;

		case 'm':
			mpoint = optarg;
			break;
			
		default:
			fprintf(stderr, "%s: bad option\n", argv[0]);
			exit(1);
		}

	signal(SIGCHLD, SIG_IGN);

	LOG("GOING to run");

	// Create password file class
	password pwfile;
	
	// Create filesystem
	homer homefs(mpoint, &pwfile);

	// Create communications with kernel
	Comm comm(homefs, outfd, infd);

	// Run -- doesn't return until EOF or error
	if (comm.Run() == -1)
		fprintf(stderr, "Something went wrong!\n");

	LOG("Stop");

	return 0;
}
