// -*- C++ -*-

// Cope with directories in an intfs filesystem

#pragma implementation

#include "intfsDirIno.h"
#include "intfsLinkIno.h"
#include "intfsIno.h"
#include "intfs.h"

#include "pushd.h"

#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <unistd.h>
#include "readdir.h"

#include "dbg.h"

ifsDirIno::ifsDirIno(intfs &fs, Handle hand,
		     ifsDirIno *parent, String dirname)
	:DirInode(fs, hand, parent == NULL ? this : parent, 30),
	 name(dirname), dotdot(parent == NULL ? this : parent),
	 filesys(fs)
{
	fd = -1;
	mtime = atime = ctime = 0;
	updating = 0;
	
	open();
	struct stat st;

	if (fstat(fd, &st) != -1)
	{
		mode = st.st_mode;
		uid = st.st_uid;
		gid = st.st_gid;
	}
	close();
}

ifsDirIno::~ifsDirIno()
{
	close();
}

String ifsDirIno::toppath()
{
	ifsDirIno *p;
	String ret = ".";

	for(p = this; p != NULL && p->dotdot != p; p = p->dotdot)
	{
		String n;
		cat("../", ret, n);
		ret = n;
	}

	return ret;
}

String ifsDirIno::path()
{
	String p = name;

	if (dotdot && dotdot != this)
		cat(dotdot->path(), "/", p, p);

	return p;
}

void ifsDirIno::update()
{
	struct dirent de;
	int ret;

	if (updating)
		return;
	updating = 1;
	
	if ((ret = open()))
	{
		DB(fprintf(stderr, "ifsDirIno::update open failed: %s\n", strerror(ret)));
		updating = 0;
		return;
	}
	
	struct stat st;

	if (fstat(fd, &st) == -1)
	{
		DB(perror("ifsDirIno::update stat failed"));
		close();
		updating = 0;
		return;
	}
	
	atime = st.st_atime;
	ctime = st.st_ctime;
	uid = st.st_uid;
	gid = st.st_gid;
	mode = st.st_mode;
	
	if (st.st_mtime == mtime)
	{
		close();
		updating = 0;
		return;
	}
	
	mtime = st.st_mtime;

	pushd here(fd);
	
	while((ret = readdir(fd, &de, 1)) > 0)
	{
		if (lstat(de.d_name, &st) == -1)
			continue;

		String entname(de.d_name, de.d_reclen);
		
		DirEntry *dent = lookup(entname);

		if (dent == NULL)
		{
			Inode *ino;
			
			if (S_ISLNK(st.st_mode))
			{
				char buf[256];
				int len;
				
				DB(printf("%s is a link\n",
					  (const char *)entname));
				if ((len = readlink((const char *)entname,
						    buf, sizeof(buf))) == -1)
					continue;
				ino = new ifsLinkIno(filesys,
						     filesys.genhand(),
						     this, entname,
						     String(buf, len));
			}
			else if (S_ISDIR(st.st_mode))
			{
				DB(printf("%s is a directory\n",
					  (const char *)entname));
				ino = new ifsDirIno(filesys, filesys.genhand(),
						    this, entname);
			}
			else
			{
				String str;
//				DB(printf("%s is a file\n",
//					  (const char *)entname));
				cat(path(), "/", entname, str);
				ino = new ifsIno(filesys,
						 filesys.genhand(),
						 this, entname, str);
			}
			link(entname, ino);
		}
		else
			((ifsDirIno *)dent->geti())->update();
	}
	if (ret != 0)
		DB(perror("ifsDirIno::update, readdir failed"));
	updating = 0;
	close();
}

int
ifsDirIno::open()
{
	String p = path();

	fd = ::open(p, O_RDONLY);
	int err = errno;
	
	return  (fd == -1) ? err : 0;
}

void
ifsDirIno::close()
{
	::close(fd);
	fd = -1;
}

int
ifsDirIno::dupfd()
{
	int fl = 0;
	
	if (fd == -1)
	{
		fl = 1;
		open();
	}

	int ret = dup(fd);

	if (fl)
		close();
	return ret;
}

int
ifsDirIno::do_unlink(const up_preamble &, upp_repl &,
		     const upp_unlink_s &arg)
{
	LOG("ifsDirIno::do_unlink");
	int ret;
	
	if (ret = open())
		return ret;

	pushd here(fd);
	close();
	
	String file((const char *)arg.name.elems, (int)arg.name.nelem);
	DirEntry *de = lookup(file);

	if (S_ISDIR(((ifsDirIno *)de->geti())->mode))
	{
		if (::rmdir((const char *)file) == -1)
			ret = errno;
	}
	else
	{
		if (::unlink((const char *)file) == -1)
			ret = errno;
	}

	if (ret == 0 || ret == ENOENT)
		ret = unlink(String((const char *)arg.name.elems,
				    (int)arg.name.nelem));
	return ret;
}

int ifsDirIno::do_symlink(const up_preamble &, upp_repl &, const upp_symlink_s &arg)
{
	open();
	pushd here(fd);
	String name((const char *)arg.name.elems, (int)arg.name.nelem);
	String link((const char *)arg.symname.elems, (int)arg.symname.nelem);
	
	if (::symlink((const char *)link, (const char *)name) == -1)
		return errno;
	return 0;
}

int ifsDirIno::do_create(const up_preamble &, upp_repl &,
			 const upp_create_s &arg, upp_create_r &ret)
{
	open();
	pushd here(fd);
	close();
	String name((const char *)arg.name.elems, (int)arg.name.nelem);
	Inode *ino;
	String str;
	Handle hand = filesys.genhand();
	
	cat(path(), "/", name, str);

	if (S_ISREG(arg.mode))
	{
		int fd = ::open((const char *)name,
				O_TRUNC|O_CREAT|O_RDONLY, arg.mode);
		if (fd == -1)
			return errno;
		::close(fd);
		ino = new ifsIno(filesys, hand, this, name, str);
	}
	else if (S_ISDIR(arg.mode))
	{
		if (::mkdir((const char *)name, arg.mode) == -1)
			return errno;
		ino = new ifsDirIno(filesys, hand, this, name);
	}
	else
	{
		if (::mknod((const char *)name, arg.mode, arg.rdev) == -1)
			return errno;
		ino = new ifsIno(filesys, hand, this, name, str);
	}
	
	link(name, ino);
	
	ret.file.handle = ino->gethandle();
	return 0;
}

int
ifsDirIno::do_rename(const up_preamble &pre, upp_repl &rep,
		     const upp_rename_s &arg)
{
	open();
	pushd here(fd);
	close();

	String oname((const char *)arg.oname.elems, (int)arg.oname.nelem);
	String nname((const char *)arg.nname.elems, (int)arg.nname.nelem);
	ifsDirIno *ndir = (ifsDirIno *)findino(arg.ndir.handle);
	String npath;

	cat(ndir->path(), "/", nname, npath);

	DB(printf("renaming from %s to %s\n",
		  (const char *)oname, (const char *)npath));
	
	if (::rename((const char *)oname, (const char *)npath) == -1)
		return errno;

	return DirInode::do_rename(pre, rep, arg);
}
	
