// -*- C++ -*-

// An inode with a command behind it
// These are transient, and only exist long enough to be opened.
// They go away once they have been put, never to be seen again.
// The contents are only generated if they are actually read.

#pragma implementation

#include "intfsFifoIno.h"
#include "intfsFifoDir.h"
#include "intfsDirIno.h"
#include "intfs.h"

#include "dbg.h"

#include <DeferComm.h>

#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <time.h>
#include <errno.h>
#include <stdio.h>
#include <unistd.h>
#include <signal.h>

ifsFifoIno::ifsFifoIno(intfs &fs, String command, ifsDirIno *dir)
	: SimpleInode(fs, fs.genhand()),
	  command(command), dir(dir), filesys(fs)
{
	atime = ctime = mtime = time(0);
	mode = S_IFIFO | 0444;
	pid = -1;
	readfd = -1;
		
	name = filesys.getfifod()->create(this);
	named = 1;

	DB(printf("ifsFifoIno::ifsFifoIno, linked into fifod name=%s, command=%s\n",
		  (const char *)name, (const char *)command));
}

ifsFifoIno::~ifsFifoIno()
{
	unname();
	::close(readfd);
	readfd = -1;
//	if (pid > 0)
//		::kill(pid, SIGTERM);
//	pid = -1;
}

void
ifsFifoIno::unname()
{
	if (named)
		filesys.getfifod()->unlink(name);
	named = 0;
}
		
int
ifsFifoIno::do_open(const up_preamble &, upp_repl &,
		    const upp_open_s &arg, upp_open_r &ret)
{
	LOG("ifsFifoIno::do_open");
	unname();

	uid = arg.cred.uid;
	euid = arg.cred.euid;
	gid = arg.cred.gid;
	egid = arg.cred.egid;
	
	return 0;
}

int
ifsFifoIno::do_iput(const up_preamble &, upp_repl &,
		    const upp_iput_s &)
{
	LOG("ifsFifoIno::do_iput");

	delete this;
	return 0;
}

int
ifsFifoIno::start()
{
	int pfd[2];
	
	if (pipe(pfd) == -1)
		return errno;

	signal(SIGCHLD, SIG_IGN);
	
	switch(pid = fork())
	{
	case -1:
		return errno;
	case 0:
		close(0);
		close(1);
		close(2);
		close(pfd[0]);
		filesys.comm()->CloseFDs(1);
		
		open("/dev/null", O_RDWR);
		dup(pfd[1]);
		dup(pfd[1]);
		close(pfd[1]);
		
		int dfd = dir->dupfd();
		fchdir(dfd);
		close(dfd);

		setreuid(uid, euid);
		setregid(gid, egid);

		signal(SIGTERM, SIG_DFL);
		
		execlp("/bin/sh", "/bin/sh", "-c", (const char *)command, 0);
		fprintf(stderr, "exec of /bin/sh -c %s failed: %s\n",
			(const char *)command, strerror(errno));
		exit(1);
	}

	close(pfd[1]);
	readfd = pfd[0];
	
	return 0;
}

int
ifsFifoIno::do_read(const up_preamble &, upp_repl &,
		    const upp_read_s &arg, upp_read_r &data)
{
	int ret;

	LOG("ifsFifoIno::do_read");
	
	if (pid == -1 && (ret = start()) != 0)
		return ret;

	int dpid = filesys.comm()->DeferRepl();

	if (dpid != -1 && dpid != 0)
		return -1;
	
	char buf[1024];
	off_t sz = arg.size > sizeof(buf) ? sizeof(buf) : arg.size;
		
	while((ret = read(readfd, buf, sz)) == -1)
		if (errno != EINTR)
			return errno;

	if (ret > 0)
	{
		data.data.alloc(ret);
		memcpy(data.data.elems, buf, ret);
	}
	data.data.nelem = ret;

	return 0;
}
