/* Archive VFS layer.  */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include "arcvfs.h"

ARC_DIRENT *
create_entry (const char *fname, const char *name, ARC_DIRENT *dir)
{
  ARC_DIRENT *ent;

  ent = malloc (sizeof (ARC_DIRENT) + strlen (fname));
  if (ent)
    {
      strcpy (ent->fullname, fname);
      ent->name = ent->fullname + (name - fname);
      ent->pdir = dir;
      ent->next = 0;
   }
  return ent;
}

ARC_DIRENT *
create_dirent (char *fname, char *name, ARC_DIRENT **p_ent, ARC_DIRENT *dir)
{
  char *s;

  if (!p_ent)
    return 0;

  while ((s = strchr (name, '/')) != 0)
    {
      *s = '\0';
      while (*p_ent && strcmp (name, (*p_ent)->name))
	p_ent = &(*p_ent)->next;
      if (!*p_ent)
	{
	  *p_ent = create_entry (fname, name, dir);
	  if (!*p_ent)
	    return 0;
	  (*p_ent)->size = -1;
	  (*p_ent)->data.dir.first = 0;
	}
      /* Check for directory/file clash.  */
      if ((*p_ent)->size != -1)
	return 0;
      dir = *p_ent;
      p_ent = &(*p_ent)->data.dir.first;
      *s++ = '/';
      name = s;
    }

  while (*p_ent)
    p_ent = &(*p_ent)->next;

  *p_ent = create_entry (fname, name, dir);
  return *p_ent;
}

char *
skip_field (char *s)
{
  while (*s == ' ')
    ++s;
  while (*s != ' ')
    ++s;
  return s;
}

static ARC_DIRENT *
root_ent (ARC_DIRENT *ent)
{
  while (ent->pdir != ent)
    ent = ent->pdir;
  return ent;
}

/* The generic read/write code.  */

int
a_open (ARC_DIRENT *ent)
{
  FILE *fp;
  char buf[4096];
  ARC_DIRENT *root = root_ent (ent);

  sprintf (buf, root->name, root->fullname, ent->fullname);
  fp = popen (buf, "r");
  if (!fp)
    return EIO;
  ent->data.file.buf = malloc (ent->size);
  if (ent->data.file.buf)
    {
      fread (ent->data.file.buf, ent->size, 1, fp);
      fclose (fp);
      ++ent->data.file.refc;
      return 0;
    }
  return EFBIG;
}

int
a_close (ARC_DIRENT *ent)
{
  if (!ent->data.file.refc)
    return EBADF;
  if (!--ent->data.file.refc)
    free (ent->data.file.buf);
  return 0;
}

int
a_read (ARC_DIRENT *ent, long off, long size,
	unsigned long *nelem, unsigned char **elems)
{
  char *s;
  long n;

  if (!ent->data.file.refc)
    return EINVAL;

  s = ent->data.file.buf;
  n = ent->size;

  if (off > n)
    off = n;
  if (size > n - off)
    size = n - off;
  *elems = s + off;
  *nelem = size;

  return 0;
}

/* The VFS code itself.  */

struct vfsfn {
  ARC_DIRENT *(*fn) (const char *);
} ents[] = {
#ifdef ARCFMT_ZIP
  { arc_read_zip },
#endif
#ifdef ARCFMT_HPACK
  { arc_read_hpack },
#endif
#ifdef ARCFMT_ZOO
  { arc_read_zoo },
#endif
#ifdef ARCFMT_TGZ
  { arc_read_tgz },
#endif
  { 0 }
};

ARC_DIRENT *
arc_read (const char *zipfile)
{
  int i;

  i = 0;
  while (ents[i].fn)
    {
      ARC_DIRENT *root = ents[i].fn (zipfile);

      if (root)
	return root;
      ++i;
    }

  fprintf (stderr, "%s: not a recognized archive type!\n", zipfile);
  return 0;
}
