// -*- C++ -*-
// Communications class
//
// This class copes with all the communication to and from the
// kernel for a userfs filesystem.
// The implementation of the base class is purely synchronous.
//
// This program is distributed under the terms of the
// Free Software Foundation General Public Licence.
// Copyright Jeremy Fitzhardinge <jeremy@sw.oz.au> 1993

#include <stdio.h>
#include <unistd.h>
#include <assert.h>
#include <string.h>

#pragma implementation

#include "CommBase.h"
#include "Filesystem.h"

CommBase::CommBase(Filesystem &fs, unsigned int to, unsigned int from)
     :tokern(to),filesys(fs)
{
	req = NULL;
	for(int i=0; i < NDISP_PRI; i++)
		dispqs[i].h = dispqs[i].t = NULL;
	dispatches = 0;
	fs.setcomm(this);
}

CommBase::~CommBase()
{
	close(tokern);
	tokern = (unsigned)-1;
	CloseFDs(1);
}

struct disp_fd
{
	disp_fd *next, *prev;
	int fd;
	DispatchFD *dfd;
	int what;
	int pri;
	
	~disp_fd();
};

disp_fd::~disp_fd()
{
	delete dfd;
}

DispatchFD::~DispatchFD()
{
}

// Queue things on the end of the list to get round-robin type scheduling
// In a multithreaded filesystem this is important to allow multiple access
// to the filesystem
disp_fd *
CommBase::addDispatch(int fd, DispatchFD *dfd, int what, int pri)
{
	disp_fd *fdl = new disp_fd;

	if (pri >= NDISP_PRI)
		pri = NDISP_PRI-1;
	
	fdl->prev = dispqs[pri].t;
	fdl->next = NULL;
	if (fdl->prev != NULL)
		fdl->prev->next = fdl;

	dispqs[pri].t = fdl;
	if (dispqs[pri].h == NULL)
		dispqs[pri].h = fdl;

	fdl->fd = fd;
	fdl->dfd = dfd;
	fdl->what = what;
	fdl->pri = pri;

	dispatches++;
	return fdl;
}

void
CommBase::delDispatch(disp_fd *fdl)
{
	if (fdl->dfd == req)
		req = NULL;

	if (fdl->next != NULL)
		fdl->next->prev = fdl->prev;
	else
		dispqs[fdl->pri].t = fdl->prev;
	
	if (fdl->prev != NULL)
		fdl->prev->next = fdl->next;
	else
		dispqs[fdl->pri].h = fdl->next;

	fdl->next = fdl->prev = NULL;

	dispatches--;
	delete fdl;
}

// Wait on input filedescriptors for something to happen.
// Things which want to know about file descriptors (for read only)
// can register themselves.  When select() returns, each
// matching dispatch routine is called so it can do what it likes.
// If a dispatch routine returns 0, it means it wants to be removed
// from the list; -1 indicates error and 1 means OK.  If there
// are no routines on the list then CommBase::Run returns.
int
CommBase::Run()
{
	int ret;
	
	while(dispatches > 0)
	{
		int i;

		fd_set r_set, w_set, e_set;
		FD_ZERO(&r_set);
		FD_ZERO(&w_set);
		FD_ZERO(&e_set);

		disp_fd *dfd;
		int topfd = 0;

		for(i = NDISP_PRI-1; i >= 0; i--)
			for(dfd = dispqs[i].h; dfd != NULL; dfd = dfd->next)
			{
				if (dfd->what & DISP_R)
					FD_SET(dfd->fd, &r_set);
				if (dfd->what & DISP_W)
					FD_SET(dfd->fd, &w_set);
				if (dfd->what & DISP_E)
					FD_SET(dfd->fd, &e_set);
				if (dfd->fd > topfd)
					topfd = dfd->fd;
			}
		
		ret = select(topfd+1, &r_set, &w_set, &e_set, NULL);

		if (ret == -1)
		{
			if (errno == EINTR)
				continue;
			else
			{
				perror("select failed");
				return ret;
			}
		}
		if (ret == 0)
		{
			fprintf(stderr, "Comm::Run ret == 0\n");
			continue;
		}
		
		disp_fd *next;
		for (i = NDISP_PRI-1; i >= 0; i--)
			for(dfd = dispqs[i].h; dfd != NULL; dfd = next)
			{
				int what = 0;
			
				next = dfd->next;
				if (dfd->what & DISP_R && FD_ISSET(dfd->fd, &r_set))
					what |= DISP_R;
				if (dfd->what & DISP_W && FD_ISSET(dfd->fd, &w_set))
					what |= DISP_W;
				if (dfd->what & DISP_E && FD_ISSET(dfd->fd, &e_set))
					what |= DISP_E;
			
				if (what == 0)
					continue;

				ret = dfd->dfd->dispatch(dfd->fd, what);

				if (ret == 1)
					continue;
				delDispatch(dfd);
				if (ret != 0)
					return ret;
			}
	}

	return 0;
}

// Close filedescriptors we really don't need
void
CommBase::CloseFDs(int all)
{
	disp_fd *fdlp, *next;

	for(int i = 0; i < NDISP_PRI; i++)
		for(fdlp = dispqs[i].h; fdlp != NULL; fdlp = next)
		{
			next = fdlp->next;
			
			if (all || fdlp->dfd != req)
				delDispatch(fdlp);
		}
}

DispToKern::DispToKern(const CommBase *comm)
{
	tokern = comm->tokern;
}
