// -*- C++ -*-

// Threaded Comms class

// This class sets up a LWP for each request

// (C) Copyright 1994 Jeremy Fitzhardinge <jeremy@sw.oz.au>
// This code is distributed under the terms of the
// GNU General Public Licence.  See COPYING for more details.

#include <Filesystem.h>
#include <ThreadComm.h>
#include <LWP.h>
#include <io.h>
#include <linux/userfs_fs.h>
#include <stdio.h>

class Thread_doreq: public Comm_doreq
{
	friend static void op_proc(int, char **, void *);
	friend class ThreadComm;

	// Argument for thread
	int fd;
	int stksz, threadpri;
	
	// Associated dispatch list and comm struct pointers
	struct disp_fd *disp;
	ThreadComm *comm;

	// Do the read work in a thread
	void doop();

	// Virtual from Comm_doreq
	int dispatch(int fd, int flags);

	void remself()	{ comm->delDispatch(disp); }
	
public:
	Thread_doreq(ThreadComm *, Filesystem &, int pri, int stk);
};

class EOF_handler: public DispatchFD
{
	int dispatch(int fd, int flags);
};

int
EOF_handler::dispatch(int fd, int)
{
	return 0;
}

ThreadComm::ThreadComm(Filesystem &fs, unsigned to, unsigned from,
		       int stk, int pri, int tpri)
	   : CommBase(fs, to, from)
{
	main = new LWP(pri);
	
	Thread_doreq *tdr = new Thread_doreq(this, filesys, stk, tpri);
	req = tdr;
	
	tdr->disp = addDispatch(from, tdr, DISP_R, NDISP_PRI);
	EOF_handler *eof = new EOF_handler();
	addDispatch(from, eof, DISP_E);
}

ThreadComm::~ThreadComm()
{
	delete main;
}

Thread_doreq::Thread_doreq(ThreadComm *tc, Filesystem &fs, int stk, int pri)
	     : Comm_doreq(tc, fs), stksz(stk), threadpri(pri), comm(tc)
{
}

// Hop-through funtion to do casting
void op_proc(int, char **, void *p)
{
	((Thread_doreq *)p)->doop();
	suicide();
	abort();	// NOTREACHED

}

// Process a request from the kernel
// Returns 0 on EOF, -1 on error and 1 for OK
int
Thread_doreq::dispatch(int infd, int)
{
	fd = infd;
	LWP *lwp = new LWP(threadpri, op_proc, stksz, 0, NULL, this);

	return 1;	// Always OK (no way of returning error)
}

void
Thread_doreq::doop()
{
	up_preamble pre;
	size_t presize = sizeof_up_preamble(&pre);
	Uchar hbuf[256];
	int ret;
	
	ret = fullread(fd, hbuf, presize);
	if (ret == 0)
	{
		remself();
		return;
	}
	
	if (ret != presize)
	{
		fprintf(stderr, "Thread_doreq::dispatch failed to get whole header (%d wanted, %d got)\n",
			presize, ret);
		remself();
		return;
	}

	upp_repl repl;
	size_t replsize = sizeof_upp_repl(&repl);

	assert(sizeof(hbuf) > replsize);
		
	decode_up_preamble(&pre, hbuf);

	Data *buf = new Data[USERFS_MAX_XFER];
	
	if (pre.size != 0 && (ret = fullread(fd, buf, pre.size)) != pre.size)
	{
		fprintf(stderr, "Thread_doreq::dispatch: failed to get whole body (%d wanted, %d got)\n",
			pre.size, ret);
		remself();
		delete buf;
		return;
	}

	ret = filesys.DoOp(pre, repl, buf);

	if (ret != 1)
	{
		remself();
		delete buf;
		return; // -1
	}

#ifdef DEBUG
	if (repl.errno != 0)
	{
		fprintf(stderr, "Request %d failing with %d %s\n",
			pre.op, repl.errno, strerror(repl.errno));
	}
#endif
		
	encode_upp_repl(&repl, hbuf);

	int err = 0;
	
	if (fullwrite(tokern, hbuf, replsize) != replsize)
		err = 1;
	else
		if (repl.size != 0 &&
		    fullwrite(tokern, buf, repl.size) != repl.size)
			err = 1;

	if (err)
		remself();

	delete buf;
	return;
}

