/*
 * $Id: fd.c,v 1.4 1998/07/24 11:59:14 marc Exp $
 *
 * Support routines for file descriptors (FD)
 *
 * James Peterson, 1987
 * (c) Copyright MCC, 1987
 * (c) Copyright Marc Vertes, 1998
 *
 */

#include <stdlib.h>
#include <stdio.h>
#include <errno.h>		/* for EINTR, EADDRINUSE, ... */
#include <sys/time.h>
#include <sys/types.h>
#include <unistd.h>

#include "scope.h"
#include "common.h"

struct FDDescriptor *FDD;	/* array of FD descriptors */
short MaxFD;			/* maximum number of FD's possible */
short nFDsInUse;		/* number of FD's actually in use */
long ReadDescriptors;		/* bit map of FD's in use -- for select  */
short HighestFD;		/* highest FD in use -- for select */

/*
 * All of this code is to support the handling of file descriptors (FD).
 * The idea is to keep a table of the FDs that are in use and why.
 * For each FD that is open for input, we keep the name of a procedure
 * to call if input arrives for that FD.  When an FD is created
 * (by an open, pipe, socket, ...) declare that by calling UsingFD.
 * When it is no longer in use (close ...), call NotUsingFD.
 */

/*----------------------------------------------------------------------*/
/*
 *  timer subsystem
 *  The timer queue is a list of Timer structures
 *  kept in order soonest first.
 *
 *  The TimerID data type is simply a counter, assigned
 *  serially to timers as they're created.
 */

typedef struct Timer {
    struct Timer *next;		/* link */
    struct timeval time;	/* time to fire */
    TimerID timer;		/* ident of timer */
    void (*handler) ();		/* handler to call */
    void *user_data;		/* user data to pass */
} Timer;

/* next timer id */
static TimerID nextTimerID = 0;

/* timer queue header */
static Timer *timerQueue = 0;

/*----------------------------------------------------------------------*/
static void add_millisecs(delta, time)
    unsigned long int delta;
    struct timeval *time;
{
    unsigned long int secs = delta / 1000;
    unsigned long int msecs = delta - (secs * 1000);

    time->tv_usec += msecs * 1000;
    time->tv_sec += secs;
    while (time->tv_usec > 1000000) {
	time->tv_sec++;
	time->tv_usec -= 1000000;
    }
}

/*----------------------------------------------------------------------*/
static int time_compare(t1, t2)
    register struct timeval *t1;
    register struct timeval *t2;
{

    if (t1->tv_sec > t2->tv_sec)
	return 1;
    if (t1->tv_sec < t2->tv_sec)
	return -1;

    if (t1->tv_usec > t2->tv_usec)
	return 1;
    if (t1->tv_usec < t2->tv_usec)
	return -1;
    return 0;

}

/*----------------------------------------------------------------------*/
static int time_subtract(out, t1, t2)
    register struct timeval *out;
    register struct timeval *t1;
    register struct timeval *t2;
{
    register i = time_compare(t1, t2);

    if (i == 0) {
	out->tv_sec = 0;
	out->tv_usec = 0;
	return 0;
    }
    if (i > 0) {
	out->tv_sec = t1->tv_sec - t2->tv_sec;
	out->tv_usec = t1->tv_usec - t2->tv_usec;
    } else {
	out->tv_sec = t2->tv_sec - t1->tv_sec;
	out->tv_usec = t2->tv_usec - t1->tv_usec;
    }

    if (out->tv_usec > 1000000) {
	out->tv_sec++;
	out->tv_usec -= 1000000;
    }
    if (out->tv_usec < 0) {
	out->tv_sec--;
	out->tv_usec += 1000000;
    }
    return i;
}

/*----------------------------------------------------------------------*/
TimerID CreateTimer(delay, handler, user_data)
    unsigned long int delay;
    void (*handler) ();
    void *user_data;
{
    TimerID result = ++nextTimerID;
    register Timer *t = (Timer *) Malloc(sizeof(Timer));
    register Timer **p;
    register Timer *n;

    /* turn the delta-time specified into absolute time */
    (void) gettimeofday(&t->time, 0);

    debug(128, (stderr,
		"insert %ld %ld %d %d\n",
		result,
		delay,
		t->time.tv_sec,
		t->time.tv_usec));

    add_millisecs(delay, &t->time);

    debug(128, (stderr,
		"insert time to fire %d %d\n",
		t->time.tv_sec,
		t->time.tv_usec));


    /* set up the list element */
    t->timer = result;
    t->handler = handler;
    t->user_data = user_data;

    /*
     * search the queue looking for the right place for this timer.
     * notice that the right place is immediately AFTER all
     * previously enqueued events with this time
     */

    p = &timerQueue;
    n = *p;
    while (n) {
	if (time_compare(&n->time, &t->time) > 0) {
	    /* the one in the queue is later than the new one */
	    break;
	}
	p = &(n->next);
	n = n->next;
    }

    /* insert t in the list right before n */
    t->next = n;
    *p = t;

    return result;

}

/*----------------------------------------------------------------------*/
void DeleteTimer(timer)
    TimerID timer;

{
    register Timer **p;
    register Timer *n;

    p = &timerQueue;
    n = *p;
    /* search for a timer element */
    while (n) {
	if (n->timer == timer) {
	    /* found the one */
	    /* break the link */
	    *p = n->next;
	    /* drop the data structure */
	    Free(n);
	    break;
	}
	p = &(n->next);
	n = n->next;
    }
    /* if we didn't find it, don't do anything */
}

/*----------------------------------------------------------------------*/
static struct timeval *handleTimers()
{
    struct timeval now;
    register Timer *t;

    static struct timeval diff;	/* pointer to this gets returned */

    /* timers */
    (void) gettimeofday(&now, 0);
    debug(128, (stderr,
		"main loop time now %d %d\n",
		now.tv_sec,
		now.tv_usec));

    /* handle all events on the queue whose time has come */
    t = timerQueue;
    while (t && (time_compare(&t->time, &now) <= 0)) {
	/* time to handle this event */
	/* point the timer queue to the next event */
	debug(128, (stderr,
		    "deliver %ld %d %d\n",
		    t->timer,
		    t->time.tv_sec,
		    t->time.tv_usec));
	timerQueue = t->next;
	(t->handler) (t->timer, t->user_data);
	Free(t);
	t = timerQueue;
    }

    /*
     * is there still something in the queue?
     * if so, figure out how long til it fires
     */
    if (t) {

	int i = time_subtract(&diff, &t->time, &now);
	if (i <= 0)
	    panic("cant happen in HandleTimers");
	debug(128, (stderr,
		    "waitfor %d %d\n",
		    diff.tv_sec,
		    diff.tv_usec));
	return &diff;		/* delta timeval */
    } else
	return 0;		/* No timers, null timeval */

}

/*----------------------------------------------------------------------*/
void InitializeFD()
{
    register short i;

    enterprocedure("InitializeFD");
    /* get the number of file descriptors the system will let us use */
#if defined(hpux) || defined(SVR4)
    MaxFD = _NFILE - 1;
#else
    MaxFD = getdtablesize();
#endif
    if (MaxFD > StaticMaxFD) {
	fprintf(stderr, "Recompile with larger StaticMaxFD value %d\n", MaxFD);
	MaxFD = StaticMaxFD;
    }
    /* allocate space for a File Descriptor (FD) Table */
    FDD = (struct FDDescriptor *)
	Malloc((long) (MaxFD * sizeof(struct FDDescriptor)));

    /* be sure all fd's are closed and marked not busy */
    for (i = 0; i < MaxFD; i++) {
	/* 0, 1, 2 are special (stdin, stdout, stderr) */
	if (i > 2)
	    (void) close(i);
	FDD[i].Busy = false;
    }

    /* save one FD for single file input or output like debugging */
    /* also the getservbyname call is currently using an FD */
    MaxFD -= 4;

    nFDsInUse = 0 /* stdin, stdout, stderr */ ;
    ReadDescriptors = 0;
    HighestFD = 0;

    UsingFD(fileno(stdin), (int (*)()) NULL);
    UsingFD(fileno(stdout), (int (*)()) NULL);
    UsingFD(fileno(stderr), (int (*)()) NULL);
}

/*----------------------------------------------------------------------*/
void UsingFD(fd, Handler)
    FD fd;
    int (*Handler) ();
{
    if (FDD[fd].Busy)
	NotUsingFD(fd);
    nFDsInUse += 1;

    FDD[fd].Busy = true;
    FDD[fd].InputHandler = Handler;
    if (Handler == NULL)
	ReadDescriptors &= ~(1 << fd) /* clear fd bit */ ;
    else
	ReadDescriptors |= 1 << fd /* set fd bit */ ;

    if (fd > HighestFD)
	HighestFD = fd;

    if (nFDsInUse >= MaxFD)
	panic("no more FDs");

    debug(128, (stderr, "Using FD %d, %d of %d in use\n", fd, nFDsInUse, MaxFD));
}

/*----------------------------------------------------------------------*/
void NotUsingFD(fd)
    FD fd;
{
    debug(128, (stderr, "Not Using FD %d\n", fd));

    if (FDD[fd].Busy)
	nFDsInUse -= 1;

    FDD[fd].Busy = false;
    ReadDescriptors &= ~(1 << fd) /* clear fd bit */ ;

    while (!FDD[HighestFD].Busy && HighestFD > 0)
	HighestFD -= 1;

    debug(128, (stderr, "Highest FD %d, in use %d\n", HighestFD, nFDsInUse));
}

/*----------------------------------------------------------------------*/
void EOFonFD(fd)
    FD fd;
{
    enterprocedure("EOFonFD");
    debug(128, (stderr, "EOF on %d\n", fd));
    (void) close(fd);
    NotUsingFD(fd);
}


/*----------------------------------------------------------------------*/
/*     Main Loop -- wait for input from any source and Process  */

extern int errno;

#define XLAB_PROMPT "(xlab) "

void MainLoop()
{
    int fd_stdin = fileno(stdin);
    int stdin_isatty = isatty(fd_stdin);

    enterprocedure("MainLoop");

    if (stdin_isatty) {
	fprintf(stdout, XLAB_PROMPT);
	fflush(stdout);
    }
    while (true) {
	int rfds, wfds, xfds;
	short nfds;
	short fd;
	struct timeval *time;

	/* wait for something */
	rfds = ReadDescriptors;
	wfds = 0;
	xfds = rfds;

	time = handleTimers();

	debug(128, (stderr, "select %d, rfds = 0%o\n", HighestFD + 1, rfds));
	nfds = select(HighestFD + 1,
		 (fd_set *) & rfds, (fd_set *) & wfds, (fd_set *) & xfds,
		      time);
	debug(128, (stderr, "select nfds = 0%o, rfds = 0%o, 0%o, xfds 0%o\n",
		    nfds, rfds, wfds, xfds));

	if (nfds < 0) {
	    if (errno == EINTR)
		continue /* to end of while loop */ ;
	    debug(1, (stderr, "Bad select - errno = %d\n", errno));
	    if (errno == EBADF) {
		/* one of the bits in rfds is invalid, close down
		   files until it goes away */
		EOFonFD(HighestFD);
		continue;
	    }
	    panic("Select returns error");
	    continue /* to end of while loop */ ;
	}
	if (nfds == 0) {
	    time = handleTimers();
	    continue;
	}
	/* check each fd to see if it has input */
	for (fd = 0; 0 < nfds && fd <= HighestFD; fd++) {
	    /*
	     * Check all returned fd's; this prevents
	     * starvation of later clients by earlier clients
	     */

	    if ((rfds & (1 << fd)) == 0)
		continue;

	    nfds -= 1;

	    if (FDD[fd].InputHandler == NULL) {
		panic("FD selected with no handler");
		debug(1, (stderr, "FD %d has NULL handler\n", fd));
	    } else
		(FDD[fd].InputHandler) (fd);
	    /*
	     * If the standard input is a tty (vs a file/pipe), issue a nice prompt
	     */
	    if (fd == fd_stdin && stdin_isatty) {
		fprintf(stdout, XLAB_PROMPT);
		fflush(stdout);
	    }
	}
    }
}
