/*
 * $Id: print11.c,v 1.4 1998/07/24 11:59:15 marc Exp $
 *
 *  Request, Reply, Event, Error Printing
 *
 *      James Peterson, 1988
 *      (c) Copyright MCC, 1988
 * Adapted for use in xscript 
 *      (c) Copyright 1993, Oliver Jones
 * Fixed for clean compile
 *      (c) Copyright 1998 Marc Vertes
 */

#include "scope.h"
#include "x11.h"
#include "server.h"
#include "decode11.h"
#include "print11.h"


/* Reply Buffer: Pseudo-buffer used to provide the opcode for the
   request to which this is a reply: Set by DecodeReply
   and used in the PrintField of the Reply procedure */
unsigned char RBf[2];


/* Sequence Buffer: Pseudo-buffer used to provide the sequence number for a
   request: Set by DecodeReply and used in a PrintField of 
   the Request procedure */
unsigned char SBf[4];

/*----------------------------------------------------------------------*/
/*
 * In printing the contents of the fields of the X11 packets, some
 * fields are of obvious value, and others are of lesser value.  To
 * control the amount of output, we generate our output according
 * to the level of Verbose-ness that was selected by the user.
 *
 * Verbose = 0 ==  Headers only, time and request/reply/... names.
 * Verbose = 1 ==  Very useful content fields.
 * Verbose = 2 ==  Almost everything.
 * Verbose = 3 ==  Every single bit and byte.
 */

/*
 * To aid in making the choice between level 1 and level 2, we
 * use *_1up and *_2up versions of printing macros.
 */

/*
 * DONTPRINT gets defined for versions of this module
 * which don't need output display.  Setting DONTPRINT
 * saves a lot of space in the object and executable.
 */

/* #define DONTPRINT */

#ifdef DONTPRINT
#define print_field_2up(a,b,c,d,e)
#define print_field_1up(a,b,c,d,e)
#define print_list_1up(a,b,c,d) (0)
#define print_list_2up(a,b,c,d) (0)
#define print_list_str_1up(a,b,c)
#define print_str8_1up(a,b,c)
#define print_str16_1up(a,b,c)
#define print_text_list8_1up(a,b,c)
#define print_text_list16_1up(a,b,c)
#define print_bytes_1up(a,b,c)
#define print_values_1up(a,b,c,d,e)
#define print_set_indent_level(a)
#else
/* conventional definitions (from xscope) */
#define print_field_2up(a,b,c,d,e) if (Verbose > 1) PrintField(a,b,c,d,e)
#define print_field_1up(a,b,c,d,e) PrintField(a,b,c,d,e)
#define print_list_1up(a,b,c,d) PrintList(a,b,c,d)
#define print_list_2up(a,b,c,d) if (Verbose > 1) PrintList(a,b,c,d)
#define print_list_str_1up(a,b,c) PrintListSTR(a,b,c)
#define print_str8_1up(a,b,c) PrintString8(a,b,c)
#define print_str16_1up(a,b,c) PrintString16(a,b,c)
#define print_text_list8_1up(a,b,c) PrintTextList8(a,b,c)
#define print_text_list16_1up(a,b,c) PrintTextList16(a,b,c)
#define print_bytes_1up(a,b,c) PrintBytes(a,b,c)
#define print_values_1up(a,b,c,d,e) PrintValues(a,b,c,d,e)
#define print_set_indent_level(a) SetIndentLevel(a)
#endif

/*----------------------------------------------------------------------*/
void PrintSetUpMessage(buf)
    unsigned char *buf;
{
    short n;
    short d;

    enterprocedure("PrintSetUpMessage");
    if (Verbose < 1)
	return;
    print_set_indent_level(PRINTCLIENT);
    print_field_1up(buf, 0, 1, BYTEMODE, "byte-order");
    print_field_1up(buf, 2, 2, CARD16, "major-version");
    print_field_1up(buf, 4, 2, CARD16, "minor-version");
    print_field_2up(buf, 6, 2, DVALUE2(n), "length of name");
    n = IShort(&buf[6]);
    print_field_2up(buf, 8, 2, DVALUE2(d), "length of data");
    d = IShort(&buf[8]);
    print_str8_1up(&buf[12], (long) n, "authorization-protocol-name");
    print_str8_1up(&buf[pad((long) (12 + n))], (long) d, "authorization-protocol-data");
}

/*----------------------------------------------------------------------*/
void PrintFailedSetUpReply(buf)
    unsigned char *buf;
{
    short n;

    print_field_1up(buf, 0, 1, 0, "SetUp Failed");
    if (Verbose < 1)
	return;
    print_field_2up(buf, 1, 1, DVALUE1(n), "length of reason in bytes");
    n = IByte(&buf[1]);
    print_field_1up(buf, 2, 2, CARD16, "major-version");
    print_field_1up(buf, 4, 2, CARD16, "minor-version");
    print_field_2up(buf, 6, 2, DVALUE2((n + p) / 4), "length of data");
    print_str8_1up(&buf[8], (long) n, "reason");
}

/*----------------------------------------------------------------------*/
void PrintSuccessfulSetUpReply(buf)
    unsigned char *buf;
{
    short v;
    short n;
    short m;

    if (Verbose < 1)
	return;
    print_field_1up(buf, 2, 2, CARD16, "protocol-major-version");
    print_field_1up(buf, 4, 2, CARD16, "protocol-minor-version");
    print_field_2up(buf, 6, 2, DVALUE2(8 + 2 * n + (v + p + m) / 4), "length of data");
    print_field_1up(buf, 8, 4, CARD32, "release-number");
    print_field_1up(buf, 12, 4, CARD32, "resource-id-base");
    print_field_1up(buf, 16, 4, CARD32, "resource-id-mask");
    print_field_1up(buf, 20, 4, CARD32, "motion-buffer-size");
    print_field_2up(buf, 24, 2, DVALUE2(v), "length of vendor");
    v = IShort(&buf[24]);
    print_field_2up(buf, 26, 2, CARD16, "maximum-request-length");
    print_field_2up(buf, 28, 1, CARD8, "number of roots");
    m = IByte(&buf[28]);
    print_field_2up(buf, 29, 1, DVALUE1(n), "number of pixmap-formats");
    n = IByte(&buf[29]);
    print_field_1up(buf, 30, 1, BYTEORDER, "image-byte-order");
    print_field_1up(buf, 31, 1, BYTEORDER, "bitmap-format-bit-order");
    print_field_1up(buf, 32, 1, CARD8, "bitmap-format-scanline-unit");
    print_field_1up(buf, 33, 1, CARD8, "bitmap-format-scanline-pad");
    print_field_1up(buf, 34, 1, KEYCODE, "min-keycode");
    print_field_1up(buf, 35, 1, KEYCODE, "max-keycode");
    print_str8_1up(&buf[40], (long) v, "vendor");
    (void) print_list_1up(&buf[pad((long) (40 + v))], (long) n, FORMAT, "pixmap-formats");
    (void) print_list_1up(&buf[pad((long) (40 + v) + 8 * n)], (long) m, SCREEN, "roots");
}

/*----------------------------------------------------------------------*/
void PrintSetUpReply(buf)
    unsigned char *buf;
{
    enterprocedure("PrintSetUpReply");
    print_set_indent_level(PRINTSERVER);
    if (IByte(&buf[0]))
	PrintSuccessfulSetUpReply(buf);
    else
	PrintFailedSetUpReply(buf);
}


/*----------------------------------------------------------------------*/
static char *REQUESTHEADER = "............REQUEST";
static char *EVENTHEADER = "..............EVENT";
static char *ERRORHEADER = "..............ERROR";
static char *REPLYHEADER = "..............REPLY";


/*----------------------------------------------------------------------*/
/* Error Printing procedures */
void RequestError(buf)
    unsigned char *buf;
{
    print_field_1up(buf, 1, 1, ERROR, ERRORHEADER);	/* Request */
    if (Verbose < 1)
	return;
    print_field_2up(buf, 2, 2, CARD16, "sequence number");
    print_field_1up(buf, 8, 2, CARD16, "minor opcode");
    print_field_1up(buf, 10, 1, CARD8, "major opcode");
}

/*----------------------------------------------------------------------*/
void ValueError(buf)
    unsigned char *buf;
{
    print_field_1up(buf, 1, 1, ERROR, ERRORHEADER);	/* Value */
    if (Verbose < 1)
	return;
    print_field_2up(buf, 2, 2, CARD16, "sequence number");
    print_field_1up(buf, 4, 4, INT32, "bad value");
    print_field_1up(buf, 8, 2, CARD16, "minor opcode");
    print_field_1up(buf, 10, 1, CARD8, "major opcode");
}

/*----------------------------------------------------------------------*/
void WindowError(buf)
    unsigned char *buf;
{
    print_field_1up(buf, 1, 1, ERROR, ERRORHEADER);	/* Window */
    if (Verbose < 1)
	return;
    print_field_2up(buf, 2, 2, CARD16, "sequence number");
    print_field_1up(buf, 4, 4, CARD32, "bad resource id");
    print_field_1up(buf, 8, 2, CARD16, "minor opcode");
    print_field_1up(buf, 10, 1, CARD8, "major opcode");
}

/*----------------------------------------------------------------------*/
void PixmapError(buf)
    unsigned char *buf;
{
    print_field_1up(buf, 1, 1, ERROR, ERRORHEADER);	/* Pixmap */
    if (Verbose < 1)
	return;
    print_field_2up(buf, 2, 2, CARD16, "sequence number");
    print_field_1up(buf, 4, 4, CARD32, "bad resource id");
    print_field_1up(buf, 8, 2, CARD16, "minor opcode");
    print_field_1up(buf, 10, 1, CARD8, "major opcode");
}

/*----------------------------------------------------------------------*/
void AtomError(buf)
    unsigned char *buf;
{
    print_field_1up(buf, 1, 1, ERROR, ERRORHEADER);	/* Atom */
    if (Verbose < 1)
	return;
    print_field_2up(buf, 2, 2, CARD16, "sequence number");
    print_field_1up(buf, 4, 4, CARD32, "bad atom id");
    print_field_1up(buf, 8, 2, CARD16, "minor opcode");
    print_field_1up(buf, 10, 1, CARD8, "major opcode");
}

/*----------------------------------------------------------------------*/
void CursorError(buf)
    unsigned char *buf;
{
    print_field_1up(buf, 1, 1, ERROR, ERRORHEADER);	/* Cursor */
    if (Verbose < 1)
	return;
    print_field_2up(buf, 2, 2, CARD16, "sequence number");
    print_field_1up(buf, 4, 4, CARD32, "bad resource id");
    print_field_1up(buf, 8, 2, CARD16, "minor opcode");
    print_field_1up(buf, 10, 1, CARD8, "major opcode");
}

/*----------------------------------------------------------------------*/
void FontError(buf)
    unsigned char *buf;
{
    print_field_1up(buf, 1, 1, ERROR, ERRORHEADER);	/* Font */
    if (Verbose < 1)
	return;
    print_field_2up(buf, 2, 2, CARD16, "sequence number");
    print_field_1up(buf, 4, 4, CARD32, "bad resource id");
    print_field_1up(buf, 8, 2, CARD16, "minor opcode");
    print_field_1up(buf, 10, 1, CARD8, "major opcode");
}

/*----------------------------------------------------------------------*/
void MatchError(buf)
    unsigned char *buf;
{
    print_field_1up(buf, 1, 1, ERROR, ERRORHEADER);	/* Match */
    if (Verbose < 1)
	return;
    print_field_2up(buf, 2, 2, CARD16, "sequence number");
    print_field_1up(buf, 8, 2, CARD16, "minor opcode");
    print_field_1up(buf, 10, 1, CARD8, "major opcode");
}

/*----------------------------------------------------------------------*/
void DrawableError(buf)
    unsigned char *buf;
{
    print_field_1up(buf, 1, 1, ERROR, ERRORHEADER);	/* Drawable */
    if (Verbose < 1)
	return;
    print_field_2up(buf, 2, 2, CARD16, "sequence number");
    print_field_1up(buf, 4, 4, CARD32, "bad resource id");
    print_field_1up(buf, 8, 2, CARD16, "minor opcode");
    print_field_1up(buf, 10, 1, CARD8, "major opcode");
}

/*----------------------------------------------------------------------*/
void AccessError(buf)
    unsigned char *buf;
{
    print_field_1up(buf, 1, 1, ERROR, ERRORHEADER);	/* Access */
    if (Verbose < 1)
	return;
    print_field_2up(buf, 2, 2, CARD16, "sequence number");
    print_field_1up(buf, 8, 2, CARD16, "minor opcode");
    print_field_1up(buf, 10, 1, CARD8, "major opcode");
}

/*----------------------------------------------------------------------*/
void AllocError(buf)
    unsigned char *buf;
{
    print_field_1up(buf, 1, 1, ERROR, ERRORHEADER);	/* Alloc */
    if (Verbose < 1)
	return;
    print_field_2up(buf, 2, 2, CARD16, "sequence number");
    print_field_1up(buf, 8, 2, CARD16, "minor opcode");
    print_field_1up(buf, 10, 1, CARD8, "major opcode");
}

/*----------------------------------------------------------------------*/
void ColormapError(buf)
    unsigned char *buf;
{
    print_field_1up(buf, 1, 1, ERROR, ERRORHEADER);	/* Colormap */
    if (Verbose < 1)
	return;
    print_field_2up(buf, 2, 2, CARD16, "sequence number");
    print_field_1up(buf, 4, 4, CARD32, "bad resource id");
    print_field_1up(buf, 8, 2, CARD16, "minor opcode");
    print_field_1up(buf, 10, 1, CARD8, "major opcode");
}

/*----------------------------------------------------------------------*/
void GContextError(buf)
    unsigned char *buf;
{
    print_field_1up(buf, 1, 1, ERROR, ERRORHEADER);	/* GContext */
    if (Verbose < 1)
	return;
    print_field_2up(buf, 2, 2, CARD16, "sequence number");
    print_field_1up(buf, 4, 4, CARD32, "bad resource id");
    print_field_1up(buf, 8, 2, CARD16, "minor opcode");
    print_field_1up(buf, 10, 1, CARD8, "major opcode");
}

/*----------------------------------------------------------------------*/
void IDChoiceError(buf)
    unsigned char *buf;
{
    print_field_1up(buf, 1, 1, ERROR, ERRORHEADER);	/* IDChoice */
    if (Verbose < 1)
	return;
    print_field_2up(buf, 2, 2, CARD16, "sequence number");
    print_field_1up(buf, 4, 4, CARD32, "bad resource id");
    print_field_1up(buf, 8, 2, CARD16, "minor opcode");
    print_field_1up(buf, 10, 1, CARD8, "major opcode");
}

/*----------------------------------------------------------------------*/
void NameError(buf)
    unsigned char *buf;
{
    print_field_1up(buf, 1, 1, ERROR, ERRORHEADER);	/* Name */
    if (Verbose < 1)
	return;
    print_field_2up(buf, 2, 2, CARD16, "sequence number");
    print_field_1up(buf, 8, 2, CARD16, "minor opcode");
    print_field_1up(buf, 10, 1, CARD8, "major opcode");
}

/*----------------------------------------------------------------------*/
void LengthError(buf)
    unsigned char *buf;
{
    print_field_1up(buf, 1, 1, ERROR, ERRORHEADER);	/* Length */
    if (Verbose < 1)
	return;
    print_field_2up(buf, 2, 2, CARD16, "sequence number");
    print_field_1up(buf, 8, 2, CARD16, "minor opcode");
    print_field_1up(buf, 10, 1, CARD8, "major opcode");
}

/*----------------------------------------------------------------------*/
void ImplementationError(buf)
    unsigned char *buf;
{
    print_field_1up(buf, 1, 1, ERROR, ERRORHEADER);	/* Implementation */
    if (Verbose < 1)
	return;
    print_field_2up(buf, 2, 2, CARD16, "sequence number");
    print_field_1up(buf, 8, 2, CARD16, "minor opcode");
    print_field_1up(buf, 10, 1, CARD8, "major opcode");
}


/*----------------------------------------------------------------------*/
/* Event Printing procedures */
void KeyPressEvent(buf)
    unsigned char *buf;
{
    print_field_1up(buf, 0, 1, EVENT, EVENTHEADER);	/* KeyPress */
    if (Verbose < 1)
	return;
    print_field_1up(buf, 1, 1, KEYCODE, "detail");
    print_field_2up(buf, 2, 2, CARD16, "sequence number");
    print_field_1up(buf, 4, 4, TIMESTAMP, "time");
    print_field_1up(buf, 8, 4, WINDOW, "root");
    print_field_1up(buf, 12, 4, WINDOW, "event");
    print_field_1up(buf, 16, 4, WINDOW, "child");
    print_field_1up(buf, 20, 2, INT16, "root-x");
    print_field_1up(buf, 22, 2, INT16, "root-y");
    print_field_1up(buf, 24, 2, INT16, "event-x");
    print_field_1up(buf, 26, 2, INT16, "event-y");
    print_field_1up(buf, 28, 2, SETofKEYBUTMASK, "state");
    print_field_1up(buf, 30, 1, BOOL, "same-screen");
}

/*----------------------------------------------------------------------*/
void KeyReleaseEvent(buf)
    unsigned char *buf;
{
    print_field_1up(buf, 0, 1, EVENT, EVENTHEADER);	/* KeyRelease */
    if (Verbose < 1)
	return;
    print_field_1up(buf, 1, 1, KEYCODE, "detail");
    print_field_2up(buf, 2, 2, CARD16, "sequence number");
    print_field_1up(buf, 4, 4, TIMESTAMP, "time");
    print_field_1up(buf, 8, 4, WINDOW, "root");
    print_field_1up(buf, 12, 4, WINDOW, "event");
    print_field_1up(buf, 16, 4, WINDOW, "child");
    print_field_1up(buf, 20, 2, INT16, "root-x");
    print_field_1up(buf, 22, 2, INT16, "root-y");
    print_field_1up(buf, 24, 2, INT16, "event-x");
    print_field_1up(buf, 26, 2, INT16, "event-y");
    print_field_1up(buf, 28, 2, SETofKEYBUTMASK, "state");
    print_field_1up(buf, 30, 1, BOOL, "same-screen");
}

/*----------------------------------------------------------------------*/
void ButtonPressEvent(buf)
    unsigned char *buf;
{
    print_field_1up(buf, 0, 1, EVENT, EVENTHEADER);	/* ButtonPress */
    if (Verbose < 1)
	return;
    print_field_1up(buf, 1, 1, BUTTON, "detail");
    print_field_2up(buf, 2, 2, CARD16, "sequence number");
    print_field_1up(buf, 4, 4, TIMESTAMP, "time");
    print_field_1up(buf, 8, 4, WINDOW, "root");
    print_field_1up(buf, 12, 4, WINDOW, "event");
    print_field_1up(buf, 16, 4, WINDOW, "child");
    print_field_1up(buf, 20, 2, INT16, "root-x");
    print_field_1up(buf, 22, 2, INT16, "root-y");
    print_field_1up(buf, 24, 2, INT16, "event-x");
    print_field_1up(buf, 26, 2, INT16, "event-y");
    print_field_1up(buf, 28, 2, SETofKEYBUTMASK, "state");
    print_field_1up(buf, 30, 1, BOOL, "same-screen");
}

/*----------------------------------------------------------------------*/
void ButtonReleaseEvent(buf)
    unsigned char *buf;
{
    print_field_1up(buf, 0, 1, EVENT, EVENTHEADER);	/* ButtonRelease */
    if (Verbose < 1)
	return;
    print_field_1up(buf, 1, 1, BUTTON, "detail");
    print_field_2up(buf, 2, 2, CARD16, "sequence number");
    print_field_1up(buf, 4, 4, TIMESTAMP, "time");
    print_field_1up(buf, 8, 4, WINDOW, "root");
    print_field_1up(buf, 12, 4, WINDOW, "event");
    print_field_1up(buf, 16, 4, WINDOW, "child");
    print_field_1up(buf, 20, 2, INT16, "root-x");
    print_field_1up(buf, 22, 2, INT16, "root-y");
    print_field_1up(buf, 24, 2, INT16, "event-x");
    print_field_1up(buf, 26, 2, INT16, "event-y");
    print_field_1up(buf, 28, 2, SETofKEYBUTMASK, "state");
    print_field_1up(buf, 30, 1, BOOL, "same-screen");
}

/*----------------------------------------------------------------------*/
void MotionNotifyEvent(buf)
    unsigned char *buf;
{
    print_field_1up(buf, 0, 1, EVENT, EVENTHEADER);	/* MotionNotify */
    if (Verbose < 1)
	return;
    print_field_1up(buf, 1, 1, MOTIONDETAIL, "detail");
    print_field_2up(buf, 2, 2, CARD16, "sequence number");
    print_field_1up(buf, 4, 4, TIMESTAMP, "time");
    print_field_1up(buf, 8, 4, WINDOW, "root");
    print_field_1up(buf, 12, 4, WINDOW, "event");
    print_field_1up(buf, 16, 4, WINDOW, "child");
    print_field_1up(buf, 20, 2, INT16, "root-x");
    print_field_1up(buf, 22, 2, INT16, "root-y");
    print_field_1up(buf, 24, 2, INT16, "event-x");
    print_field_1up(buf, 26, 2, INT16, "event-y");
    print_field_1up(buf, 28, 2, SETofKEYBUTMASK, "state");
    print_field_1up(buf, 30, 1, BOOL, "same-screen");
}

/*----------------------------------------------------------------------*/
void EnterNotifyEvent(buf)
    unsigned char *buf;
{
    print_field_1up(buf, 0, 1, EVENT, EVENTHEADER);	/* EnterNotify */
    if (Verbose < 1)
	return;
    print_field_1up(buf, 1, 1, ENTERDETAIL, "detail");
    print_field_2up(buf, 2, 2, CARD16, "sequence number");
    print_field_1up(buf, 4, 4, TIMESTAMP, "time");
    print_field_1up(buf, 8, 4, WINDOW, "root");
    print_field_1up(buf, 12, 4, WINDOW, "event");
    print_field_1up(buf, 16, 4, WINDOW, "child");
    print_field_1up(buf, 20, 2, INT16, "root-x");
    print_field_1up(buf, 22, 2, INT16, "root-y");
    print_field_1up(buf, 24, 2, INT16, "event-x");
    print_field_1up(buf, 26, 2, INT16, "event-y");
    print_field_1up(buf, 28, 2, SETofKEYBUTMASK, "state");
    print_field_1up(buf, 30, 1, BUTTONMODE, "mode");
    print_field_1up(buf, 31, 1, SCREENFOCUS, "same-screen, focus");
}

/*----------------------------------------------------------------------*/
void LeaveNotifyEvent(buf)
    unsigned char *buf;
{
    print_field_1up(buf, 0, 1, EVENT, EVENTHEADER);	/* LeaveNotify */
    if (Verbose < 1)
	return;
    print_field_1up(buf, 1, 1, ENTERDETAIL, "detail");
    print_field_2up(buf, 2, 2, CARD16, "sequence number");
    print_field_1up(buf, 4, 4, TIMESTAMP, "time");
    print_field_1up(buf, 8, 4, WINDOW, "root");
    print_field_1up(buf, 12, 4, WINDOW, "event");
    print_field_1up(buf, 16, 4, WINDOW, "child");
    print_field_1up(buf, 20, 2, INT16, "root-x");
    print_field_1up(buf, 22, 2, INT16, "root-y");
    print_field_1up(buf, 24, 2, INT16, "event-x");
    print_field_1up(buf, 26, 2, INT16, "event-y");
    print_field_1up(buf, 28, 2, SETofKEYBUTMASK, "state");
    print_field_1up(buf, 30, 1, BUTTONMODE, "mode");
    print_field_1up(buf, 31, 1, SCREENFOCUS, "same-screen, focus");
}

/*----------------------------------------------------------------------*/
void FocusInEvent(buf)
    unsigned char *buf;
{
    print_field_1up(buf, 0, 1, EVENT, EVENTHEADER);	/* FocusIn */
    if (Verbose < 1)
	return;
    print_field_1up(buf, 1, 1, ENTERDETAIL, "detail");
    print_field_2up(buf, 2, 2, CARD16, "sequence number");
    print_field_1up(buf, 4, 4, WINDOW, "event");
    print_field_1up(buf, 8, 1, BUTTONMODE, "mode");
}

/*----------------------------------------------------------------------*/
void FocusOutEvent(buf)
    unsigned char *buf;
{
    print_field_1up(buf, 0, 1, EVENT, EVENTHEADER);	/* FocusOut */
    if (Verbose < 1)
	return;
    print_field_1up(buf, 1, 1, ENTERDETAIL, "detail");
    print_field_2up(buf, 2, 2, CARD16, "sequence number");
    print_field_1up(buf, 4, 4, WINDOW, "event");
    print_field_1up(buf, 8, 1, BUTTONMODE, "mode");
}

/*----------------------------------------------------------------------*/
void KeymapNotifyEvent(buf)
    unsigned char *buf;
{
    print_field_1up(buf, 0, 1, EVENT, EVENTHEADER);	/* KeymapNotify */
    if (Verbose < 1)
	return;
    print_bytes_1up(&buf[1], (long) 31, "keys");
}

/*----------------------------------------------------------------------*/
void ExposeEvent(buf)
    unsigned char *buf;
{
    print_field_1up(buf, 0, 1, EVENT, EVENTHEADER);	/* Expose */
    if (Verbose < 1)
	return;
    print_field_2up(buf, 2, 2, CARD16, "sequence number");
    print_field_1up(buf, 4, 4, WINDOW, "window");
    print_field_1up(buf, 8, 2, CARD16, "x");
    print_field_1up(buf, 10, 2, CARD16, "y");
    print_field_1up(buf, 12, 2, CARD16, "width");
    print_field_1up(buf, 14, 2, CARD16, "height");
    print_field_1up(buf, 16, 2, CARD16, "count");
}

/*----------------------------------------------------------------------*/
void GraphicsExposureEvent(buf)
    unsigned char *buf;
{
    print_field_1up(buf, 0, 1, EVENT, EVENTHEADER);	/* GraphicsExposure */
    if (Verbose < 1)
	return;
    print_field_2up(buf, 2, 2, CARD16, "sequence number");
    print_field_1up(buf, 4, 4, DRAWABLE, "drawable");
    print_field_1up(buf, 8, 2, CARD16, "x");
    print_field_1up(buf, 10, 2, CARD16, "y");
    print_field_1up(buf, 12, 2, CARD16, "width");
    print_field_1up(buf, 14, 2, CARD16, "height");
    print_field_1up(buf, 16, 2, CARD16, "minor-opcode");
    print_field_1up(buf, 18, 2, CARD16, "count");
    print_field_1up(buf, 20, 1, CARD8, "major-opcode");
}

/*----------------------------------------------------------------------*/
void NoExposureEvent(buf)
    unsigned char *buf;
{
    print_field_1up(buf, 0, 1, EVENT, EVENTHEADER);	/* NoExposure */
    if (Verbose < 1)
	return;
    print_field_2up(buf, 2, 2, CARD16, "sequence number");
    print_field_1up(buf, 4, 4, DRAWABLE, "drawable");
    print_field_1up(buf, 8, 2, CARD16, "minor-opcode");
    print_field_1up(buf, 10, 1, CARD8, "major-opcode");
}

/*----------------------------------------------------------------------*/
void VisibilityNotifyEvent(buf)
    unsigned char *buf;
{
    print_field_1up(buf, 0, 1, EVENT, EVENTHEADER);	/* VisibilityNotify */
    if (Verbose < 1)
	return;
    print_field_2up(buf, 2, 2, CARD16, "sequence number");
    print_field_1up(buf, 4, 4, WINDOW, "window");
    print_field_1up(buf, 8, 1, VISIBLE, "state");
}

/*----------------------------------------------------------------------*/
void CreateNotifyEvent(buf)
    unsigned char *buf;
{
    print_field_1up(buf, 0, 1, EVENT, EVENTHEADER);	/* CreateNotify */
    if (Verbose < 1)
	return;
    print_field_2up(buf, 2, 2, CARD16, "sequence number");
    print_field_1up(buf, 4, 4, WINDOW, "parent");
    print_field_1up(buf, 8, 4, WINDOW, "window");
    print_field_1up(buf, 12, 2, INT16, "x");
    print_field_1up(buf, 14, 2, INT16, "y");
    print_field_1up(buf, 16, 2, CARD16, "width");
    print_field_1up(buf, 18, 2, CARD16, "height");
    print_field_1up(buf, 20, 2, CARD16, "border-width");
    print_field_1up(buf, 22, 1, BOOL, "override-redirect");
}

/*----------------------------------------------------------------------*/
void DestroyNotifyEvent(buf)
    unsigned char *buf;
{
    print_field_1up(buf, 0, 1, EVENT, EVENTHEADER);	/* DestroyNotify */
    if (Verbose < 1)
	return;
    print_field_2up(buf, 2, 2, CARD16, "sequence number");
    print_field_1up(buf, 4, 4, WINDOW, "event");
    print_field_1up(buf, 8, 4, WINDOW, "window");
}

/*----------------------------------------------------------------------*/
void UnmapNotifyEvent(buf)
    unsigned char *buf;
{
    print_field_1up(buf, 0, 1, EVENT, EVENTHEADER);	/* UnmapNotify */
    if (Verbose < 1)
	return;
    print_field_2up(buf, 2, 2, CARD16, "sequence number");
    print_field_1up(buf, 4, 4, WINDOW, "event");
    print_field_1up(buf, 8, 4, WINDOW, "window");
    print_field_1up(buf, 12, 1, BOOL, "from-configure");
}

/*----------------------------------------------------------------------*/
void MapNotifyEvent(buf)
    unsigned char *buf;
{
    print_field_1up(buf, 0, 1, EVENT, EVENTHEADER);	/* MapNotify */
    if (Verbose < 1)
	return;
    print_field_2up(buf, 2, 2, CARD16, "sequence number");
    print_field_1up(buf, 4, 4, WINDOW, "event");
    print_field_1up(buf, 8, 4, WINDOW, "window");
    print_field_1up(buf, 12, 1, BOOL, "override-redirect");
}

/*----------------------------------------------------------------------*/
void MapRequestEvent(buf)
    unsigned char *buf;
{
    print_field_1up(buf, 0, 1, EVENT, EVENTHEADER);	/* MapRequest */
    if (Verbose < 1)
	return;
    print_field_2up(buf, 2, 2, CARD16, "sequence number");
    print_field_1up(buf, 4, 4, WINDOW, "parent");
    print_field_1up(buf, 8, 4, WINDOW, "window");
}

/*----------------------------------------------------------------------*/
void ReparentNotifyEvent(buf)
    unsigned char *buf;
{
    print_field_1up(buf, 0, 1, EVENT, EVENTHEADER);	/* ReparentNotify */
    if (Verbose < 1)
	return;
    print_field_2up(buf, 2, 2, CARD16, "sequence number");
    print_field_1up(buf, 4, 4, WINDOW, "event");
    print_field_1up(buf, 8, 4, WINDOW, "window");
    print_field_1up(buf, 12, 4, WINDOW, "parent");
    print_field_1up(buf, 16, 2, INT16, "x");
    print_field_1up(buf, 18, 2, INT16, "y");
    print_field_1up(buf, 20, 1, BOOL, "override-redirect");
}

/*----------------------------------------------------------------------*/
void ConfigureNotifyEvent(buf)
    unsigned char *buf;
{
    print_field_1up(buf, 0, 1, EVENT, EVENTHEADER);	/* ConfigureNotify */
    if (Verbose < 1)
	return;
    print_field_2up(buf, 2, 2, CARD16, "sequence number");
    print_field_1up(buf, 4, 4, WINDOW, "event");
    print_field_1up(buf, 8, 4, WINDOW, "window");
    print_field_1up(buf, 12, 4, WINDOW, "above-sibling");
    print_field_1up(buf, 16, 2, INT16, "x");
    print_field_1up(buf, 18, 2, INT16, "y");
    print_field_1up(buf, 20, 2, CARD16, "width");
    print_field_1up(buf, 22, 2, CARD16, "height");
    print_field_1up(buf, 24, 2, CARD16, "border-width");
    print_field_1up(buf, 26, 1, BOOL, "override-redirect");
}

/*----------------------------------------------------------------------*/
void ConfigureRequestEvent(buf)
    unsigned char *buf;
{
    print_field_1up(buf, 0, 1, EVENT, EVENTHEADER);	/* ConfigureRequest */
    if (Verbose < 1)
	return;
    print_field_1up(buf, 1, 1, STACKMODE, "stack-mode");
    print_field_2up(buf, 2, 2, CARD16, "sequence number");
    print_field_1up(buf, 4, 4, WINDOW, "parent");
    print_field_1up(buf, 8, 4, WINDOW, "window");
    print_field_1up(buf, 12, 4, WINDOW, "sibling");
    print_field_1up(buf, 16, 2, INT16, "x");
    print_field_1up(buf, 18, 2, INT16, "y");
    print_field_1up(buf, 20, 2, CARD16, "width");
    print_field_1up(buf, 22, 2, CARD16, "height");
    print_field_1up(buf, 24, 2, CARD16, "border-width");
    print_field_1up(buf, 26, 2, CONFIGURE_BITMASK, "value-mask");
}

/*----------------------------------------------------------------------*/
void GravityNotifyEvent(buf)
    unsigned char *buf;
{
    print_field_1up(buf, 0, 1, EVENT, EVENTHEADER);	/* GravityNotify */
    if (Verbose < 1)
	return;
    print_field_2up(buf, 2, 2, CARD16, "sequence number");
    print_field_1up(buf, 4, 4, WINDOW, "event");
    print_field_1up(buf, 8, 4, WINDOW, "window");
    print_field_1up(buf, 12, 2, INT16, "x");
    print_field_1up(buf, 14, 2, INT16, "y");
}

/*----------------------------------------------------------------------*/
void ResizeRequestEvent(buf)
    unsigned char *buf;
{
    print_field_1up(buf, 0, 1, EVENT, EVENTHEADER);	/* ResizeRequest */
    if (Verbose < 1)
	return;
    print_field_2up(buf, 2, 2, CARD16, "sequence number");
    print_field_1up(buf, 4, 4, WINDOW, "window");
    print_field_1up(buf, 8, 2, CARD16, "width");
    print_field_1up(buf, 10, 2, CARD16, "height");
}

/*----------------------------------------------------------------------*/
void CirculateNotifyEvent(buf)
    unsigned char *buf;
{
    print_field_1up(buf, 0, 1, EVENT, EVENTHEADER);	/* CirculateNotify */
    if (Verbose < 1)
	return;
    print_field_2up(buf, 2, 2, CARD16, "sequence number");
    print_field_1up(buf, 4, 4, WINDOW, "event");
    print_field_1up(buf, 8, 4, WINDOW, "window");
    print_field_1up(buf, 12, 4, WINDOW, "parent");
    print_field_1up(buf, 16, 1, CIRSTAT, "place");
}

/*----------------------------------------------------------------------*/
void CirculateRequestEvent(buf)
    unsigned char *buf;
{
    print_field_1up(buf, 0, 1, EVENT, EVENTHEADER);	/* CirculateRequest */
    if (Verbose < 1)
	return;
    print_field_2up(buf, 2, 2, CARD16, "sequence number");
    print_field_1up(buf, 4, 4, WINDOW, "parent");
    print_field_1up(buf, 8, 4, WINDOW, "window");
    print_field_1up(buf, 16, 1, CIRSTAT, "place");
}

/*----------------------------------------------------------------------*/
void PropertyNotifyEvent(buf)
    unsigned char *buf;
{
    print_field_1up(buf, 0, 1, EVENT, EVENTHEADER);	/* PropertyNotify */
    if (Verbose < 1)
	return;
    print_field_2up(buf, 2, 2, CARD16, "sequence number");
    print_field_1up(buf, 4, 4, WINDOW, "window");
    print_field_1up(buf, 8, 4, ATOM, "atom");
    print_field_1up(buf, 12, 4, TIMESTAMP, "time");
    print_field_1up(buf, 16, 1, PROPCHANGE, "state");
}

/*----------------------------------------------------------------------*/
void SelectionClearEvent(buf)
    unsigned char *buf;
{
    print_field_1up(buf, 0, 1, EVENT, EVENTHEADER);	/* SelectionClear */
    if (Verbose < 1)
	return;
    print_field_2up(buf, 2, 2, CARD16, "sequence number");
    print_field_1up(buf, 4, 4, TIMESTAMP, "time");
    print_field_1up(buf, 8, 4, WINDOW, "owner");
    print_field_1up(buf, 12, 4, ATOM, "selection");
}

/*----------------------------------------------------------------------*/
void SelectionRequestEvent(buf)
    unsigned char *buf;
{
    print_field_1up(buf, 0, 1, EVENT, EVENTHEADER);	/* SelectionRequest */
    if (Verbose < 1)
	return;
    print_field_2up(buf, 2, 2, CARD16, "sequence number");
    print_field_1up(buf, 4, 4, TIMESTAMP, "time");
    print_field_1up(buf, 8, 4, WINDOW, "owner");
    print_field_1up(buf, 12, 4, WINDOW, "requestor");
    print_field_1up(buf, 16, 4, ATOM, "selection");
    print_field_1up(buf, 20, 4, ATOM, "target");
    print_field_1up(buf, 24, 4, ATOM, "property");
}

/*----------------------------------------------------------------------*/
void SelectionNotifyEvent(buf)
    unsigned char *buf;
{
    print_field_1up(buf, 0, 1, EVENT, EVENTHEADER);	/* SelectionNotify */
    if (Verbose < 1)
	return;
    print_field_2up(buf, 2, 2, CARD16, "sequence number");
    print_field_1up(buf, 4, 4, TIMESTAMP, "time");
    print_field_1up(buf, 8, 4, WINDOW, "requestor");
    print_field_1up(buf, 12, 4, ATOM, "selection");
    print_field_1up(buf, 16, 4, ATOM, "target");
    print_field_1up(buf, 20, 4, ATOM, "property");
}

/*----------------------------------------------------------------------*/
void ColormapNotifyEvent(buf)
    unsigned char *buf;
{
    print_field_1up(buf, 0, 1, EVENT, EVENTHEADER);	/* ColormapNotify */
    if (Verbose < 1)
	return;
    print_field_2up(buf, 2, 2, CARD16, "sequence number");
    print_field_1up(buf, 4, 4, WINDOW, "window");
    print_field_1up(buf, 8, 4, COLORMAP, "colormap");
    print_field_1up(buf, 12, 1, BOOL, "new");
    print_field_1up(buf, 13, 1, CMAPCHANGE, "state");
}

/*----------------------------------------------------------------------*/
void ClientMessageEvent(buf)
    unsigned char *buf;
{
    short format;
    long type;

    print_field_1up(buf, 0, 1, EVENT, EVENTHEADER);	/* ClientMessage */
    if (Verbose < 1)
	return;
    print_field_1up(buf, 1, 1, CARD8, "format");
    format = IByte(&buf[1]);
    print_field_2up(buf, 2, 2, CARD16, "sequence number");
    print_field_1up(buf, 4, 4, WINDOW, "window");
    print_field_1up(buf, 8, 4, ATOM, "type");
    type = ILong(&buf[8]);
    if (type == 31 /* string */ )
	print_str8_1up(&buf[12], 20L, "data");
    else if (format == 16)
	(void) print_list_1up(&buf[12], 10L, INT16, "data");
    else if (format == 32)
	(void) print_list_1up(&buf[12], 5L, INT32, "data");
    else
	print_bytes_1up(&buf[12], 20L, "data");
}

/*----------------------------------------------------------------------*/
void MappingNotifyEvent(buf)
    unsigned char *buf;
{
    print_field_1up(buf, 0, 1, EVENT, EVENTHEADER);	/* MappingNotify */
    if (Verbose < 1)
	return;
    print_field_2up(buf, 2, 2, CARD16, "sequence number");
    print_field_1up(buf, 4, 1, MAPOBJECT, "request");
    print_field_1up(buf, 5, 1, KEYCODE, "first-keycode");
    print_field_1up(buf, 6, 1, CARD8, "count");
}

/*----------------------------------------------------------------------*/
/* Request and Reply Printing procedures */
void CreateWindow(buf)
    unsigned char *buf;
{
    /* Request CreateWindow is opcode 1 */
    print_field_1up(buf, 0, 1, REQUEST, REQUESTHEADER);		/* CreateWindow */
    if (Verbose < 1)
	return;
    if (Verbose > 1)
	print_field_1up(SBf, 0, 4, CARD32, "sequence number");

    print_field_1up(buf, 1, 1, CARD8, "depth");
    print_field_2up(buf, 2, 2, DVALUE2(8 + n), "request length");
    print_field_1up(buf, 4, 4, WINDOW, "wid");
    print_field_1up(buf, 8, 4, WINDOW, "parent");
    print_field_1up(buf, 12, 2, INT16, "x");
    print_field_1up(buf, 14, 2, INT16, "y");
    print_field_1up(buf, 16, 2, CARD16, "width");
    print_field_1up(buf, 18, 2, CARD16, "height");
    print_field_1up(buf, 20, 2, CARD16, "border-width");
    print_field_1up(buf, 22, 2, WINDOWCLASS, "class");
    print_field_1up(buf, 24, 4, VISUALIDC, "visual");
    print_field_1up(buf, 28, 4, WINDOW_BITMASK, "value-mask");
    print_values_1up(&buf[28], 4, WINDOW_BITMASK, &buf[32], "value-list");
}

/*----------------------------------------------------------------------*/
void ChangeWindowAttributes(buf)
    unsigned char *buf;
{
    /* Request ChangeWindowAttributes is opcode 2 */
    print_field_1up(buf, 0, 1, REQUEST, REQUESTHEADER);		/* ChangeWindowAttributes */
    if (Verbose < 1)
	return;
    if (Verbose > 1)
	print_field_1up(SBf, 0, 4, CARD32, "sequence number");

    print_field_2up(buf, 2, 2, DVALUE2(3 + n), "request length");
    print_field_1up(buf, 4, 4, WINDOW, "window");
    print_field_1up(buf, 8, 4, WINDOW_BITMASK, "value-mask");
    print_values_1up(&buf[8], 4, WINDOW_BITMASK, &buf[12], "value-list");
}

/*----------------------------------------------------------------------*/
void GetWindowAttributes(buf)
    unsigned char *buf;
{
    /* Request GetWindowAttributes is opcode 3 */
    print_field_1up(buf, 0, 1, REQUEST, REQUESTHEADER);		/* GetWindowAttributes */
    if (Verbose < 1)
	return;
    if (Verbose > 1)
	print_field_1up(SBf, 0, 4, CARD32, "sequence number");

    print_field_2up(buf, 2, 2, CONST2(2), "request length");
    print_field_1up(buf, 4, 4, WINDOW, "window");
}

/*----------------------------------------------------------------------*/
void GetWindowAttributesReply(buf)
    unsigned char *buf;
{
    print_field_1up(RBf, 0, 1, REPLY, REPLYHEADER);	/* GetWindowAttributes */
    if (Verbose < 1)
	return;
    print_field_1up(buf, 1, 1, BACKSTORE, "backing-store");
    print_field_2up(buf, 2, 2, CARD16, "sequence number");
    print_field_2up(buf, 4, 4, CONST4(3), "reply length");
    print_field_1up(buf, 8, 4, VISUALID, "visual");
    print_field_1up(buf, 12, 2, WINDOWCLASS, "class");
    print_field_1up(buf, 14, 1, BITGRAVITY, "bit-gravity");
    print_field_1up(buf, 15, 1, WINGRAVITY, "win-gravity");
    print_field_1up(buf, 16, 4, CARD32, "backing-planes");
    print_field_1up(buf, 20, 4, CARD32, "backing-pixel");
    print_field_1up(buf, 24, 1, BOOL, "save-under");
    print_field_1up(buf, 25, 1, BOOL, "map-is-installed");
    print_field_1up(buf, 26, 1, MAPSTATE, "map-state");
    print_field_1up(buf, 27, 1, BOOL, "override-redirect");
    print_field_1up(buf, 28, 4, COLORMAP, "colormap");
    print_field_1up(buf, 32, 4, SETofEVENT, "all-event-masks");
    print_field_1up(buf, 36, 4, SETofEVENT, "your-event-mask");
    print_field_1up(buf, 40, 2, SETofDEVICEEVENT, "do-not-propagate-mask");
}

/*----------------------------------------------------------------------*/
void DestroyWindow(buf)
    unsigned char *buf;
{
    /* Request DestroyWindow is opcode 4 */
    print_field_1up(buf, 0, 1, REQUEST, REQUESTHEADER);		/* DestroyWindow */
    if (Verbose < 1)
	return;
    if (Verbose > 1)
	print_field_1up(SBf, 0, 4, CARD32, "sequence number");

    print_field_2up(buf, 2, 2, CONST2(2), "request length");
    print_field_1up(buf, 4, 4, WINDOW, "window");
}

/*----------------------------------------------------------------------*/
void DestroySubwindows(buf)
    unsigned char *buf;
{
    /* Request DestroySubwindows is opcode 5 */
    print_field_1up(buf, 0, 1, REQUEST, REQUESTHEADER);		/* DestroySubwindows */
    if (Verbose < 1)
	return;
    if (Verbose > 1)
	print_field_1up(SBf, 0, 4, CARD32, "sequence number");

    print_field_2up(buf, 2, 2, CONST2(2), "request length");
    print_field_1up(buf, 4, 4, WINDOW, "window");
}

/*----------------------------------------------------------------------*/
void ChangeSaveSet(buf)
    unsigned char *buf;
{
    /* Request ChangeSaveSet is opcode 6 */
    print_field_1up(buf, 0, 1, REQUEST, REQUESTHEADER);		/* ChangeSaveSet */
    if (Verbose < 1)
	return;
    if (Verbose > 1)
	print_field_1up(SBf, 0, 4, CARD32, "sequence number");

    print_field_1up(buf, 1, 1, INS_DEL, "mode");
    print_field_2up(buf, 2, 2, CONST2(2), "request length");
    print_field_1up(buf, 4, 4, WINDOW, "window");
}

/*----------------------------------------------------------------------*/
void ReparentWindow(buf)
    unsigned char *buf;
{
    /* Request ReparentWindow is opcode 7 */
    print_field_1up(buf, 0, 1, REQUEST, REQUESTHEADER);		/* ReparentWindow */
    if (Verbose < 1)
	return;
    if (Verbose > 1)
	print_field_1up(SBf, 0, 4, CARD32, "sequence number");

    print_field_2up(buf, 2, 2, CONST2(4), "request length");
    print_field_1up(buf, 4, 4, WINDOW, "window");
    print_field_1up(buf, 8, 4, WINDOW, "parent");
    print_field_1up(buf, 12, 2, INT16, "x");
    print_field_1up(buf, 14, 2, INT16, "y");
}

/*----------------------------------------------------------------------*/
void MapWindow(buf)
    unsigned char *buf;
{
    /* Request MapWindow is opcode 8 */
    print_field_1up(buf, 0, 1, REQUEST, REQUESTHEADER);		/* MapWindow */
    if (Verbose < 1)
	return;
    if (Verbose > 1)
	print_field_1up(SBf, 0, 4, CARD32, "sequence number");

    print_field_2up(buf, 2, 2, CONST2(2), "request length");
    print_field_1up(buf, 4, 4, WINDOW, "window");
}

/*----------------------------------------------------------------------*/
void MapSubwindows(buf)
    unsigned char *buf;
{
    /* Request MapSubwindows is opcode 9 */
    print_field_1up(buf, 0, 1, REQUEST, REQUESTHEADER);		/* MapSubwindows */
    if (Verbose < 1)
	return;
    if (Verbose > 1)
	print_field_1up(SBf, 0, 4, CARD32, "sequence number");

    print_field_2up(buf, 2, 2, CONST2(2), "request length");
    print_field_1up(buf, 4, 4, WINDOW, "window");
}

/*----------------------------------------------------------------------*/
void UnmapWindow(buf)
    unsigned char *buf;
{
    /* Request UnmapWindow is opcode 10 */
    print_field_1up(buf, 0, 1, REQUEST, REQUESTHEADER);		/* UnmapWindow */
    if (Verbose < 1)
	return;
    if (Verbose > 1)
	print_field_1up(SBf, 0, 4, CARD32, "sequence number");

    print_field_2up(buf, 2, 2, CONST2(2), "request length");
    print_field_1up(buf, 4, 4, WINDOW, "window");
}

/*----------------------------------------------------------------------*/
void UnmapSubwindows(buf)
    unsigned char *buf;
{
    /* Request UnmapSubwindows is opcode 11 */
    print_field_1up(buf, 0, 1, REQUEST, REQUESTHEADER);		/* UnmapSubwindows */
    if (Verbose < 1)
	return;
    if (Verbose > 1)
	print_field_1up(SBf, 0, 4, CARD32, "sequence number");

    print_field_2up(buf, 2, 2, CONST2(2), "request length");
    print_field_1up(buf, 4, 4, WINDOW, "window");
}

/*----------------------------------------------------------------------*/
void ConfigureWindow(buf)
    unsigned char *buf;
{
    /* Request ConfigureWindow is opcode 12 */
    print_field_1up(buf, 0, 1, REQUEST, REQUESTHEADER);		/* ConfigureWindow */
    if (Verbose < 1)
	return;
    if (Verbose > 1)
	print_field_1up(SBf, 0, 4, CARD32, "sequence number");

    print_field_2up(buf, 2, 2, DVALUE2(3 + n), "request length");
    print_field_1up(buf, 4, 4, WINDOW, "window");
    print_field_1up(buf, 8, 2, CONFIGURE_BITMASK, "value-mask");
    print_values_1up(&buf[8], 2, CONFIGURE_BITMASK, &buf[12], "value-list");
}

/*----------------------------------------------------------------------*/
void CirculateWindow(buf)
    unsigned char *buf;
{
    /* Request CirculateWindow is opcode 13 */
    print_field_1up(buf, 0, 1, REQUEST, REQUESTHEADER);		/* CirculateWindow */
    if (Verbose < 1)
	return;
    if (Verbose > 1)
	print_field_1up(SBf, 0, 4, CARD32, "sequence number");

    print_field_1up(buf, 1, 1, CIRMODE, "direction");
    print_field_2up(buf, 2, 2, CONST2(2), "request length");
    print_field_1up(buf, 4, 4, WINDOW, "window");
}

/*----------------------------------------------------------------------*/
void GetGeometry(buf)
    unsigned char *buf;
{
    /* Request GetGeometry is opcode 14 */
    print_field_1up(buf, 0, 1, REQUEST, REQUESTHEADER);		/* GetGeometry */
    if (Verbose < 1)
	return;
    if (Verbose > 1)
	print_field_1up(SBf, 0, 4, CARD32, "sequence number");

    print_field_2up(buf, 2, 2, CONST2(2), "request length");
    print_field_1up(buf, 4, 4, DRAWABLE, "drawable");
}

/*----------------------------------------------------------------------*/
void GetGeometryReply(buf)
    unsigned char *buf;
{
    print_field_1up(RBf, 0, 1, REPLY, REPLYHEADER);	/* GetGeometry */
    if (Verbose < 1)
	return;
    print_field_1up(buf, 1, 1, CARD8, "depth");
    print_field_2up(buf, 2, 2, CARD16, "sequence number");
    print_field_2up(buf, 4, 4, CONST4(0), "reply length");
    print_field_1up(buf, 8, 4, WINDOW, "root");
    print_field_1up(buf, 12, 2, INT16, "x");
    print_field_1up(buf, 14, 2, INT16, "y");
    print_field_1up(buf, 16, 2, CARD16, "width");
    print_field_1up(buf, 18, 2, CARD16, "height");
    print_field_1up(buf, 20, 2, CARD16, "border-width");
}

/*----------------------------------------------------------------------*/
void QueryTree(buf)
    unsigned char *buf;
{
    /* Request QueryTree is opcode 15 */
    print_field_1up(buf, 0, 1, REQUEST, REQUESTHEADER);		/* QueryTree */
    if (Verbose < 1)
	return;
    if (Verbose > 1)
	print_field_1up(SBf, 0, 4, CARD32, "sequence number");

    print_field_2up(buf, 2, 2, CONST2(2), "request length");
    print_field_1up(buf, 4, 4, WINDOW, "window");
}

/*----------------------------------------------------------------------*/
void QueryTreeReply(buf)
    unsigned char *buf;
{
    short n;

    print_field_1up(RBf, 0, 1, REPLY, REPLYHEADER);	/* QueryTree */
    if (Verbose < 1)
	return;
    print_field_2up(buf, 2, 2, CARD16, "sequence number");
    print_field_2up(buf, 4, 4, DVALUE4(n), "reply length");
    print_field_1up(buf, 8, 4, WINDOW, "root");
    print_field_1up(buf, 12, 4, WINDOW, "parent");
    print_field_2up(buf, 16, 2, DVALUE2(n), "number of children");
    n = IShort(&buf[16]);
    (void) print_list_1up(&buf[32], (long) n, WINDOW, "children");
}

/*----------------------------------------------------------------------*/
void InternAtom(buf)
    unsigned char *buf;
{
    short n;

    /* Request InternAtom is opcode 16 */
    print_field_1up(buf, 0, 1, REQUEST, REQUESTHEADER);		/* InternAtom */
    if (Verbose < 1)
	return;
    if (Verbose > 1)
	print_field_1up(SBf, 0, 4, CARD32, "sequence number");

    print_field_1up(buf, 1, 1, BOOL, "only-if-exists");
    print_field_2up(buf, 2, 2, DVALUE2(2 + (n + p) / 4), "request length");
    print_field_2up(buf, 4, 2, DVALUE2(n), "length of name");
    n = IShort(&buf[4]);
    print_str8_1up(&buf[8], (long) n, "name");
}

/*----------------------------------------------------------------------*/
void InternAtomReply(buf)
    unsigned char *buf;
{
    print_field_1up(RBf, 0, 1, REPLY, REPLYHEADER);	/* InternAtom */
    if (Verbose < 1)
	return;
    print_field_2up(buf, 2, 2, CARD16, "sequence number");
    print_field_2up(buf, 4, 4, CONST4(0), "reply length");
    print_field_1up(buf, 8, 4, ATOM, "atom");
}

/*----------------------------------------------------------------------*/
void GetAtomName(buf)
    unsigned char *buf;
{
    /* Request GetAtomName is opcode 17 */
    print_field_1up(buf, 0, 1, REQUEST, REQUESTHEADER);		/* GetAtomName */
    if (Verbose < 1)
	return;
    if (Verbose > 1)
	print_field_1up(SBf, 0, 4, CARD32, "sequence number");

    print_field_2up(buf, 2, 2, CONST2(2), "request length");
    print_field_1up(buf, 4, 4, ATOM, "atom");
}

/*----------------------------------------------------------------------*/
void GetAtomNameReply(buf)
    unsigned char *buf;
{
    short n;

    print_field_1up(RBf, 0, 1, REPLY, REPLYHEADER);	/* GetAtomName */
    if (Verbose < 1)
	return;
    print_field_2up(buf, 2, 2, CARD16, "sequence number");
    print_field_2up(buf, 4, 4, DVALUE4((n + p) / 4), "reply length");
    print_field_2up(buf, 8, 2, DVALUE2(n), "length of name");
    n = IShort(&buf[8]);
    print_str8_1up(&buf[32], (long) n, "name");
}

/*----------------------------------------------------------------------*/
void ChangeProperty(buf)
    unsigned char *buf;
{
    long n;
    short format;
    long type;

    n = ILong(&buf[20]);
    type = ILong(&buf[12]);
    format = IByte(&buf[16]);

    /* Request ChangeProperty is opcode 18 */
    print_field_1up(buf, 0, 1, REQUEST, REQUESTHEADER);		/* ChangeProperty */
    if (Verbose < 1)
	return;
    if (Verbose > 1)
	print_field_1up(SBf, 0, 4, CARD32, "sequence number");

    print_field_1up(buf, 1, 1, CHANGEMODE, "mode");
    print_field_2up(buf, 2, 2, DVALUE2(6 + (n + p) / 4), "request length");
    print_field_1up(buf, 4, 4, WINDOW, "window");
    print_field_1up(buf, 8, 4, ATOM, "property");
    print_field_1up(buf, 12, 4, ATOM, "type");
    print_field_1up(buf, 16, 1, CARD8, "format");
    print_field_2up(buf, 20, 4, CARD32, "length of data");
    if (type == 31 /* string */ ) {
	if (Verbose >= 3 || n < 40) {
	    /* print entire string */
	    print_str8_1up(&buf[24], n * format / 8, "data");
	} else {
	    /* print truncated string -- shortening */
	    n = 40;
	    print_str8_1up(&buf[24], n * format / 8, "data(partial)");
	}
    } else if (format == 16)
	(void) print_list_1up(&buf[24], n, INT16, "data");
    else if (format == 32)
	(void) print_list_1up(&buf[24], n, INT32, "data");
    else
	print_bytes_1up(&buf[24], n * format / 8, "data");
}

/*----------------------------------------------------------------------*/
void DeleteProperty(buf)
    unsigned char *buf;
{
    /* Request DeleteProperty is opcode 19 */
    print_field_1up(buf, 0, 1, REQUEST, REQUESTHEADER);		/* DeleteProperty */
    if (Verbose < 1)
	return;
    if (Verbose > 1)
	print_field_1up(SBf, 0, 4, CARD32, "sequence number");

    print_field_2up(buf, 2, 2, CONST2(3), "request length");
    print_field_1up(buf, 4, 4, WINDOW, "window");
    print_field_1up(buf, 8, 4, ATOM, "property");
}

/*----------------------------------------------------------------------*/
void GetProperty(buf)
    unsigned char *buf;
{
    /* Request GetProperty is opcode 20 */
    print_field_1up(buf, 0, 1, REQUEST, REQUESTHEADER);		/* GetProperty */
    if (Verbose < 1)
	return;
    if (Verbose > 1)
	print_field_1up(SBf, 0, 4, CARD32, "sequence number");

    print_field_1up(buf, 1, 1, BOOL, "delete");
    print_field_2up(buf, 2, 2, CONST2(6), "request length");
    print_field_1up(buf, 4, 4, WINDOW, "window");
    print_field_1up(buf, 8, 4, ATOM, "property");
    print_field_1up(buf, 12, 4, ATOMT, "type");
    print_field_1up(buf, 16, 4, CARD32, "long-offset");
    print_field_2up(buf, 20, 4, CARD32, "long-length");
}

/*----------------------------------------------------------------------*/
void GetPropertyReply(buf)
    unsigned char *buf;
{
    long n;
    short format;
    long type;

    type = ILong(&buf[8]);
    format = IByte(&buf[1]);
    n = ILong(&buf[16]);

    print_field_1up(RBf, 0, 1, REPLY, REPLYHEADER);	/* GetProperty */
    if (Verbose < 1)
	return;
    print_field_1up(buf, 1, 1, CARD8, "format");
    print_field_2up(buf, 2, 2, CARD16, "sequence number");
    print_field_2up(buf, 4, 4, DVALUE4((n + p) / 4), "reply length");
    print_field_1up(buf, 8, 4, ATOM, "type");
    print_field_1up(buf, 12, 4, CARD32, "bytes-after");
    print_field_2up(buf, 16, 4, CARD32, "length of value");

    if (type == 31 /* string */ ) {
	if (Verbose >= 3 || n < 40) {
	    /* print entire string */
	    print_str8_1up(&buf[32], n * format / 8, "data");
	} else {
	    /* print truncated string -- shortening */
	    n = 40;
	    print_str8_1up(&buf[32], n * format / 8, "data(partial)");
	}
    } else if (format == 16)
	(void) print_list_1up(&buf[32], n, INT16, "data");
    else if (format == 32)
	(void) print_list_1up(&buf[32], n, INT32, "data");
    else
	print_bytes_1up(&buf[32], n * format / 8, "data");
}

/*----------------------------------------------------------------------*/
void ListProperties(buf)
    unsigned char *buf;
{
    /* Request ListProperties is opcode 21 */
    print_field_1up(buf, 0, 1, REQUEST, REQUESTHEADER);		/* ListProperties */
    if (Verbose < 1)
	return;
    if (Verbose > 1)
	print_field_1up(SBf, 0, 4, CARD32, "sequence number");

    print_field_2up(buf, 2, 2, CONST2(2), "request length");
    print_field_1up(buf, 4, 4, WINDOW, "window");
}

/*----------------------------------------------------------------------*/
void ListPropertiesReply(buf)
    unsigned char *buf;
{
    short n;

    print_field_1up(RBf, 0, 1, REPLY, REPLYHEADER);	/* ListProperties */
    if (Verbose < 1)
	return;
    print_field_2up(buf, 2, 2, CARD16, "sequence number");
    print_field_2up(buf, 4, 4, DVALUE4(n), "reply length");
    print_field_2up(buf, 8, 2, DVALUE2(n), "number of atoms");
    n = IShort(&buf[8]);
    (void) print_list_1up(&buf[32], (long) n, ATOM, "atoms");
}

/*----------------------------------------------------------------------*/
void SetSelectionOwner(buf)
    unsigned char *buf;
{
    /* Request SetSelectionOwner is opcode 22 */
    print_field_1up(buf, 0, 1, REQUEST, REQUESTHEADER);		/* SetSelectionOwner */
    if (Verbose < 1)
	return;
    if (Verbose > 1)
	print_field_1up(SBf, 0, 4, CARD32, "sequence number");

    print_field_2up(buf, 2, 2, CONST2(4), "request length");
    print_field_1up(buf, 4, 4, WINDOW, "owner");
    print_field_1up(buf, 8, 4, ATOM, "selection");
    print_field_1up(buf, 12, 4, TIMESTAMP, "time");
}

/*----------------------------------------------------------------------*/
void GetSelectionOwner(buf)
    unsigned char *buf;
{
    /* Request GetSelectionOwner is opcode 23 */
    print_field_1up(buf, 0, 1, REQUEST, REQUESTHEADER);		/* GetSelectionOwner */
    if (Verbose < 1)
	return;
    if (Verbose > 1)
	print_field_1up(SBf, 0, 4, CARD32, "sequence number");

    print_field_2up(buf, 2, 2, CONST2(2), "request length");
    print_field_1up(buf, 4, 4, ATOM, "selection");
}

/*----------------------------------------------------------------------*/
void GetSelectionOwnerReply(buf)
    unsigned char *buf;
{
    print_field_1up(RBf, 0, 1, REPLY, REPLYHEADER);	/* GetSelectionOwner */
    if (Verbose < 1)
	return;
    print_field_2up(buf, 2, 2, CARD16, "sequence number");
    print_field_2up(buf, 4, 4, CONST4(0), "reply length");
    print_field_1up(buf, 8, 4, WINDOW, "owner");
}

/*----------------------------------------------------------------------*/
void ConvertSelection(buf)
    unsigned char *buf;
{
    /* Request ConvertSelection is opcode 24 */
    print_field_1up(buf, 0, 1, REQUEST, REQUESTHEADER);		/* ConvertSelection */
    if (Verbose < 1)
	return;
    if (Verbose > 1)
	print_field_1up(SBf, 0, 4, CARD32, "sequence number");

    print_field_2up(buf, 2, 2, CONST2(6), "request length");
    print_field_1up(buf, 4, 4, WINDOW, "requestor");
    print_field_1up(buf, 8, 4, ATOM, "selection");
    print_field_1up(buf, 12, 4, ATOM, "target");
    print_field_1up(buf, 16, 4, ATOM, "property");
    print_field_1up(buf, 20, 4, TIMESTAMP, "time");
}

/*----------------------------------------------------------------------*/
void SendEvent(buf)
    unsigned char *buf;
{
    /* Request SendEvent is opcode 25 */
    print_field_1up(buf, 0, 1, REQUEST, REQUESTHEADER);		/* SendEvent */
    if (Verbose < 1)
	return;
    if (Verbose > 1)
	print_field_1up(SBf, 0, 4, CARD32, "sequence number");

    print_field_1up(buf, 1, 1, BOOL, "propagate");
    print_field_2up(buf, 2, 2, CONST2(11), "request length");
    print_field_1up(buf, 4, 4, WINDOWD, "destination");
    print_field_1up(buf, 8, 4, SETofEVENT, "event-mask");
    print_field_1up(buf, 12, 32, EVENTFORM, "event");
}

/*----------------------------------------------------------------------*/
void GrabPointer(buf)
    unsigned char *buf;
{
    /* Request GrabPointer is opcode 26 */
    print_field_1up(buf, 0, 1, REQUEST, REQUESTHEADER);		/* GrabPointer */
    if (Verbose < 1)
	return;
    if (Verbose > 1)
	print_field_1up(SBf, 0, 4, CARD32, "sequence number");

    print_field_1up(buf, 1, 1, BOOL, "owner-events");
    print_field_2up(buf, 2, 2, CONST2(6), "request length");
    print_field_1up(buf, 4, 4, WINDOW, "grab-window");
    print_field_1up(buf, 8, 2, SETofPOINTEREVENT, "event-mask");
    print_field_1up(buf, 10, 1, PK_MODE, "pointer-mode");
    print_field_1up(buf, 11, 1, PK_MODE, "keyboard-mode");
    print_field_1up(buf, 12, 4, WINDOW, "confine-to");
    print_field_1up(buf, 16, 4, CURSOR, "cursor");
    print_field_1up(buf, 20, 4, TIMESTAMP, "time");
}

/*----------------------------------------------------------------------*/
void GrabPointerReply(buf)
    unsigned char *buf;
{
    print_field_1up(RBf, 0, 1, REPLY, REPLYHEADER);	/* GrabPointer */
    if (Verbose < 1)
	return;
    print_field_1up(buf, 1, 1, GRABSTAT, "status");
    print_field_2up(buf, 2, 2, CARD16, "sequence number");
    print_field_2up(buf, 4, 4, CONST4(0), "reply length");
}

/*----------------------------------------------------------------------*/
void UngrabPointer(buf)
    unsigned char *buf;
{
    /* Request UngrabPointer is opcode 27 */
    print_field_1up(buf, 0, 1, REQUEST, REQUESTHEADER);		/* UngrabPointer */
    if (Verbose < 1)
	return;
    if (Verbose > 1)
	print_field_1up(SBf, 0, 4, CARD32, "sequence number");

    print_field_2up(buf, 2, 2, CONST2(2), "request length");
    print_field_1up(buf, 4, 4, TIMESTAMP, "time");
}

/*----------------------------------------------------------------------*/
void GrabButton(buf)
    unsigned char *buf;
{
    /* Request GrabButton is opcode 28 */
    print_field_1up(buf, 0, 1, REQUEST, REQUESTHEADER);		/* GrabButton */
    if (Verbose < 1)
	return;
    if (Verbose > 1)
	print_field_1up(SBf, 0, 4, CARD32, "sequence number");

    print_field_1up(buf, 1, 1, BOOL, "owner-events");
    print_field_2up(buf, 2, 2, CONST2(6), "request length");
    print_field_1up(buf, 4, 4, WINDOW, "grab-window");
    print_field_1up(buf, 8, 2, SETofPOINTEREVENT, "event-mask");
    print_field_1up(buf, 10, 1, PK_MODE, "pointer-mode");
    print_field_1up(buf, 11, 1, PK_MODE, "keyboard-mode");
    print_field_1up(buf, 12, 4, WINDOW, "confine-to");
    print_field_1up(buf, 16, 4, CURSOR, "cursor");
    print_field_1up(buf, 20, 1, BUTTONA, "button");
    print_field_1up(buf, 22, 2, SETofKEYMASK, "modifiers");
}

/*----------------------------------------------------------------------*/
void UngrabButton(buf)
    unsigned char *buf;
{
    /* Request UngrabButton is opcode 29 */
    print_field_1up(buf, 0, 1, REQUEST, REQUESTHEADER);		/* UngrabButton */
    if (Verbose < 1)
	return;
    if (Verbose > 1)
	print_field_1up(SBf, 0, 4, CARD32, "sequence number");

    print_field_1up(buf, 1, 1, BUTTONA, "button");
    print_field_2up(buf, 2, 2, CONST2(3), "request length");
    print_field_1up(buf, 4, 4, WINDOW, "grab-window");
    print_field_1up(buf, 8, 2, SETofKEYMASK, "modifiers");
}

/*----------------------------------------------------------------------*/
void ChangeActivePointerGrab(buf)
    unsigned char *buf;
{
    /* Request ChangeActivePointerGrab is opcode 30 */
    print_field_1up(buf, 0, 1, REQUEST, REQUESTHEADER);		/* ChangeActivePointerGrab */
    if (Verbose < 1)
	return;
    if (Verbose > 1)
	print_field_1up(SBf, 0, 4, CARD32, "sequence number");

    print_field_2up(buf, 2, 2, CONST2(4), "request length");
    print_field_1up(buf, 4, 4, CURSOR, "cursor");
    print_field_1up(buf, 8, 4, TIMESTAMP, "time");
    print_field_1up(buf, 12, 2, SETofPOINTEREVENT, "event-mask");
}

/*----------------------------------------------------------------------*/
void GrabKeyboard(buf)
    unsigned char *buf;
{
    /* Request GrabKeyboard is opcode 31 */
    print_field_1up(buf, 0, 1, REQUEST, REQUESTHEADER);		/* GrabKeyboard */
    if (Verbose < 1)
	return;
    if (Verbose > 1)
	print_field_1up(SBf, 0, 4, CARD32, "sequence number");

    print_field_1up(buf, 1, 1, BOOL, "owner-events");
    print_field_2up(buf, 2, 2, CONST2(4), "request length");
    print_field_1up(buf, 4, 4, WINDOW, "grab-window");
    print_field_1up(buf, 8, 4, TIMESTAMP, "time");
    print_field_1up(buf, 12, 1, PK_MODE, "pointer-mode");
    print_field_1up(buf, 13, 1, PK_MODE, "keyboard-mode");
}

/*----------------------------------------------------------------------*/
void GrabKeyboardReply(buf)
    unsigned char *buf;
{
    print_field_1up(RBf, 0, 1, REPLY, REPLYHEADER);	/* GrabKeyboard */
    if (Verbose < 1)
	return;
    print_field_1up(buf, 1, 1, GRABSTAT, "status");
    print_field_2up(buf, 2, 2, CARD16, "sequence number");
    print_field_2up(buf, 4, 4, CONST4(0), "reply length");
}

/*----------------------------------------------------------------------*/
void UngrabKeyboard(buf)
    unsigned char *buf;
{
    /* Request UngrabKeyboard is opcode 32 */
    print_field_1up(buf, 0, 1, REQUEST, REQUESTHEADER);		/* UngrabKeyboard */
    if (Verbose < 1)
	return;
    if (Verbose > 1)
	print_field_1up(SBf, 0, 4, CARD32, "sequence number");

    print_field_2up(buf, 2, 2, CONST2(2), "request length");
    print_field_1up(buf, 4, 4, TIMESTAMP, "time");
}

/*----------------------------------------------------------------------*/
void GrabKey(buf)
    unsigned char *buf;
{
    /* Request GrabKey is opcode 33 */
    print_field_1up(buf, 0, 1, REQUEST, REQUESTHEADER);		/* GrabKey */
    if (Verbose < 1)
	return;
    if (Verbose > 1)
	print_field_1up(SBf, 0, 4, CARD32, "sequence number");

    print_field_1up(buf, 1, 1, BOOL, "owner-events");
    print_field_2up(buf, 2, 2, CONST2(4), "request length");
    print_field_1up(buf, 4, 4, WINDOW, "grab-window");
    print_field_1up(buf, 8, 2, SETofKEYMASK, "modifiers");
    print_field_1up(buf, 10, 1, KEYCODEA, "key");
    print_field_1up(buf, 11, 1, PK_MODE, "pointer-mode");
    print_field_1up(buf, 12, 1, PK_MODE, "keyboard-mode");
}

/*----------------------------------------------------------------------*/
void UngrabKey(buf)
    unsigned char *buf;
{
    /* Request UngrabKey is opcode 34 */
    print_field_1up(buf, 0, 1, REQUEST, REQUESTHEADER);		/* UngrabKey */
    if (Verbose < 1)
	return;
    if (Verbose > 1)
	print_field_1up(SBf, 0, 4, CARD32, "sequence number");

    print_field_1up(buf, 1, 1, KEYCODEA, "key");
    print_field_2up(buf, 2, 2, CONST2(3), "request length");
    print_field_1up(buf, 4, 4, WINDOW, "grab-window");
    print_field_1up(buf, 8, 2, SETofKEYMASK, "modifiers");
}

/*----------------------------------------------------------------------*/
void AllowEvents(buf)
    unsigned char *buf;
{
    /* Request AllowEvents is opcode 35 */
    print_field_1up(buf, 0, 1, REQUEST, REQUESTHEADER);		/* AllowEvents */
    if (Verbose < 1)
	return;
    if (Verbose > 1)
	print_field_1up(SBf, 0, 4, CARD32, "sequence number");

    print_field_1up(buf, 1, 1, EVENTMODE, "mode");
    print_field_2up(buf, 2, 2, CONST2(2), "request length");
    print_field_1up(buf, 4, 4, TIMESTAMP, "time");
}

/*----------------------------------------------------------------------*/
void GrabServer(buf)
    unsigned char *buf;
{
    /* Request GrabServer is opcode 36 */
    print_field_1up(buf, 0, 1, REQUEST, REQUESTHEADER);		/* GrabServer */
    if (Verbose < 1)
	return;
    if (Verbose > 1)
	print_field_1up(SBf, 0, 4, CARD32, "sequence number");

    print_field_2up(buf, 2, 2, CONST2(1), "request length");
}

/*----------------------------------------------------------------------*/
void UngrabServer(buf)
    unsigned char *buf;
{
    /* Request UngrabServer is opcode 37 */
    print_field_1up(buf, 0, 1, REQUEST, REQUESTHEADER);		/* UngrabServer */
    if (Verbose < 1)
	return;
    if (Verbose > 1)
	print_field_1up(SBf, 0, 4, CARD32, "sequence number");

    print_field_2up(buf, 2, 2, CONST2(1), "request length");
}

/*----------------------------------------------------------------------*/
void QueryPointer(buf)
    unsigned char *buf;
{
    /* Request QueryPointer is opcode 38 */
    print_field_1up(buf, 0, 1, REQUEST, REQUESTHEADER);		/* QueryPointer */
    if (Verbose < 1)
	return;
    if (Verbose > 1)
	print_field_1up(SBf, 0, 4, CARD32, "sequence number");

    print_field_2up(buf, 2, 2, CONST2(2), "request length");
    print_field_1up(buf, 4, 4, WINDOW, "window");
}

/*----------------------------------------------------------------------*/
void QueryPointerReply(buf)
    unsigned char *buf;
{
    print_field_1up(RBf, 0, 1, REPLY, REPLYHEADER);	/* QueryPointer */
    if (Verbose < 1)
	return;
    print_field_1up(buf, 1, 1, BOOL, "same-screen");
    print_field_2up(buf, 2, 2, CARD16, "sequence number");
    print_field_2up(buf, 4, 4, CONST4(0), "reply length");
    print_field_1up(buf, 8, 4, WINDOW, "root");
    print_field_1up(buf, 12, 4, WINDOW, "child");
    print_field_1up(buf, 16, 2, INT16, "root-x");
    print_field_1up(buf, 18, 2, INT16, "root-y");
    print_field_1up(buf, 20, 2, INT16, "win-x");
    print_field_1up(buf, 22, 2, INT16, "win-y");
    print_field_1up(buf, 24, 2, SETofKEYBUTMASK, "mask");
}

/*----------------------------------------------------------------------*/
void GetMotionEvents(buf)
    unsigned char *buf;
{
    /* Request GetMotionEvents is opcode 39 */
    print_field_1up(buf, 0, 1, REQUEST, REQUESTHEADER);		/* GetMotionEvents */
    if (Verbose < 1)
	return;
    if (Verbose > 1)
	print_field_1up(SBf, 0, 4, CARD32, "sequence number");

    print_field_2up(buf, 2, 2, CONST2(4), "request length");
    print_field_1up(buf, 4, 4, WINDOW, "window");
    print_field_1up(buf, 8, 4, TIMESTAMP, "start");
    print_field_1up(buf, 12, 4, TIMESTAMP, "stop");
}

/*----------------------------------------------------------------------*/
void GetMotionEventsReply(buf)
    unsigned char *buf;
{
    long n;
    print_field_1up(RBf, 0, 1, REPLY, REPLYHEADER);	/* GetMotionEvents */
    if (Verbose < 1)
	return;
    print_field_2up(buf, 2, 2, CARD16, "sequence number");
    print_field_2up(buf, 4, 4, DVALUE4(2 * n), "reply length");
    print_field_2up(buf, 8, 4, DVALUE4(n), "number of events");
    n = ILong(&buf[8]);
    (void) print_list_1up(&buf[32], n, TIMECOORD, "events");
}

/*----------------------------------------------------------------------*/
void TranslateCoordinates(buf)
    unsigned char *buf;
{
    /* Request TranslateCoordinates is opcode 40 */
    print_field_1up(buf, 0, 1, REQUEST, REQUESTHEADER);		/* TranslateCoordinates */
    if (Verbose < 1)
	return;
    if (Verbose > 1)
	print_field_1up(SBf, 0, 4, CARD32, "sequence number");

    print_field_2up(buf, 2, 2, CONST2(4), "request length");
    print_field_1up(buf, 4, 4, WINDOW, "src-window");
    print_field_1up(buf, 8, 4, WINDOW, "dst-window");
    print_field_1up(buf, 12, 2, INT16, "src-x");
    print_field_1up(buf, 14, 2, INT16, "src-y");
}

/*----------------------------------------------------------------------*/
void TranslateCoordinatesReply(buf)
    unsigned char *buf;
{
    print_field_1up(RBf, 0, 1, REPLY, REPLYHEADER);	/* TranslateCoordinates */
    if (Verbose < 1)
	return;
    print_field_1up(buf, 1, 1, BOOL, "same-screen");
    print_field_2up(buf, 2, 2, CARD16, "sequence number");
    print_field_2up(buf, 4, 4, CONST4(0), "reply length");
    print_field_1up(buf, 8, 4, WINDOW, "child");
    print_field_1up(buf, 12, 2, INT16, "dst-x");
    print_field_1up(buf, 14, 2, INT16, "dst-y");
}

/*----------------------------------------------------------------------*/
void WarpPointer(buf)
    unsigned char *buf;
{
    /* Request WarpPointer is opcode 41 */
    print_field_1up(buf, 0, 1, REQUEST, REQUESTHEADER);		/* WarpPointer */
    if (Verbose < 1)
	return;
    if (Verbose > 1)
	print_field_1up(SBf, 0, 4, CARD32, "sequence number");

    print_field_2up(buf, 2, 2, CONST2(6), "request length");
    print_field_1up(buf, 4, 4, WINDOW, "src-window");
    print_field_1up(buf, 8, 4, WINDOW, "dst-window");
    print_field_1up(buf, 12, 2, INT16, "src-x");
    print_field_1up(buf, 14, 2, INT16, "src-y");
    print_field_1up(buf, 16, 2, CARD16, "src-width");
    print_field_1up(buf, 18, 2, CARD16, "src-height");
    print_field_1up(buf, 20, 2, INT16, "dst-x");
    print_field_1up(buf, 22, 2, INT16, "dst-y");
}

/*----------------------------------------------------------------------*/
void SetInputFocus(buf)
    unsigned char *buf;
{
    /* Request SetInputFocus is opcode 42 */
    print_field_1up(buf, 0, 1, REQUEST, REQUESTHEADER);		/* SetInputFocus */
    if (Verbose < 1)
	return;
    if (Verbose > 1)
	print_field_1up(SBf, 0, 4, CARD32, "sequence number");

    print_field_1up(buf, 1, 1, FOCUSAGENT, "revert-to");
    print_field_2up(buf, 2, 2, CONST2(3), "request length");
    print_field_1up(buf, 4, 4, WINDOWNR, "focus");
    print_field_1up(buf, 8, 4, TIMESTAMP, "time");
}

/*----------------------------------------------------------------------*/
void GetInputFocus(buf)
    unsigned char *buf;
{
    /* Request GetInputFocus is opcode 43 */
    print_field_1up(buf, 0, 1, REQUEST, REQUESTHEADER);		/* GetInputFocus */
    if (Verbose < 1)
	return;
    if (Verbose > 1)
	print_field_1up(SBf, 0, 4, CARD32, "sequence number");

    print_field_2up(buf, 2, 2, CONST2(1), "request length");
}

/*----------------------------------------------------------------------*/
void GetInputFocusReply(buf)
    unsigned char *buf;
{
    print_field_1up(RBf, 0, 1, REPLY, REPLYHEADER);	/* GetInputFocus */
    if (Verbose < 1)
	return;
    print_field_1up(buf, 1, 1, FOCUSAGENT, "revert-to");
    print_field_2up(buf, 2, 2, CARD16, "sequence number");
    print_field_2up(buf, 4, 4, CONST4(0), "reply length");
    print_field_1up(buf, 8, 4, WINDOWNR, "focus");
}

/*----------------------------------------------------------------------*/
void QueryKeymap(buf)
    unsigned char *buf;
{
    /* Request QueryKeymap is opcode 44 */
    print_field_1up(buf, 0, 1, REQUEST, REQUESTHEADER);		/* QueryKeymap */
    if (Verbose < 1)
	return;
    if (Verbose > 1)
	print_field_1up(SBf, 0, 4, CARD32, "sequence number");

    print_field_2up(buf, 2, 2, CONST2(1), "request length");
}

/*----------------------------------------------------------------------*/
void QueryKeymapReply(buf)
    unsigned char *buf;
{
    print_field_1up(RBf, 0, 1, REPLY, REPLYHEADER);	/* QueryKeymap */
    if (Verbose < 1)
	return;
    print_field_2up(buf, 2, 2, CARD16, "sequence number");
    print_field_2up(buf, 4, 4, CONST4(2), "reply length");
    print_bytes_1up(&buf[8], 32L, "keys");
}

/*----------------------------------------------------------------------*/
void OpenFont(buf)
    unsigned char *buf;
{
    short n;

    /* Request OpenFont is opcode 45 */
    print_field_1up(buf, 0, 1, REQUEST, REQUESTHEADER);		/* OpenFont */
    if (Verbose < 1)
	return;
    if (Verbose > 1)
	print_field_1up(SBf, 0, 4, CARD32, "sequence number");

    print_field_2up(buf, 2, 2, DVALUE2(3 + (n + p) / 4), "request length");
    print_field_1up(buf, 4, 4, FONT, "font-id");
    print_field_2up(buf, 8, 2, DVALUE2(n), "length of name");
    n = IShort(&buf[8]);
    print_str8_1up(&buf[12], (long) n, "name");
}

/*----------------------------------------------------------------------*/
void CloseFont(buf)
    unsigned char *buf;
{
    /* Request CloseFont is opcode 46 */
    print_field_1up(buf, 0, 1, REQUEST, REQUESTHEADER);		/* CloseFont */
    if (Verbose < 1)
	return;
    if (Verbose > 1)
	print_field_1up(SBf, 0, 4, CARD32, "sequence number");

    print_field_2up(buf, 2, 2, CONST2(2), "request length");
    print_field_1up(buf, 4, 4, FONT, "font");
}

/*----------------------------------------------------------------------*/
void QueryFont(buf)
    unsigned char *buf;
{
    /* Request QueryFont is opcode 47 */
    print_field_1up(buf, 0, 1, REQUEST, REQUESTHEADER);		/* QueryFont */
    if (Verbose < 1)
	return;
    if (Verbose > 1)
	print_field_1up(SBf, 0, 4, CARD32, "sequence number");

    print_field_2up(buf, 2, 2, CONST2(2), "request length");
    print_field_1up(buf, 4, 4, FONTABLE, "font");
}

/*----------------------------------------------------------------------*/
void QueryFontReply(buf)
    unsigned char *buf;
{
    short n;
    long m;
    long k;

    print_field_1up(RBf, 0, 1, REPLY, REPLYHEADER);	/* QueryFont */
    if (Verbose < 1)
	return;
    print_field_2up(buf, 2, 2, CARD16, "sequence number");
    print_field_2up(buf, 4, 4, DVALUE4(7 + 2 * n + 3 * m), "reply length");
    print_field_1up(buf, 8, 12, CHARINFO, "min-bounds");
    print_field_1up(buf, 24, 12, CHARINFO, "max-bounds");
    print_field_1up(buf, 40, 2, CARD16, "min-char-or-byte2");
    print_field_1up(buf, 42, 2, CARD16, "max-char-or-byte2");
    print_field_1up(buf, 44, 2, CARD16, "default-char");
    print_field_2up(buf, 46, 2, DVALUE2(n), "number of FONTPROPs");
    n = IShort(&buf[46]);
    print_field_1up(buf, 48, 1, DIRECT, "draw-direction");
    print_field_1up(buf, 49, 1, CARD8, "min-byte1");
    print_field_1up(buf, 50, 1, CARD8, "max-byte1");
    print_field_1up(buf, 51, 1, BOOL, "all-chars-exist");
    print_field_1up(buf, 52, 2, INT16, "font-ascent");
    print_field_1up(buf, 54, 2, INT16, "font-descent");
    print_field_2up(buf, 56, 4, DVALUE4(m), "number of CHARINFOs");
    if (Verbose >= 3) {
	m = ILong(&buf[56]);
	k = print_list_1up(&buf[60], (long) n, FONTPROP, "properties");
	(void) print_list_1up(&buf[60 + k], (long) m, CHARINFO, "char-infos");
    }
}

/*----------------------------------------------------------------------*/
void QueryTextExtents(buf)
    unsigned char *buf;
{
    short n;

    /* Request QueryTextExtents is opcode 48 */
    print_field_1up(buf, 0, 1, REQUEST, REQUESTHEADER);		/* QueryTextExtents */
    if (Verbose < 1)
	return;
    if (Verbose > 1)
	print_field_1up(SBf, 0, 4, CARD32, "sequence number");

    print_field_2up(buf, 1, 1, BOOL, "odd length?");
    print_field_2up(buf, 2, 2, DVALUE2(2 + (2 * n + p) / 4), "request length");
    n = (IShort(&buf[2]) - 2) * 4 / 2;
    if (IBool(&buf[1]))
	n -= 1;
    print_field_1up(buf, 4, 4, FONTABLE, "font");
    print_str16_1up(&buf[8], (long) n, "string");
}

/*----------------------------------------------------------------------*/
void QueryTextExtentsReply(buf)
    unsigned char *buf;
{
    print_field_1up(RBf, 0, 1, REPLY, REPLYHEADER);	/* QueryTextExtents */
    if (Verbose < 1)
	return;
    print_field_1up(buf, 1, 1, DIRECT, "draw-direction");
    print_field_2up(buf, 2, 2, CARD16, "sequence number");
    print_field_2up(buf, 4, 4, CONST4(0), "reply length");
    print_field_1up(buf, 8, 2, INT16, "font-ascent");
    print_field_1up(buf, 10, 2, INT16, "font-descent");
    print_field_1up(buf, 12, 2, INT16, "overall-ascent");
    print_field_1up(buf, 14, 2, INT16, "overall-descent");
    print_field_1up(buf, 16, 4, INT32, "overall-width");
    print_field_1up(buf, 20, 4, INT32, "overall-left");
    print_field_1up(buf, 24, 4, INT32, "overall-right");
}

/*----------------------------------------------------------------------*/
void ListFonts(buf)
    unsigned char *buf;
{
    short n;

    /* Request ListFonts is opcode 49 */
    print_field_1up(buf, 0, 1, REQUEST, REQUESTHEADER);		/* ListFonts */
    if (Verbose < 1)
	return;
    if (Verbose > 1)
	print_field_1up(SBf, 0, 4, CARD32, "sequence number");

    print_field_2up(buf, 2, 2, DVALUE2(2 + (n + p) / 4), "request length");
    print_field_1up(buf, 4, 2, CARD16, "max-names");
    print_field_2up(buf, 6, 2, DVALUE2(n), "length of pattern");
    n = IShort(&buf[6]);
    print_str8_1up(&buf[8], (long) n, "pattern");
}

/*----------------------------------------------------------------------*/
void ListFontsReply(buf)
    unsigned char *buf;
{
    short n;

    print_field_1up(RBf, 0, 1, REPLY, REPLYHEADER);	/* ListFonts */
    if (Verbose < 1)
	return;
    print_field_2up(buf, 2, 2, CARD16, "sequence number");
    print_field_2up(buf, 4, 4, DVALUE4((n + p) / 4), "reply length");
    print_field_2up(buf, 8, 2, CARD16, "number of names");
    n = IShort(&buf[8]);
    print_list_str_1up(&buf[32], (long) n, "names");
}

/*----------------------------------------------------------------------*/
void ListFontsWithInfo(buf)
    unsigned char *buf;
{
    short n;

    /* Request ListFontsWithInfo is opcode 50 */
    print_field_1up(buf, 0, 1, REQUEST, REQUESTHEADER);		/* ListFontsWithInfo */
    if (Verbose < 1)
	return;
    if (Verbose > 1)
	print_field_1up(SBf, 0, 4, CARD32, "sequence number");

    print_field_2up(buf, 2, 2, DVALUE2(2 + (n + p) / 4), "request length");
    print_field_1up(buf, 4, 2, CARD16, "max-names");
    print_field_2up(buf, 6, 2, DVALUE2(n), "length of pattern");
    n = IShort(&buf[6]);
    print_str8_1up(&buf[8], (long) n, "pattern");
}

/*----------------------------------------------------------------------*/
void ListFontsWithInfoReply(buf)
    unsigned char *buf;
{
    short which;

    print_field_1up(RBf, 0, 1, REPLY, REPLYHEADER);	/* ListFontsWithInfo */
    if (Verbose < 1)
	return;
    which = IByte(&buf[1]);
    if (which != 0) {
	ListFontsWithInfoReply1(buf);
	KeepLastReplyExpected();
    } else
	ListFontsWithInfoReply2(buf);
}

/*----------------------------------------------------------------------*/
void ListFontsWithInfoReply1(buf)
    unsigned char *buf;
{
    short n;
    short m;

    print_field_2up(buf, 1, 1, DVALUE1(n), "length of name in bytes");
    n = IByte(&buf[1]);
    print_field_2up(buf, 2, 2, CARD16, "sequence number");
    print_field_2up(buf, 4, 4, DVALUE4(7 + 2 * m + (n + p) / 4), "reply length");
    print_field_1up(buf, 8, 12, CHARINFO, "min-bounds");
    print_field_1up(buf, 24, 12, CHARINFO, "max-bounds");
    print_field_1up(buf, 40, 2, CARD16, "min-char-or-byte2");
    print_field_1up(buf, 42, 2, CARD16, "max-char-or-byte2");
    print_field_1up(buf, 44, 2, CARD16, "default-char");
    print_field_2up(buf, 46, 2, DVALUE2(m), "number of FONTPROPs");
    m = IShort(&buf[46]);
    print_field_1up(buf, 48, 1, DIRECT, "draw-direction");
    print_field_1up(buf, 49, 1, CARD8, "min-byte1");
    print_field_1up(buf, 50, 1, CARD8, "max-byte1");
    print_field_1up(buf, 51, 1, BOOL, "all-chars-exist");
    print_field_1up(buf, 52, 2, INT16, "font-ascent");
    print_field_1up(buf, 54, 2, INT16, "font-descent");
    print_field_1up(buf, 56, 4, CARD32, "replies-hint");
    (void) print_list_1up(&buf[60], (long) m, FONTPROP, "properties");
    print_str8_1up(&buf[60 + 8 * m], (long) n, "name");
}

/*----------------------------------------------------------------------*/
void ListFontsWithInfoReply2(buf)
    unsigned char *buf;
{
    print_field_1up(buf, 1, 1, CONST1(0), "last-reply indicator");
    print_field_2up(buf, 2, 2, CARD16, "sequence number");
    print_field_2up(buf, 4, 4, CONST4(7), "reply length");
}

/*----------------------------------------------------------------------*/
void SetFontPath(buf)
    unsigned char *buf;
{
    short n;

    /* Request SetFontPath is opcode 51 */
    print_field_1up(buf, 0, 1, REQUEST, REQUESTHEADER);		/* SetFontPath */
    if (Verbose < 1)
	return;
    if (Verbose > 1)
	print_field_1up(SBf, 0, 4, CARD32, "sequence number");

    print_field_2up(buf, 2, 2, DVALUE2(2 + (n + p) / 4), "request length");
    print_field_2up(buf, 4, 2, CARD16, "number of paths");
    n = IShort(&buf[4]);
    print_list_str_1up(&buf[8], (long) n, "paths");
}

/*----------------------------------------------------------------------*/
void GetFontPath(buf)
    unsigned char *buf;
{
    /* Request GetFontPath is opcode 52 */
    print_field_1up(buf, 0, 1, REQUEST, REQUESTHEADER);		/* GetFontPath */
    if (Verbose < 1)
	return;
    if (Verbose > 1)
	print_field_1up(SBf, 0, 4, CARD32, "sequence number");

    print_field_1up(buf, 2, 2, CONST2(1), "request list");
}

/*----------------------------------------------------------------------*/
void GetFontPathReply(buf)
    unsigned char *buf;
{
    short n;

    print_field_1up(RBf, 0, 1, REPLY, REPLYHEADER);	/* GetFontPath */
    if (Verbose < 1)
	return;
    print_field_2up(buf, 2, 2, CARD16, "sequence number");
    print_field_2up(buf, 4, 4, DVALUE4((n + p) / 4), "reply length");
    print_field_2up(buf, 8, 2, CARD16, "number of paths");
    n = IShort(&buf[8]);
    print_list_str_1up(&buf[32], (long) n, "paths");
}

/*----------------------------------------------------------------------*/
void CreatePixmap(buf)
    unsigned char *buf;
{
    /* Request CreatePixmap is opcode 53 */
    print_field_1up(buf, 0, 1, REQUEST, REQUESTHEADER);		/* CreatePixmap */
    if (Verbose < 1)
	return;
    if (Verbose > 1)
	print_field_1up(SBf, 0, 4, CARD32, "sequence number");

    print_field_1up(buf, 1, 1, CARD8, "depth");
    print_field_2up(buf, 2, 2, CONST2(4), "request length");
    print_field_1up(buf, 4, 4, PIXMAP, "pixmap-id");
    print_field_1up(buf, 8, 4, DRAWABLE, "drawable");
    print_field_1up(buf, 12, 2, CARD16, "width");
    print_field_1up(buf, 14, 2, CARD16, "height");
}

/*----------------------------------------------------------------------*/
void FreePixmap(buf)
    unsigned char *buf;
{
    /* Request FreePixmap is opcode 54 */
    print_field_1up(buf, 0, 1, REQUEST, REQUESTHEADER);		/* FreePixmap */
    if (Verbose < 1)
	return;
    if (Verbose > 1)
	print_field_1up(SBf, 0, 4, CARD32, "sequence number");

    print_field_2up(buf, 2, 2, CONST2(2), "request length");
    print_field_1up(buf, 4, 4, PIXMAP, "pixmap");
}

/*----------------------------------------------------------------------*/
void CreateGC(buf)
    unsigned char *buf;
{
    /* Request CreateGC is opcode 55 */
    print_field_1up(buf, 0, 1, REQUEST, REQUESTHEADER);		/* CreateGC */
    if (Verbose < 1)
	return;
    if (Verbose > 1)
	print_field_1up(SBf, 0, 4, CARD32, "sequence number");

    print_field_2up(buf, 2, 2, DVALUE2(4 + n), "request length");
    print_field_1up(buf, 4, 4, GCONTEXT, "graphic-context-id");
    print_field_1up(buf, 8, 4, DRAWABLE, "drawable");
    print_field_1up(buf, 12, 4, GC_BITMASK, "value-mask");
    print_values_1up(&buf[12], 4, GC_BITMASK, &buf[16], "value-list");
}

/*----------------------------------------------------------------------*/
void ChangeGC(buf)
    unsigned char *buf;
{
    /* Request ChangeGC is opcode 56 */
    print_field_1up(buf, 0, 1, REQUEST, REQUESTHEADER);		/* ChangeGC */
    if (Verbose < 1)
	return;
    if (Verbose > 1)
	print_field_1up(SBf, 0, 4, CARD32, "sequence number");

    print_field_2up(buf, 2, 2, DVALUE2(3 + n), "request length");
    print_field_1up(buf, 4, 4, GCONTEXT, "gc");
    print_field_1up(buf, 8, 4, GC_BITMASK, "value-mask");
    print_values_1up(&buf[8], 4, GC_BITMASK, &buf[12], "value-list");
}

/*----------------------------------------------------------------------*/
void CopyGC(buf)
    unsigned char *buf;
{
    /* Request CopyGC is opcode 57 */
    print_field_1up(buf, 0, 1, REQUEST, REQUESTHEADER);		/* CopyGC */
    if (Verbose < 1)
	return;
    if (Verbose > 1)
	print_field_1up(SBf, 0, 4, CARD32, "sequence number");

    print_field_2up(buf, 2, 2, CONST2(4), "request length");
    print_field_1up(buf, 4, 4, GCONTEXT, "src-gc");
    print_field_1up(buf, 8, 4, GCONTEXT, "dst-gc");
    print_field_1up(buf, 12, 4, GC_BITMASK, "value-mask");
}

/*----------------------------------------------------------------------*/
void SetDashes(buf)
    unsigned char *buf;
{
    short n;

    /* Request SetDashes is opcode 58 */
    print_field_1up(buf, 0, 1, REQUEST, REQUESTHEADER);		/* SetDashes */
    if (Verbose < 1)
	return;
    if (Verbose > 1)
	print_field_1up(SBf, 0, 4, CARD32, "sequence number");

    print_field_2up(buf, 2, 2, DVALUE2(3 + (n + p) / 4), "request length");
    print_field_1up(buf, 4, 4, GCONTEXT, "gc");
    print_field_1up(buf, 8, 2, CARD16, "dash-offset");
    print_field_2up(buf, 10, 2, DVALUE2(n), "length of dashes");
    n = IShort(&buf[10]);
    print_bytes_1up(&buf[12], (long) n, "dashes");
}

/*----------------------------------------------------------------------*/
void SetClipRectangles(buf)
    unsigned char *buf;
{
    short n;

    /* Request SetClipRectangles is opcode 59 */
    print_field_1up(buf, 0, 1, REQUEST, REQUESTHEADER);		/* SetClipRectangles */
    if (Verbose < 1)
	return;
    if (Verbose > 1)
	print_field_1up(SBf, 0, 4, CARD32, "sequence number");

    print_field_1up(buf, 1, 1, RECTORDER, "ordering");
    print_field_2up(buf, 2, 2, DVALUE2(3 + 2 * n), "request length");
    n = (IShort(&buf[2]) - 3) / 2;
    print_field_1up(buf, 4, 4, GCONTEXT, "gc");
    print_field_1up(buf, 8, 2, INT16, "clip-x-origin");
    print_field_1up(buf, 10, 2, INT16, "clip-y-origin");
    (void) print_list_1up(&buf[12], (long) n, RECTANGLE, "rectangles");
}

/*----------------------------------------------------------------------*/
void FreeGC(buf)
    unsigned char *buf;
{
    /* Request FreeGC is opcode 60 */
    print_field_1up(buf, 0, 1, REQUEST, REQUESTHEADER);		/* FreeGC */
    if (Verbose < 1)
	return;
    if (Verbose > 1)
	print_field_1up(SBf, 0, 4, CARD32, "sequence number");

    print_field_2up(buf, 2, 2, CONST2(2), "request length");
    print_field_1up(buf, 4, 4, GCONTEXT, "gc");
}

/*----------------------------------------------------------------------*/
void ClearArea(buf)
    unsigned char *buf;
{
    /* Request ClearArea is opcode 61 */
    print_field_1up(buf, 0, 1, REQUEST, REQUESTHEADER);		/* ClearArea */
    if (Verbose < 1)
	return;
    if (Verbose > 1)
	print_field_1up(SBf, 0, 4, CARD32, "sequence number");

    print_field_1up(buf, 1, 1, BOOL, "exposures");
    print_field_2up(buf, 2, 2, CONST2(4), "request length");
    print_field_1up(buf, 4, 4, WINDOW, "window");
    print_field_1up(buf, 8, 2, INT16, "x");
    print_field_1up(buf, 10, 2, INT16, "y");
    print_field_1up(buf, 12, 2, CARD16, "width");
    print_field_1up(buf, 14, 2, CARD16, "height");
}

/*----------------------------------------------------------------------*/
void CopyArea(buf)
    unsigned char *buf;
{
    /* Request CopyArea is opcode 62 */
    print_field_1up(buf, 0, 1, REQUEST, REQUESTHEADER);		/* CopyArea */
    if (Verbose < 1)
	return;
    if (Verbose > 1)
	print_field_1up(SBf, 0, 4, CARD32, "sequence number");

    print_field_2up(buf, 2, 2, CONST2(7), "request length");
    print_field_1up(buf, 4, 4, DRAWABLE, "src-drawable");
    print_field_1up(buf, 8, 4, DRAWABLE, "dst-drawable");
    print_field_1up(buf, 12, 4, GCONTEXT, "gc");
    print_field_1up(buf, 16, 2, INT16, "src-x");
    print_field_1up(buf, 18, 2, INT16, "src-y");
    print_field_1up(buf, 20, 2, INT16, "dst-x");
    print_field_1up(buf, 22, 2, INT16, "dst-y");
    print_field_1up(buf, 24, 2, CARD16, "width");
    print_field_1up(buf, 26, 2, CARD16, "height");
}

/*----------------------------------------------------------------------*/
void CopyPlane(buf)
    unsigned char *buf;
{
    /* Request CopyPlane is opcode 63 */
    print_field_1up(buf, 0, 1, REQUEST, REQUESTHEADER);		/* CopyPlane */
    if (Verbose < 1)
	return;
    if (Verbose > 1)
	print_field_1up(SBf, 0, 4, CARD32, "sequence number");

    print_field_2up(buf, 2, 2, CONST2(8), "request length");
    print_field_1up(buf, 4, 4, DRAWABLE, "src-drawable");
    print_field_1up(buf, 8, 4, DRAWABLE, "dst-drawable");
    print_field_1up(buf, 12, 4, GCONTEXT, "gc");
    print_field_1up(buf, 16, 2, INT16, "src-x");
    print_field_1up(buf, 18, 2, INT16, "src-y");
    print_field_1up(buf, 20, 2, INT16, "dst-x");
    print_field_1up(buf, 22, 2, INT16, "dst-y");
    print_field_1up(buf, 24, 2, CARD16, "width");
    print_field_1up(buf, 26, 2, CARD16, "height");
    print_field_1up(buf, 28, 4, CARD32, "bit-plane");
}

/*----------------------------------------------------------------------*/
void PolyPoint(buf)
    unsigned char *buf;
{
    short n;

    /* Request PolyPoint is opcode 64 */
    print_field_1up(buf, 0, 1, REQUEST, REQUESTHEADER);		/* PolyPoint */
    if (Verbose < 1)
	return;
    if (Verbose > 1)
	print_field_1up(SBf, 0, 4, CARD32, "sequence number");

    print_field_1up(buf, 1, 1, COORMODE, "coordinate-mode");
    print_field_2up(buf, 2, 2, DVALUE2(3 + n), "request length");
    n = (IShort(&buf[2]) - 3);
    print_field_1up(buf, 4, 4, DRAWABLE, "drawable");
    print_field_1up(buf, 8, 4, GCONTEXT, "gc");
    (void) print_list_1up(&buf[12], (long) n, POINT, "points");
}

/*----------------------------------------------------------------------*/
void PolyLine(buf)
    unsigned char *buf;
{
    short n;

    /* Request PolyLine is opcode 65 */
    print_field_1up(buf, 0, 1, REQUEST, REQUESTHEADER);		/* PolyLine */
    if (Verbose < 1)
	return;
    if (Verbose > 1)
	print_field_1up(SBf, 0, 4, CARD32, "sequence number");

    print_field_1up(buf, 1, 1, COORMODE, "coordinate-mode");
    print_field_2up(buf, 2, 2, DVALUE2(3 + n), "request length");
    n = (IShort(&buf[2]) - 3);
    print_field_1up(buf, 4, 4, DRAWABLE, "drawable");
    print_field_1up(buf, 8, 4, GCONTEXT, "gc");
    (void) print_list_1up(&buf[12], (long) n, POINT, "points");
}

/*----------------------------------------------------------------------*/
void PolySegment(buf)
    unsigned char *buf;
{
    short n;

    /* Request PolySegment is opcode 66 */
    print_field_1up(buf, 0, 1, REQUEST, REQUESTHEADER);		/* PolySegment */
    if (Verbose < 1)
	return;
    if (Verbose > 1)
	print_field_1up(SBf, 0, 4, CARD32, "sequence number");

    print_field_2up(buf, 2, 2, DVALUE2(3 + 2 * n), "request length");
    n = (IShort(&buf[2]) - 3) / 2;
    print_field_1up(buf, 4, 4, DRAWABLE, "drawable");
    print_field_1up(buf, 8, 4, GCONTEXT, "gc");
    (void) print_list_1up(&buf[12], (long) n, SEGMENT, "segments");
}

/*----------------------------------------------------------------------*/
void PolyRectangle(buf)
    unsigned char *buf;
{
    short n;

    /* Request PolyRectangle is opcode 67 */
    print_field_1up(buf, 0, 1, REQUEST, REQUESTHEADER);		/* PolyRectangle */
    if (Verbose < 1)
	return;
    if (Verbose > 1)
	print_field_1up(SBf, 0, 4, CARD32, "sequence number");

    print_field_2up(buf, 2, 2, DVALUE2(3 + 2 * n), "request length");
    n = (IShort(&buf[2]) - 3) / 2;
    print_field_1up(buf, 4, 4, DRAWABLE, "drawable");
    print_field_1up(buf, 8, 4, GCONTEXT, "gc");
    (void) print_list_1up(&buf[12], (long) n, RECTANGLE, "rectangles");
}

/*----------------------------------------------------------------------*/
void PolyArc(buf)
    unsigned char *buf;
{
    short n;

    /* Request PolyArc is opcode 68 */
    print_field_1up(buf, 0, 1, REQUEST, REQUESTHEADER);		/* PolyArc */
    if (Verbose < 1)
	return;
    if (Verbose > 1)
	print_field_1up(SBf, 0, 4, CARD32, "sequence number");

    print_field_2up(buf, 2, 2, DVALUE2(3 + 3 * n), "request length");
    n = (IShort(&buf[2]) - 3) / 3;
    print_field_1up(buf, 4, 4, DRAWABLE, "drawable");
    print_field_1up(buf, 8, 4, GCONTEXT, "gc");
    (void) print_list_1up(&buf[12], (long) n, ARC, "arcs");
}

/*----------------------------------------------------------------------*/
void FillPoly(buf)
    unsigned char *buf;
{
    short n;

    /* Request FillPoly is opcode 69 */
    print_field_1up(buf, 0, 1, REQUEST, REQUESTHEADER);		/* FillPoly */
    if (Verbose < 1)
	return;
    if (Verbose > 1)
	print_field_1up(SBf, 0, 4, CARD32, "sequence number");

    print_field_2up(buf, 2, 2, DVALUE2(4 + n), "request length");
    n = (IShort(&buf[2]) - 4);
    print_field_1up(buf, 4, 4, DRAWABLE, "drawable");
    print_field_1up(buf, 8, 4, GCONTEXT, "gc");
    print_field_1up(buf, 12, 1, POLYSHAPE, "shape");
    print_field_1up(buf, 13, 1, COORMODE, "coordinate-mode");
    (void) print_list_1up(&buf[16], (long) n, POINT, "points");
}

/*----------------------------------------------------------------------*/
void PolyFillRectangle(buf)
    unsigned char *buf;
{
    short n;

    /* Request PolyFillRectangle is opcode 70 */
    print_field_1up(buf, 0, 1, REQUEST, REQUESTHEADER);		/* PolyFillRectangle */
    if (Verbose < 1)
	return;
    if (Verbose > 1)
	print_field_1up(SBf, 0, 4, CARD32, "sequence number");

    print_field_2up(buf, 2, 2, DVALUE2(3 + 2 * n), "request length");
    n = (IShort(&buf[2]) - 3) / 2;
    print_field_1up(buf, 4, 4, DRAWABLE, "drawable");
    print_field_1up(buf, 8, 4, GCONTEXT, "gc");
    (void) print_list_1up(&buf[12], (long) n, RECTANGLE, "rectangles");
}

/*----------------------------------------------------------------------*/
void PolyFillArc(buf)
    unsigned char *buf;
{
    short n;

    /* Request PolyFillArc is opcode 71 */
    print_field_1up(buf, 0, 1, REQUEST, REQUESTHEADER);		/* PolyFillArc */
    if (Verbose < 1)
	return;
    if (Verbose > 1)
	print_field_1up(SBf, 0, 4, CARD32, "sequence number");

    print_field_2up(buf, 2, 2, DVALUE2(3 + 3 * n), "request length");
    n = (IShort(&buf[2]) - 3) / 3;
    print_field_1up(buf, 4, 4, DRAWABLE, "drawable");
    print_field_1up(buf, 8, 4, GCONTEXT, "gc");
    (void) print_list_1up(&buf[12], (long) n, ARC, "arcs");
}

/*----------------------------------------------------------------------*/
void PutImage(buf)
    unsigned char *buf;
{
    short n;

    /* Request PutImage is opcode 72 */
    print_field_1up(buf, 0, 1, REQUEST, REQUESTHEADER);		/* PutImage */
    if (Verbose < 1)
	return;
    if (Verbose > 1)
	print_field_1up(SBf, 0, 4, CARD32, "sequence number");

    print_field_1up(buf, 1, 1, IMAGEMODE, "format");
    print_field_2up(buf, 2, 2, DVALUE2(6 + (n + p) / 4), "request length");

    /* 
     * The size of the Image is overestimated by the following computation of n,
     * because we ignore that padding of the request to a multiple of 4 bytes.
     * The image may not be a multiple of 4 bytes.  The actual size of the image
     * is determined as follows: for format = Bitmap or format = XYPixmap, the
     * size is (left-pad + width) [ pad to multiple of bitmap-scanline-pad from
     * SetUpReply ] divide by 8 to get bytes times height times depth for format
     * = ZPixmap, take the depth and use it to find the bits-per-pixel and
     * scanline-pad given in one of the SetUpReply DEPTH records. width *
     * bits-per-pixel pad to multiple of scanline-pad divide by 8 to get bytes
     * times height times depth For simplicity, we ignore all this and just use
     * the request length to (over)estimate the size of the image 
     */

    n = (IShort(&buf[2]) - 6) * 4;
    print_field_1up(buf, 4, 4, DRAWABLE, "drawable");
    print_field_1up(buf, 8, 4, GCONTEXT, "gc");
    print_field_1up(buf, 12, 2, CARD16, "width");
    print_field_1up(buf, 14, 2, CARD16, "height");
    print_field_1up(buf, 16, 2, INT16, "dst-x");
    print_field_1up(buf, 18, 2, INT16, "dst-y");
    print_field_1up(buf, 20, 1, CARD8, "left-pad");
    print_field_1up(buf, 21, 1, CARD8, "depth");
    print_bytes_1up(&buf[24], (long) n, "data");
}

/*----------------------------------------------------------------------*/
void GetImage(buf)
    unsigned char *buf;
{
    /* Request GetImage is opcode 73 */
    print_field_1up(buf, 0, 1, REQUEST, REQUESTHEADER);		/* GetImage */
    if (Verbose < 1)
	return;
    if (Verbose > 1)
	print_field_1up(SBf, 0, 4, CARD32, "sequence number");

    print_field_1up(buf, 1, 1, IMAGEMODE, "format");
    print_field_2up(buf, 2, 2, CONST2(5), "request length");
    print_field_1up(buf, 4, 4, DRAWABLE, "drawable");
    print_field_1up(buf, 8, 2, INT16, "x");
    print_field_1up(buf, 10, 2, INT16, "y");
    print_field_1up(buf, 12, 2, CARD16, "width");
    print_field_1up(buf, 14, 2, CARD16, "height");
    print_field_1up(buf, 16, 4, CARD32, "plane-mask");
}

/*----------------------------------------------------------------------*/
void GetImageReply(buf)
    unsigned char *buf;
{
    long n;

    print_field_1up(RBf, 0, 1, REPLY, REPLYHEADER);	/* GetImage */
    if (Verbose < 1)
	return;
    print_field_1up(buf, 1, 1, CARD8, "depth");
    print_field_2up(buf, 2, 2, CARD16, "sequence number");
    print_field_2up(buf, 4, 4, DVALUE4((n + p) / 4), "reply length");

    /* to properly compute the actual size of the image, we have to remember the
       width and height values from the request.  Again, we (over)estimate its
       length from the length of the reply */
    n = ILong(&buf[4]) * 4;
    print_field_1up(buf, 8, 4, VISUALID, "visual");
    print_bytes_1up(&buf[32], n, "data");
}

/*----------------------------------------------------------------------*/
void PolyText8(buf)
    unsigned char *buf;
{
    short n;

    /* Request PolyText8 is opcode 74 */
    print_field_1up(buf, 0, 1, REQUEST, REQUESTHEADER);		/* PolyText8 */
    if (Verbose < 1)
	return;
    if (Verbose > 1)
	print_field_1up(SBf, 0, 4, CARD32, "sequence number");

    print_field_2up(buf, 2, 2, DVALUE2(4 + (n + p) / 4), "request length");
    n = (IShort(&buf[2]) - 4) * 4;
    print_field_1up(buf, 4, 4, DRAWABLE, "drawable");
    print_field_1up(buf, 8, 4, GCONTEXT, "gc");
    print_field_1up(buf, 12, 2, INT16, "x");
    print_field_1up(buf, 14, 2, INT16, "y");
    print_text_list8_1up(&buf[16], n, "items");
}

/*----------------------------------------------------------------------*/
void PolyText16(buf)
    unsigned char *buf;
{
    short n;

    /* Request PolyText16 is opcode 75 */
    print_field_1up(buf, 0, 1, REQUEST, REQUESTHEADER);		/* PolyText16 */
    if (Verbose < 1)
	return;
    if (Verbose > 1)
	print_field_1up(SBf, 0, 4, CARD32, "sequence number");

    print_field_2up(buf, 2, 2, DVALUE2(4 + (n + p) / 4), "request length");
    n = (IShort(&buf[2]) - 4) * 4;
    print_field_1up(buf, 4, 4, DRAWABLE, "drawable");
    print_field_1up(buf, 8, 4, GCONTEXT, "gc");
    print_field_1up(buf, 12, 2, INT16, "x");
    print_field_1up(buf, 14, 2, INT16, "y");
    print_text_list16_1up(&buf[16], n, "items");
}

/*----------------------------------------------------------------------*/
void ImageText8(buf)
    unsigned char *buf;
{
    short n;

    /* Request ImageText8 is opcode 76 */
    print_field_1up(buf, 0, 1, REQUEST, REQUESTHEADER);		/* ImageText8 */
    if (Verbose < 1)
	return;
    if (Verbose > 1)
	print_field_1up(SBf, 0, 4, CARD32, "sequence number");

    print_field_2up(buf, 1, 1, DVALUE1(n), "length of string");
    n = IByte(&buf[1]);
    print_field_2up(buf, 2, 2, DVALUE2(4 + (n + p) / 4), "request length");
    print_field_1up(buf, 4, 4, DRAWABLE, "drawable");
    print_field_1up(buf, 8, 4, GCONTEXT, "gc");
    print_field_1up(buf, 12, 2, INT16, "x");
    print_field_1up(buf, 14, 2, INT16, "y");
    print_str8_1up(&buf[16], (long) n, "string");
}

/*----------------------------------------------------------------------*/
void ImageText16(buf)
    unsigned char *buf;
{
    short n;

    /* Request ImageText16 is opcode 77 */
    print_field_1up(buf, 0, 1, REQUEST, REQUESTHEADER);		/* ImageText16 */
    if (Verbose < 1)
	return;
    if (Verbose > 1)
	print_field_1up(SBf, 0, 4, CARD32, "sequence number");

    print_field_2up(buf, 1, 1, DVALUE1(n), "length of string");
    n = IByte(&buf[1]);
    print_field_2up(buf, 2, 2, DVALUE2(4 + (2 * n + p) / 4), "request length");
    print_field_1up(buf, 4, 4, DRAWABLE, "drawable");
    print_field_1up(buf, 8, 4, GCONTEXT, "gc");
    print_field_1up(buf, 12, 2, INT16, "x");
    print_field_1up(buf, 14, 2, INT16, "y");
    print_str16_1up(&buf[16], (long) n, "string");
}

/*----------------------------------------------------------------------*/
void CreateColormap(buf)
    unsigned char *buf;
{
    /* Request CreateColormap is opcode 78 */
    print_field_1up(buf, 0, 1, REQUEST, REQUESTHEADER);		/* CreateColormap */
    if (Verbose < 1)
	return;
    if (Verbose > 1)
	print_field_1up(SBf, 0, 4, CARD32, "sequence number");

    print_field_1up(buf, 1, 1, ALLORNONE, "alloc");
    print_field_2up(buf, 2, 2, CONST2(4), "request length");
    print_field_1up(buf, 4, 4, COLORMAP, "color-map-id");
    print_field_1up(buf, 8, 4, WINDOW, "window");
    print_field_1up(buf, 12, 4, VISUALID, "visual");
}

/*----------------------------------------------------------------------*/
void FreeColormap(buf)
    unsigned char *buf;
{
    /* Request FreeColormap is opcode 79 */
    print_field_1up(buf, 0, 1, REQUEST, REQUESTHEADER);		/* FreeColormap */
    if (Verbose < 1)
	return;
    if (Verbose > 1)
	print_field_1up(SBf, 0, 4, CARD32, "sequence number");

    print_field_2up(buf, 2, 2, CONST2(2), "request length");
    print_field_1up(buf, 4, 4, COLORMAP, "cmap");
}

/*----------------------------------------------------------------------*/
void CopyColormapAndFree(buf)
    unsigned char *buf;
{
    /* Request CopyColormapAndFree is opcode 80 */
    print_field_1up(buf, 0, 1, REQUEST, REQUESTHEADER);		/* CopyColormapAndFree */
    if (Verbose < 1)
	return;
    if (Verbose > 1)
	print_field_1up(SBf, 0, 4, CARD32, "sequence number");

    print_field_2up(buf, 2, 2, CONST2(3), "request length");
    print_field_1up(buf, 4, 4, COLORMAP, "color-map-id");
    print_field_1up(buf, 8, 4, COLORMAP, "src-cmap");
}

/*----------------------------------------------------------------------*/
void InstallColormap(buf)
    unsigned char *buf;
{
    /* Request InstallColormap is opcode 81 */
    print_field_1up(buf, 0, 1, REQUEST, REQUESTHEADER);		/* InstallColormap */
    if (Verbose < 1)
	return;
    if (Verbose > 1)
	print_field_1up(SBf, 0, 4, CARD32, "sequence number");

    print_field_2up(buf, 2, 2, CONST2(2), "request length");
    print_field_1up(buf, 4, 4, COLORMAP, "cmap");
}

/*----------------------------------------------------------------------*/
void UninstallColormap(buf)
    unsigned char *buf;
{
    /* Request UninstallColormap is opcode 82 */
    print_field_1up(buf, 0, 1, REQUEST, REQUESTHEADER);		/* UninstallColormap */
    if (Verbose < 1)
	return;
    if (Verbose > 1)
	print_field_1up(SBf, 0, 4, CARD32, "sequence number");

    print_field_2up(buf, 2, 2, CONST2(2), "request length");
    print_field_1up(buf, 4, 4, COLORMAP, "cmap");
}

/*----------------------------------------------------------------------*/
void ListInstalledColormaps(buf)
    unsigned char *buf;
{
    /* Request ListInstalledColormaps is opcode 83 */
    print_field_1up(buf, 0, 1, REQUEST, REQUESTHEADER);		/* ListInstalledColormaps */
    if (Verbose < 1)
	return;
    if (Verbose > 1)
	print_field_1up(SBf, 0, 4, CARD32, "sequence number");

    print_field_2up(buf, 2, 2, CONST2(2), "request length");
    print_field_1up(buf, 4, 4, WINDOW, "window");
}

/*----------------------------------------------------------------------*/
void ListInstalledColormapsReply(buf)
    unsigned char *buf;
{
    short n;

    print_field_1up(RBf, 0, 1, REPLY, REPLYHEADER);	/* ListInstalledColormaps */
    if (Verbose < 1)
	return;
    print_field_2up(buf, 2, 2, CARD16, "sequence number");
    print_field_2up(buf, 4, 4, DVALUE4(n), "reply length");
    print_field_2up(buf, 8, 2, DVALUE2(n), "number of cmaps");
    n = IShort(&buf[8]);
    (void) print_list_1up(&buf[32], (long) n, COLORMAP, "cmaps");
}

/*----------------------------------------------------------------------*/
void AllocColor(buf)
    unsigned char *buf;
{
    /* Request AllocColor is opcode 84 */
    print_field_1up(buf, 0, 1, REQUEST, REQUESTHEADER);		/* AllocColor */
    if (Verbose < 1)
	return;
    if (Verbose > 1)
	print_field_1up(SBf, 0, 4, CARD32, "sequence number");

    print_field_2up(buf, 2, 2, CONST2(4), "request length");
    print_field_1up(buf, 4, 4, COLORMAP, "cmap");
    print_field_1up(buf, 8, 2, CARD16, "red");
    print_field_1up(buf, 10, 2, CARD16, "green");
    print_field_1up(buf, 12, 2, CARD16, "blue");
}

/*----------------------------------------------------------------------*/
void AllocColorReply(buf)
    unsigned char *buf;
{
    print_field_1up(RBf, 0, 1, REPLY, REPLYHEADER);	/* AllocColor */
    if (Verbose < 1)
	return;
    print_field_2up(buf, 2, 2, CARD16, "sequence number");
    print_field_2up(buf, 4, 4, CONST4(0), "reply length");
    print_field_1up(buf, 8, 2, CARD16, "red");
    print_field_1up(buf, 10, 2, CARD16, "green");
    print_field_1up(buf, 12, 2, CARD16, "blue");
    print_field_1up(buf, 16, 4, CARD32, "pixel");
}

/*----------------------------------------------------------------------*/
void AllocNamedColor(buf)
    unsigned char *buf;
{
    short n;

    /* Request AllocNamedColor is opcode 85 */
    print_field_1up(buf, 0, 1, REQUEST, REQUESTHEADER);		/* AllocNamedColor */
    if (Verbose < 1)
	return;
    if (Verbose > 1)
	print_field_1up(SBf, 0, 4, CARD32, "sequence number");

    print_field_2up(buf, 2, 2, DVALUE2(3 + (n + p) / 4), "request length");
    print_field_1up(buf, 4, 4, COLORMAP, "cmap");
    print_field_2up(buf, 8, 2, DVALUE2(n), "length of name");
    n = IShort(&buf[8]);
    print_str8_1up(&buf[12], (long) n, "name");
}

/*----------------------------------------------------------------------*/
void AllocNamedColorReply(buf)
    unsigned char *buf;
{
    print_field_1up(RBf, 0, 1, REPLY, REPLYHEADER);	/* AllocNamedColor */
    if (Verbose < 1)
	return;
    print_field_2up(buf, 2, 2, CARD16, "sequence number");
    print_field_2up(buf, 4, 4, CONST4(0), "reply length");
    print_field_1up(buf, 8, 4, CARD32, "pixel");
    print_field_1up(buf, 12, 2, CARD16, "exact-red");
    print_field_1up(buf, 14, 2, CARD16, "exact-green");
    print_field_1up(buf, 16, 2, CARD16, "exact-blue");
    print_field_1up(buf, 18, 2, CARD16, "visual-red");
    print_field_1up(buf, 20, 2, CARD16, "visual-green");
    print_field_1up(buf, 22, 2, CARD16, "visual-blue");
}

/*----------------------------------------------------------------------*/
void AllocColorCells(buf)
    unsigned char *buf;
{
    /* Request AllocColorCells is opcode 86 */
    print_field_1up(buf, 0, 1, REQUEST, REQUESTHEADER);		/* AllocColorCells */
    if (Verbose < 1)
	return;
    if (Verbose > 1)
	print_field_1up(SBf, 0, 4, CARD32, "sequence number");

    print_field_1up(buf, 1, 1, BOOL, "contiguous");
    print_field_2up(buf, 2, 2, CONST2(3), "request length");
    print_field_1up(buf, 4, 4, COLORMAP, "cmap");
    print_field_1up(buf, 8, 2, CARD16, "colors");
    print_field_1up(buf, 10, 2, CARD16, "planes");
}

/*----------------------------------------------------------------------*/
void AllocColorCellsReply(buf)
    unsigned char *buf;
{
    short n;
    short m;
    short k;

    print_field_1up(RBf, 0, 1, REPLY, REPLYHEADER);	/* AllocColorCells */
    if (Verbose < 1)
	return;
    print_field_2up(buf, 2, 2, CARD16, "sequence number");
    print_field_2up(buf, 4, 4, DVALUE4(n + m), "reply length");
    print_field_2up(buf, 8, 2, DVALUE2(n), "number of pixels");
    n = IShort(&buf[8]);
    print_field_2up(buf, 10, 2, DVALUE2(m), "number of masks");
    m = IShort(&buf[10]);
    k = print_list_1up(&buf[32], (long) n, CARD32, "pixels");
    (void) print_list_1up(&buf[32 + k], (long) m, CARD32, "masks");
}

/*----------------------------------------------------------------------*/
void AllocColorPlanes(buf)
    unsigned char *buf;
{
    /* Request AllocColorPlanes is opcode 87 */
    print_field_1up(buf, 0, 1, REQUEST, REQUESTHEADER);		/* AllocColorPlanes */
    if (Verbose < 1)
	return;
    if (Verbose > 1)
	print_field_1up(SBf, 0, 4, CARD32, "sequence number");

    print_field_1up(buf, 1, 1, BOOL, "contiguous");
    print_field_2up(buf, 2, 2, CONST2(4), "request length");
    print_field_1up(buf, 4, 4, COLORMAP, "cmap");
    print_field_1up(buf, 8, 2, CARD16, "colors");
    print_field_1up(buf, 10, 2, CARD16, "reds");
    print_field_1up(buf, 12, 2, CARD16, "greens");
    print_field_1up(buf, 14, 2, CARD16, "blues");
}

/*----------------------------------------------------------------------*/
void AllocColorPlanesReply(buf)
    unsigned char *buf;
{
    short n;

    print_field_1up(RBf, 0, 1, REPLY, REPLYHEADER);	/* AllocColorPlanes */
    if (Verbose < 1)
	return;
    print_field_2up(buf, 2, 2, CARD16, "sequence number");
    print_field_2up(buf, 4, 4, DVALUE4(n), "reply length");
    print_field_2up(buf, 8, 2, DVALUE2(n), "number of pixels");
    n = IShort(&buf[8]);
    print_field_1up(buf, 12, 4, CARD32, "red-mask");
    print_field_1up(buf, 16, 4, CARD32, "green-mask");
    print_field_1up(buf, 20, 4, CARD32, "blue-mask");
    (void) print_list_1up(&buf[32], (long) n, CARD32, "pixels");
}

/*----------------------------------------------------------------------*/
void FreeColors(buf)
    unsigned char *buf;
{
    short n;

    /* Request FreeColors is opcode 88 */
    print_field_1up(buf, 0, 1, REQUEST, REQUESTHEADER);		/* FreeColors */
    if (Verbose < 1)
	return;
    if (Verbose > 1)
	print_field_1up(SBf, 0, 4, CARD32, "sequence number");

    print_field_2up(buf, 2, 2, DVALUE2(3 + n), "request length");
    n = IShort(&buf[2]) - 3;
    print_field_1up(buf, 4, 4, COLORMAP, "cmap");
    print_field_1up(buf, 8, 4, CARD32, "plane-mask");
    (void) print_list_1up(&buf[12], (long) n, CARD32, "pixels");
}

/*----------------------------------------------------------------------*/
void StoreColors(buf)
    unsigned char *buf;
{
    short n;

    /* Request StoreColors is opcode 89 */
    print_field_1up(buf, 0, 1, REQUEST, REQUESTHEADER);		/* StoreColors */
    if (Verbose < 1)
	return;
    if (Verbose > 1)
	print_field_1up(SBf, 0, 4, CARD32, "sequence number");

    print_field_2up(buf, 2, 2, DVALUE2(2 + 3 * n), "request length");
    n = (IShort(&buf[2]) - 2) / 3;
    print_field_1up(buf, 4, 4, COLORMAP, "cmap");
    (void) print_list_1up(&buf[8], (long) n, COLORITEM, "items");
}

/*----------------------------------------------------------------------*/
void StoreNamedColor(buf)
    unsigned char *buf;
{
    short n;

    /* Request StoreNamedColor is opcode 90 */
    print_field_1up(buf, 0, 1, REQUEST, REQUESTHEADER);		/* StoreNamedColor */
    if (Verbose < 1)
	return;
    if (Verbose > 1)
	print_field_1up(SBf, 0, 4, CARD32, "sequence number");

    print_field_1up(buf, 1, 1, COLORMASK, "which colors?");
    print_field_2up(buf, 2, 2, DVALUE2(4 + (n + p) / 4), "request length");
    print_field_1up(buf, 4, 4, COLORMAP, "cmap");
    print_field_1up(buf, 8, 4, CARD32, "pixel");
    print_field_2up(buf, 12, 2, DVALUE2(n), "length of name");
    n = IShort(&buf[12]);
    print_str8_1up(&buf[16], (long) n, "name");
}

/*----------------------------------------------------------------------*/
void QueryColors(buf)
    unsigned char *buf;
{
    short n;

    /* Request QueryColors is opcode 91 */
    print_field_1up(buf, 0, 1, REQUEST, REQUESTHEADER);		/* QueryColors */
    if (Verbose < 1)
	return;
    if (Verbose > 1)
	print_field_1up(SBf, 0, 4, CARD32, "sequence number");

    print_field_2up(buf, 2, 2, DVALUE2(2 + n), "request length");
    n = IShort(&buf[2]) - 2;
    print_field_1up(buf, 4, 4, COLORMAP, "cmap");
    (void) print_list_1up(&buf[8], (long) n, CARD32, "pixels");
}

/*----------------------------------------------------------------------*/
void QueryColorsReply(buf)
    unsigned char *buf;
{
    short n;

    print_field_1up(RBf, 0, 1, REPLY, REPLYHEADER);	/* QueryColors */
    if (Verbose < 1)
	return;
    print_field_2up(buf, 2, 2, CARD16, "sequence number");
    print_field_2up(buf, 4, 4, DVALUE4(2 * n), "reply length");
    print_field_2up(buf, 8, 2, DVALUE2(n), "number of colors");
    n = IShort(&buf[8]);
    (void) print_list_1up(&buf[32], (long) n, RGB, "colors");
}

/*----------------------------------------------------------------------*/
void LookupColor(buf)
    unsigned char *buf;
{
    short n;

    /* Request LookupColor is opcode 92 */
    print_field_1up(buf, 0, 1, REQUEST, REQUESTHEADER);		/* LookupColor */
    if (Verbose < 1)
	return;
    if (Verbose > 1)
	print_field_1up(SBf, 0, 4, CARD32, "sequence number");

    print_field_2up(buf, 2, 2, DVALUE2(3 + (n + p) / 4), "request length");
    print_field_1up(buf, 4, 4, COLORMAP, "cmap");
    print_field_2up(buf, 8, 2, DVALUE2(n), "length of name");
    n = IShort(&buf[8]);
    print_str8_1up(&buf[12], (long) n, "name");
}

/*----------------------------------------------------------------------*/
void LookupColorReply(buf)
    unsigned char *buf;
{
    print_field_1up(RBf, 0, 1, REPLY, REPLYHEADER);	/* LookupColor */
    if (Verbose < 1)
	return;
    print_field_2up(buf, 2, 2, CARD16, "sequence number");
    print_field_2up(buf, 4, 4, CONST4(0), "reply length");
    print_field_1up(buf, 8, 2, CARD16, "exact-red");
    print_field_1up(buf, 10, 2, CARD16, "exact-green");
    print_field_1up(buf, 12, 2, CARD16, "exact-blue");
    print_field_1up(buf, 14, 2, CARD16, "visual-red");
    print_field_1up(buf, 16, 2, CARD16, "visual-green");
    print_field_1up(buf, 18, 2, CARD16, "visual-blue");
}

/*----------------------------------------------------------------------*/
void CreateCursor(buf)
    unsigned char *buf;
{
    /* Request CreateCursor is opcode 93 */
    print_field_1up(buf, 0, 1, REQUEST, REQUESTHEADER);		/* CreateCursor */
    if (Verbose < 1)
	return;
    if (Verbose > 1)
	print_field_1up(SBf, 0, 4, CARD32, "sequence number");

    print_field_2up(buf, 2, 2, CONST2(8), "request length");
    print_field_1up(buf, 4, 4, CURSOR, "cursor-id");
    print_field_1up(buf, 8, 4, PIXMAP, "source");
    print_field_1up(buf, 12, 4, PIXMAP, "mask");
    print_field_1up(buf, 16, 2, CARD16, "fore-red");
    print_field_1up(buf, 18, 2, CARD16, "fore-green");
    print_field_1up(buf, 20, 2, CARD16, "fore-blue");
    print_field_1up(buf, 22, 2, CARD16, "back-red");
    print_field_1up(buf, 24, 2, CARD16, "back-green");
    print_field_1up(buf, 26, 2, CARD16, "back-blue");
    print_field_1up(buf, 28, 2, CARD16, "x");
    print_field_1up(buf, 30, 2, CARD16, "y");
}

/*----------------------------------------------------------------------*/
void CreateGlyphCursor(buf)
    unsigned char *buf;
{
    /* Request CreateGlyphCursor is opcode 94 */
    print_field_1up(buf, 0, 1, REQUEST, REQUESTHEADER);		/* CreateGlyphCursor */
    if (Verbose < 1)
	return;
    if (Verbose > 1)
	print_field_1up(SBf, 0, 4, CARD32, "sequence number");

    print_field_2up(buf, 2, 2, CONST2(8), "request length");
    print_field_1up(buf, 4, 4, CURSOR, "cursor-id");
    print_field_1up(buf, 8, 4, FONT, "source-font");
    print_field_1up(buf, 12, 4, FONT, "mask-font");
    print_field_1up(buf, 16, 2, CARD16, "source-char");
    print_field_1up(buf, 18, 2, CARD16, "mask-char");
    print_field_1up(buf, 20, 2, CARD16, "fore-red");
    print_field_1up(buf, 22, 2, CARD16, "fore-green");
    print_field_1up(buf, 24, 2, CARD16, "fore-blue");
    print_field_1up(buf, 26, 2, CARD16, "back-red");
    print_field_1up(buf, 28, 2, CARD16, "back-green");
    print_field_1up(buf, 30, 2, CARD16, "back-blue");
}

/*----------------------------------------------------------------------*/
void FreeCursor(buf)
    unsigned char *buf;
{
    /* Request FreeCursor is opcode 95 */
    print_field_1up(buf, 0, 1, REQUEST, REQUESTHEADER);		/* FreeCursor */
    if (Verbose < 1)
	return;
    if (Verbose > 1)
	print_field_1up(SBf, 0, 4, CARD32, "sequence number");

    print_field_2up(buf, 2, 2, CONST2(2), "request length");
    print_field_1up(buf, 4, 4, CURSOR, "cursor");
}

/*----------------------------------------------------------------------*/
void RecolorCursor(buf)
    unsigned char *buf;
{
    /* Request RecolorCursor is opcode 96 */
    print_field_1up(buf, 0, 1, REQUEST, REQUESTHEADER);		/* RecolorCursor */
    if (Verbose < 1)
	return;
    if (Verbose > 1)
	print_field_1up(SBf, 0, 4, CARD32, "sequence number");

    print_field_2up(buf, 2, 2, CONST2(5), "request length");
    print_field_1up(buf, 4, 4, CURSOR, "cursor");
    print_field_1up(buf, 8, 2, CARD16, "fore-red");
    print_field_1up(buf, 10, 2, CARD16, "fore-green");
    print_field_1up(buf, 12, 2, CARD16, "fore-blue");
    print_field_1up(buf, 14, 2, CARD16, "back-red");
    print_field_1up(buf, 16, 2, CARD16, "back-green");
    print_field_1up(buf, 18, 2, CARD16, "back-blue");
}

/*----------------------------------------------------------------------*/
void QueryBestSize(buf)
    unsigned char *buf;
{
    /* Request QueryBestSize is opcode 97 */
    print_field_1up(buf, 0, 1, REQUEST, REQUESTHEADER);		/* QueryBestSize */
    if (Verbose < 1)
	return;
    if (Verbose > 1)
	print_field_1up(SBf, 0, 4, CARD32, "sequence number");

    print_field_1up(buf, 1, 1, OBJECTCLASS, "class");
    print_field_2up(buf, 2, 2, CONST2(3), "request length");
    print_field_1up(buf, 4, 4, DRAWABLE, "drawable");
    print_field_1up(buf, 8, 2, CARD16, "width");
    print_field_1up(buf, 10, 2, CARD16, "height");
}

/*----------------------------------------------------------------------*/
void QueryBestSizeReply(buf)
    unsigned char *buf;
{
    print_field_1up(RBf, 0, 1, REPLY, REPLYHEADER);	/* QueryBestSize */
    if (Verbose < 1)
	return;
    print_field_2up(buf, 2, 2, CARD16, "sequence number");
    print_field_2up(buf, 4, 4, CONST4(0), "reply length");
    print_field_1up(buf, 8, 2, CARD16, "width");
    print_field_1up(buf, 10, 2, CARD16, "height");
}

/*----------------------------------------------------------------------*/
void QueryExtension(buf)
    unsigned char *buf;
{
    short n;

    /* Request QueryExtension is opcode 98 */
    print_field_1up(buf, 0, 1, REQUEST, REQUESTHEADER);		/* QueryExtension */
    if (Verbose < 1)
	return;
    if (Verbose > 1)
	print_field_1up(SBf, 0, 4, CARD32, "sequence number");

    print_field_2up(buf, 2, 2, DVALUE2(2 + (n + p) / 4), "request length");
    print_field_2up(buf, 4, 2, DVALUE2(n), "length of name");
    n = IShort(&buf[4]);
    print_str8_1up(&buf[8], (long) n, "name");
}

/*----------------------------------------------------------------------*/
void QueryExtensionReply(buf)
    unsigned char *buf;
{
    print_field_1up(RBf, 0, 1, REPLY, REPLYHEADER);	/* QueryExtension */
    if (Verbose < 1)
	return;
    print_field_2up(buf, 2, 2, CARD16, "sequence number");
    print_field_2up(buf, 4, 4, CONST4(0), "reply length");
    print_field_1up(buf, 8, 1, BOOL, "present");
    print_field_1up(buf, 9, 1, CARD8, "major-opcode");
    print_field_1up(buf, 10, 1, CARD8, "first-event");
    print_field_1up(buf, 11, 1, CARD8, "first-error");
}

/*----------------------------------------------------------------------*/
void ListExtensions(buf)
    unsigned char *buf;
{
    /* Request ListExtensions is opcode 99 */
    print_field_1up(buf, 0, 1, REQUEST, REQUESTHEADER);		/* ListExtensions */
    if (Verbose < 1)
	return;
    if (Verbose > 1)
	print_field_1up(SBf, 0, 4, CARD32, "sequence number");

    print_field_2up(buf, 2, 2, CONST2(1), "request length");
}

/*----------------------------------------------------------------------*/
void ListExtensionsReply(buf)
    unsigned char *buf;
{
    short n;

    print_field_1up(RBf, 0, 1, REPLY, REPLYHEADER);	/* ListExtensions */
    if (Verbose < 1)
	return;
    print_field_2up(buf, 1, 1, CARD8, "number names");
    n = IByte(&buf[1]);
    print_field_2up(buf, 2, 2, CARD16, "sequence number");
    print_field_2up(buf, 4, 4, DVALUE4((n + p) / 4), "reply length");
    print_list_str_1up(&buf[32], (long) n, "names");
}

/*----------------------------------------------------------------------*/
void ChangeKeyboardMapping(buf)
    unsigned char *buf;
{
    short n;
    short m;

    /* Request ChangeKeyboardMapping is opcode 100 */
    print_field_1up(buf, 0, 1, REQUEST, REQUESTHEADER);		/* ChangeKeyboardMapping */
    if (Verbose < 1)
	return;
    if (Verbose > 1)
	print_field_1up(SBf, 0, 4, CARD32, "sequence number");

    print_field_1up(buf, 1, 1, DVALUE1(n), "keycode-count");
    n = IByte(&buf[1]);
    print_field_2up(buf, 2, 2, DVALUE2(2 + nm), "request length");
    print_field_1up(buf, 4, 1, KEYCODE, "first-keycode");
    print_field_1up(buf, 5, 1, DVALUE1(m), "keysyms-per-keycode");
    m = IByte(&buf[5]);
    (void) print_list_1up(&buf[8], (long) (n * m), KEYSYM, "keysyms");
}

/*----------------------------------------------------------------------*/
void GetKeyboardMapping(buf)
    unsigned char *buf;
{
    /* Request GetKeyboardMapping is opcode 101 */
    print_field_1up(buf, 0, 1, REQUEST, REQUESTHEADER);		/* GetKeyboardMapping */
    if (Verbose < 1)
	return;
    if (Verbose > 1)
	print_field_1up(SBf, 0, 4, CARD32, "sequence number");

    print_field_2up(buf, 2, 2, CONST2(2), "request length");
    print_field_1up(buf, 4, 1, KEYCODE, "first-keycode");
    print_field_1up(buf, 5, 1, CARD8, "count");
}

/*----------------------------------------------------------------------*/
void GetKeyboardMappingReply(buf)
    unsigned char *buf;
{
    long n;
    print_field_1up(RBf, 0, 1, REPLY, REPLYHEADER);	/* GetKeyboardMapping */
    if (Verbose < 1)
	return;
    print_field_1up(buf, 1, 1, DVALUE1(n), "keysyms-per-keycode");
    print_field_2up(buf, 2, 2, CARD16, "sequence number");
    print_field_2up(buf, 4, 4, DVALUE4(n * m), "reply length");
    n = ILong(&buf[4]);
    print_list_2up(&buf[32], n, KEYSYM, "keysyms");
}

/*----------------------------------------------------------------------*/
void ChangeKeyboardControl(buf)
    unsigned char *buf;
{
    /* Request ChangeKeyboardControl is opcode 102 */
    print_field_1up(buf, 0, 1, REQUEST, REQUESTHEADER);		/* ChangeKeyboardControl */
    if (Verbose < 1)
	return;
    if (Verbose > 1)
	print_field_1up(SBf, 0, 4, CARD32, "sequence number");

    print_field_2up(buf, 2, 2, DVALUE2(2 + n), "request length");
    print_field_1up(buf, 4, 4, KEYBOARD_BITMASK, "value-mask");
    print_values_1up(&buf[4], 4, KEYBOARD_BITMASK, &buf[8], "value-list");
}

/*----------------------------------------------------------------------*/
void GetKeyboardControl(buf)
    unsigned char *buf;
{
    /* Request GetKeyboardControl is opcode 103 */
    print_field_1up(buf, 0, 1, REQUEST, REQUESTHEADER);		/* GetKeyboardControl */
    if (Verbose < 1)
	return;
    if (Verbose > 1)
	print_field_1up(SBf, 0, 4, CARD32, "sequence number");

    print_field_2up(buf, 2, 2, CONST2(1), "request length");
}

/*----------------------------------------------------------------------*/
void GetKeyboardControlReply(buf)
    unsigned char *buf;
{
    print_field_1up(RBf, 0, 1, REPLY, REPLYHEADER);	/* GetKeyboardControl */
    if (Verbose < 1)
	return;
    print_field_1up(buf, 1, 1, OFF_ON, "global-auto-repeat");
    print_field_2up(buf, 2, 2, CARD16, "sequence number");
    print_field_2up(buf, 4, 4, CONST4(5), "reply length");
    print_field_1up(buf, 8, 4, CARD32, "led-mask");
    print_field_1up(buf, 12, 1, CARD8, "key-click-percent");
    print_field_1up(buf, 13, 1, CARD8, "bell-percent");
    print_field_1up(buf, 14, 2, CARD16, "bell-pitch");
    print_field_1up(buf, 16, 2, CARD16, "bell-duration");
    print_bytes_1up(&buf[20], 32L, "auto-repeats");
}

/*----------------------------------------------------------------------*/
void Bell(buf)
    unsigned char *buf;
{
    /* Request Bell is opcode 104 */
    print_field_1up(buf, 0, 1, REQUEST, REQUESTHEADER);		/* Bell */
    if (Verbose < 1)
	return;
    if (Verbose > 1)
	print_field_1up(SBf, 0, 4, CARD32, "sequence number");

    print_field_1up(buf, 1, 1, INT8, "percent");
    print_field_2up(buf, 2, 2, CONST2(1), "request length");
}

/*----------------------------------------------------------------------*/
void ChangePointerControl(buf)
    unsigned char *buf;
{
    /* Request ChangePointerControl is opcode 105 */
    print_field_1up(buf, 0, 1, REQUEST, REQUESTHEADER);		/* ChangePointerControl */
    if (Verbose < 1)
	return;
    if (Verbose > 1)
	print_field_1up(SBf, 0, 4, CARD32, "sequence number");

    print_field_2up(buf, 2, 2, CONST2(3), "request length");
    print_field_1up(buf, 4, 2, INT16, "acceleration-numerator");
    print_field_1up(buf, 6, 2, INT16, "acceleration-denominator");
    print_field_1up(buf, 8, 2, INT16, "threshold");
    print_field_1up(buf, 10, 1, BOOL, "do-acceleration");
    print_field_1up(buf, 11, 1, BOOL, "do-threshold");
}

/*----------------------------------------------------------------------*/
void GetPointerControl(buf)
    unsigned char *buf;
{
    /* Request GetPointerControl is opcode 106 */
    print_field_1up(buf, 0, 1, REQUEST, REQUESTHEADER);		/* GetPointerControl */
    if (Verbose < 1)
	return;
    if (Verbose > 1)
	print_field_1up(SBf, 0, 4, CARD32, "sequence number");

    print_field_2up(buf, 2, 2, CONST2(1), "request length");
}

/*----------------------------------------------------------------------*/
void GetPointerControlReply(buf)
    unsigned char *buf;
{
    print_field_1up(RBf, 0, 1, REPLY, REPLYHEADER);	/* GetPointerControl */
    if (Verbose < 1)
	return;
    print_field_2up(buf, 2, 2, CARD16, "sequence number");
    print_field_2up(buf, 4, 4, CONST4(0), "reply length");
    print_field_1up(buf, 8, 2, CARD16, "acceleration-numerator");
    print_field_1up(buf, 10, 2, CARD16, "acceleration-denominator");
    print_field_1up(buf, 12, 2, CARD16, "threshold");
}

/*----------------------------------------------------------------------*/
void SetScreenSaver(buf)
    unsigned char *buf;
{
    /* Request SetScreenSaver is opcode 107 */
    print_field_1up(buf, 0, 1, REQUEST, REQUESTHEADER);		/* SetScreenSaver */
    if (Verbose < 1)
	return;
    if (Verbose > 1)
	print_field_1up(SBf, 0, 4, CARD32, "sequence number");

    print_field_2up(buf, 2, 2, CONST2(3), "request length");
    print_field_1up(buf, 4, 2, INT16, "timeout");
    print_field_1up(buf, 6, 2, INT16, "interval");
    print_field_1up(buf, 8, 1, NO_YES, "prefer-blanking");
    print_field_1up(buf, 9, 1, NO_YES, "allow-exposures");
}

/*----------------------------------------------------------------------*/
void GetScreenSaver(buf)
    unsigned char *buf;
{
    /* Request GetScreenSaver is opcode 108 */
    print_field_1up(buf, 0, 1, REQUEST, REQUESTHEADER);		/* GetScreenSaver */
    if (Verbose < 1)
	return;
    if (Verbose > 1)
	print_field_1up(SBf, 0, 4, CARD32, "sequence number");

    print_field_2up(buf, 2, 2, CONST2(1), "request length");
}

/*----------------------------------------------------------------------*/
void GetScreenSaverReply(buf)
    unsigned char *buf;
{
    print_field_1up(RBf, 0, 1, REPLY, REPLYHEADER);	/* GetScreenSaver */
    if (Verbose < 1)
	return;
    print_field_2up(buf, 2, 2, CARD16, "sequence number");
    print_field_2up(buf, 4, 4, CONST4(0), "reply length");
    print_field_1up(buf, 8, 2, CARD16, "timeout");
    print_field_1up(buf, 10, 2, CARD16, "interval");
    print_field_1up(buf, 12, 1, NO_YES, "prefer-blanking");
    print_field_1up(buf, 13, 1, NO_YES, "allow-exposures");
}

/*----------------------------------------------------------------------*/
void ChangeHosts(buf)
    unsigned char *buf;
{
    short n;

    /* Request ChangeHosts is opcode 109 */
    print_field_1up(buf, 0, 1, REQUEST, REQUESTHEADER);		/* ChangeHosts */
    if (Verbose < 1)
	return;
    if (Verbose > 1)
	print_field_1up(SBf, 0, 4, CARD32, "sequence number");

    print_field_1up(buf, 1, 1, INS_DEL, "mode");
    print_field_2up(buf, 2, 2, DVALUE2(2 + (n + p) / 4), "request length");
    print_field_1up(buf, 4, 1, HOSTFAMILY, "family");
    print_field_2up(buf, 6, 2, CARD16, "length of address");
    n = IShort(&buf[6]);
    print_bytes_1up(&buf[8], (long) n, "address");
}

/*----------------------------------------------------------------------*/
void ListHosts(buf)
    unsigned char *buf;
{
    /* Request ListHosts is opcode 110 */
    print_field_1up(buf, 0, 1, REQUEST, REQUESTHEADER);		/* ListHosts */
    if (Verbose < 1)
	return;
    if (Verbose > 1)
	print_field_1up(SBf, 0, 4, CARD32, "sequence number");

    print_field_2up(buf, 2, 2, CONST2(1), "request length");
}

/*----------------------------------------------------------------------*/
void ListHostsReply(buf)
    unsigned char *buf;
{
    short n;

    print_field_1up(RBf, 0, 1, REPLY, REPLYHEADER);	/* ListHosts */
    if (Verbose < 1)
	return;
    print_field_1up(buf, 1, 1, DIS_EN, "mode");
    print_field_2up(buf, 2, 2, CARD16, "sequence number");
    print_field_2up(buf, 4, 4, DVALUE4(n / 4), "reply length");
    print_field_2up(buf, 8, 2, CARD16, "number of hosts");
    n = IShort(&buf[8]);
    (void) print_list_1up(&buf[32], (long) n, HOST, "hosts");
}

/*----------------------------------------------------------------------*/
void SetAccessControl(buf)
    unsigned char *buf;
{
    /* Request SetAccessControl is opcode 111 */
    print_field_1up(buf, 0, 1, REQUEST, REQUESTHEADER);		/* SetAccessControl */
    if (Verbose < 1)
	return;
    if (Verbose > 1)
	print_field_1up(SBf, 0, 4, CARD32, "sequence number");

    print_field_1up(buf, 1, 1, DIS_EN, "mode");
    print_field_2up(buf, 2, 2, CONST2(1), "request length");
}

/*----------------------------------------------------------------------*/
void SetCloseDownMode(buf)
    unsigned char *buf;
{
    /* Request SetCloseDownMode is opcode 112 */
    print_field_1up(buf, 0, 1, REQUEST, REQUESTHEADER);		/* SetCloseDownMode */
    if (Verbose < 1)
	return;
    if (Verbose > 1)
	print_field_1up(SBf, 0, 4, CARD32, "sequence number");

    print_field_1up(buf, 1, 1, CLOSEMODE, "mode");
    print_field_2up(buf, 2, 2, CONST2(1), "request length");
}

/*----------------------------------------------------------------------*/
void KillClient(buf)
    unsigned char *buf;
{
    /* Request KillClient is opcode 113 */
    print_field_1up(buf, 0, 1, REQUEST, REQUESTHEADER);		/* KillClient */
    if (Verbose < 1)
	return;
    if (Verbose > 1)
	print_field_1up(SBf, 0, 4, CARD32, "sequence number");

    print_field_2up(buf, 2, 2, CONST2(2), "request length");
    print_field_1up(buf, 4, 4, RESOURCEID, "resource");
}

/*----------------------------------------------------------------------*/
void RotateProperties(buf)
    unsigned char *buf;
{
    short n;

    /* Request RotateProperties is opcode 114 */
    print_field_1up(buf, 0, 1, REQUEST, REQUESTHEADER);		/* RotateProperties */
    if (Verbose < 1)
	return;
    if (Verbose > 1)
	print_field_1up(SBf, 0, 4, CARD32, "sequence number");

    print_field_2up(buf, 2, 2, DVALUE2(3 + n), "request length");
    print_field_1up(buf, 4, 4, WINDOW, "window");
    print_field_2up(buf, 8, 2, DVALUE2(n), "number of properties");
    n = IShort(&buf[8]);
    print_field_1up(buf, 10, 2, INT16, "delta");
    (void) print_list_1up(&buf[12], (long) n, ATOM, "properties");
}

/*----------------------------------------------------------------------*/
void ForceScreenSaver(buf)
    unsigned char *buf;
{
    /* Request ForceScreenSaver is opcode 115 */
    print_field_1up(buf, 0, 1, REQUEST, REQUESTHEADER);		/* ForceScreenSaver */
    if (Verbose < 1)
	return;
    if (Verbose > 1)
	print_field_1up(SBf, 0, 4, CARD32, "sequence number");

    print_field_1up(buf, 1, 1, SAVEMODE, "mode");
    print_field_2up(buf, 2, 2, CONST2(1), "request length");
}

/*----------------------------------------------------------------------*/
void SetPointerMapping(buf)
    unsigned char *buf;
{
    short n;

    /* Request SetPointerMapping is opcode 116 */
    print_field_1up(buf, 0, 1, REQUEST, REQUESTHEADER);		/* SetPointerMapping */
    if (Verbose < 1)
	return;
    if (Verbose > 1)
	print_field_1up(SBf, 0, 4, CARD32, "sequence number");

    print_field_2up(buf, 1, 1, DVALUE1(n), "length of map");
    n = IByte(&buf[1]);
    print_field_2up(buf, 2, 2, DVALUE2(1 + (n + p) / 4), "request length");
    print_bytes_1up(&buf[4], (long) n, "map");
}

/*----------------------------------------------------------------------*/
void SetPointerMappingReply(buf)
    unsigned char *buf;
{
    print_field_1up(RBf, 0, 1, REPLY, REPLYHEADER);	/* SetPointerMapping */
    if (Verbose < 1)
	return;
    print_field_1up(buf, 1, 1, RSTATUS, "status");
    print_field_2up(buf, 2, 2, CARD16, "sequence number");
    print_field_2up(buf, 4, 4, CONST4(0), "reply length");
}

/*----------------------------------------------------------------------*/
void GetPointerMapping(buf)
    unsigned char *buf;
{
    /* Request GetPointerMapping is opcode 117 */
    print_field_1up(buf, 0, 1, REQUEST, REQUESTHEADER);		/* GetPointerMapping */
    if (Verbose < 1)
	return;
    if (Verbose > 1)
	print_field_1up(SBf, 0, 4, CARD32, "sequence number");

    print_field_2up(buf, 2, 2, CONST2(1), "request length");
}

/*----------------------------------------------------------------------*/
void GetPointerMappingReply(buf)
    unsigned char *buf;
{
    short n;

    print_field_1up(RBf, 0, 1, REPLY, REPLYHEADER);	/* GetPointerMapping */
    if (Verbose < 1)
	return;
    print_field_2up(buf, 1, 1, DVALUE1(n), "length of map");
    n = IByte(&buf[1]);
    print_field_2up(buf, 2, 2, CARD16, "sequence number");
    print_field_2up(buf, 4, 4, DVALUE4((n + p) / 4), "reply length");
    print_bytes_1up(&buf[32], (long) n, "map");
}

/*----------------------------------------------------------------------*/
void SetModifierMapping(buf)
    unsigned char *buf;
{
    short n;

    /* Request SetModifierMapping is opcode 118 */
    print_field_1up(buf, 0, 1, REQUEST, REQUESTHEADER);		/* SetModifierMapping */
    if (Verbose < 1)
	return;
    if (Verbose > 1)
	print_field_1up(SBf, 0, 4, CARD32, "sequence number");

    print_field_1up(buf, 1, 1, DVALUE1(n), "keycodes-per-modifier");
    n = IByte(&buf[1]);
    print_field_2up(buf, 2, 2, DVALUE2(1 + 2 * n), "request length");
    print_bytes_1up(&buf[4 + 0 * n], (long) n, "Shift keycodes");
    print_bytes_1up(&buf[4 + 1 * n], (long) n, "Lock keycodes");
    print_bytes_1up(&buf[4 + 2 * n], (long) n, "Control keycodes");
    print_bytes_1up(&buf[4 + 3 * n], (long) n, "Mod1 keycodes");
    print_bytes_1up(&buf[4 + 4 * n], (long) n, "Mod2 keycodes");
    print_bytes_1up(&buf[4 + 5 * n], (long) n, "Mod3 keycodes");
    print_bytes_1up(&buf[4 + 6 * n], (long) n, "Mod4 keycodes");
    print_bytes_1up(&buf[4 + 7 * n], (long) n, "Mod5 keycodes");
}

/*----------------------------------------------------------------------*/
void SetModifierMappingReply(buf)
    unsigned char *buf;
{
    print_field_1up(RBf, 0, 1, REPLY, REPLYHEADER);	/* SetModifierMapping */
    if (Verbose < 1)
	return;
    print_field_1up(buf, 1, 1, RSTATUS, "status");
    print_field_2up(buf, 2, 2, CARD16, "sequence number");
    print_field_2up(buf, 4, 4, CONST4(0), "reply length");
}

/*----------------------------------------------------------------------*/
void GetModifierMapping(buf)
    unsigned char *buf;
{
    /* Request GetModifierMapping is opcode 119 */
    print_field_1up(buf, 0, 1, REQUEST, REQUESTHEADER);		/* GetModifierMapping */
    if (Verbose < 1)
	return;
    if (Verbose > 1)
	print_field_1up(SBf, 0, 4, CARD32, "sequence number");

    print_field_2up(buf, 2, 2, CONST2(1), "request length");
}

/*----------------------------------------------------------------------*/
void GetModifierMappingReply(buf)
    unsigned char *buf;
{
    short n;

    print_field_1up(RBf, 0, 1, REPLY, REPLYHEADER);	/* GetModifierMapping */
    if (Verbose < 1)
	return;
    print_field_1up(buf, 1, 1, DVALUE1(n), "keycodes-per-modifier");
    n = IByte(&buf[1]);
    print_field_2up(buf, 2, 2, CARD16, "sequence number");
    print_field_2up(buf, 4, 4, DVALUE4(2 * n), "reply length");
    (void) print_list_1up(&buf[32], (long) n, KEYCODE, "keycodes");
}

/*----------------------------------------------------------------------*/
void NoOperation(buf)
    unsigned char *buf;
{
    /* Request NoOperation is opcode 127 */
    print_field_1up(buf, 0, 1, REQUEST, REQUESTHEADER);		/* NoOperation */
    if (Verbose < 1)
	return;
    if (Verbose > 1)
	print_field_1up(SBf, 0, 4, CARD32, "sequence number");

    print_field_2up(buf, 2, 2, CONST2(1), "request length");
}
