/* 
 * $Id: record.c,v 1.8 1998/07/24 11:59:16 marc Exp $ 
 *
 *  Code to record / play back X events in script
 *  
 * (c) Copyright 1996, 1997, 1998 Marc Vertes
 */

#include <stdio.h>
#include <stdlib.h>
#include <X11/Xlib.h>
#include <X11/Xutil.h>
#include <X11/keysym.h>
#include "scope.h"
#include "server.h"
#include "record.h"
#include "x11.h"

extern Display *Dpy;

static char script_line[256];	 /* stored script line */

struct WinTab wintab[MAXWTAB];
struct WinAlias winalias[MAXWTAB];

long maxalias = 0;
long maxwtab = 0;
int replay_counter = 0;
int first_replay = 1;
int first_event_in_replay = 0;

int xRootPointer, yRootPointer;	/* coordinates of pointer in the root window */
void (*Resource_Handler)() = 0;	/* handler to call */
void *Resource_userdata = 0;	/* user data to pass */
XID Resource_resource = 0;
TimerID Resource_timeout = 0;	/* timeout when resource doesn't appear */
int Resource_index = 0;		/* resource index within fd */
FD Resource_fd = 0;		/* resource file descriptor */

/* Synthetic X event rebuild from script */
struct Synth_ev {
    Window win;
    Window subw;
    Window rootw;
    Time time;
    FD fd;
    unsigned short type;	/* Event type in binary form */
    short state;
    short detail;
    short same;
    short x, y;
    short rootx, rooty;
    unsigned short seq;
    unsigned char evbuf[32];
} synth_ev;

#define TheRootWindow wintab[0].newpar

/*--------------------------------------------------------------*/
int start_record(filename)
    char *filename;
{
    if (Playback != PLAYBACK_INACTIVE && Playback != PLAYBACK_COMPLETE) {
	fprintf(stderr, "Cannot record and play back in same session.\n");
	return (1);
    }
    strcpy(RecordFileName, filename);
    RecordFile = fopen(RecordFileName, "w");
    if (RecordFile == 0) {
	perror(RecordFileName);
	return (1);
    }
    fclose(RecordFile);
    RecordFile = 0;
    Record = 1;
    return (0);
}

/*--------------------------------------------------------------*/
void stop_record()
{
    CloseFile(RecordFile);
    fclose(RecordFile);
    Record = 0;
    first_replay = 0;
}

/*--------------------------------------------------------------*/
static FILE *OpenFile(name, priv)
    char *name;
    char *priv;
{
    register FILE *file;
    register int fid;

    /* open file */
    file = fopen(name, priv);

    /* deal with failure.  snarky messages are OK, we've pretested files */
    if (file == 0) {
	panic("OpenFile failure.\n");
    }
    /*
     * set up file descriptor in this program's table.
     * Seems to be necessary at least for SUN Solaris.
     */
    fid = fileno(file);
    UsingFD(fid, 0);
    return file;

}

/*--------------------------------------------------------------*/
void CloseFile(file)
    FILE *file;
{
    int fid = fileno(file);
    NotUsingFD(fid);
}

/*--------------------------------------------------------------*/
/*
 *  TimeDilation expresses the time dilation in percent.
 *  if TimeDilation = 200, then we want the thing to play back
 *  twice as fast, so we want to cut all times in half.
 */
unsigned long int Dilate(deltat)
    unsigned long int deltat;
{
    unsigned long int result;
    result = deltat;
    result = ((deltat * 100) + ((TimeDilation / 2) - 1)) / TimeDilation;
    return result;
}

/*--------------------------------------------------------------*/
static long int NextScriptEvent(buf, n)
    char *buf;
    long int n;
{
    /* 
     * This function reads the next script event
     * and returns it in a buffer.
     */
    char *result;
    int goodline = 0;
    int event;

    /* open up the recording file if necessary, first time through */
    if (PlaybackFile == 0) {
	PlaybackFile = OpenFile(PlaybackFileName, "r");
	Playback = PLAYBACK_ACTIVE;
	first_event_in_replay = 1;
    }
    /* skip blanks and comments */
    while (goodline == 0) {
	buf[0] = '\0';
	result = fgets(buf, n, PlaybackFile);
	if (result == 0)
	    return -1;

	/* skip comment-type lines */
	if (buf[0] != '\0') {
	    event = ascii_to_event_type(buf);
	    if (event != 0) {
		goodline = 1;
	    }
	}
    }
    return strlen(buf);
}

/*--------------------------------------------------------------*/
/*
 *  this timer function is called when 
 *  the wait for a window to become mapped 
 *  runs out...
 *  It issues a warning message, then proceeds as
 *  if the window had, in fact, appeared.
 *
 *  Under some failure conditions, this causes an
 *  error cascade.
 */
static void ResourceNeverAppeared(timer, window)
    TimerID timer;
    XID window;
{
    if (window != TheRootWindow) {
	fprintf(stderr, "Playback failure\n");
	fprintf(stderr, "  Timed out waiting for window %8.8x ( %8.8x )\n", (unsigned int) window,
		OriginalWid(window));
	fprintf(stderr, "  %s\n", script_line);
	Errorcount++;
    }
    /* call the wait handler, attempting to carry on with the run */

    if (Resource_Handler)
	(*Resource_Handler) (Resource_resource, Resource_userdata);

    /* clear off the waiting flag in the resource status map */
    if (Resource_index < MAXRESOURCESTATUS) {
	CS[Resource_fd].ResourceStatus[Resource_index] &= ~RES_Waiting;
    }
    /* Restore the resource-wait information to initial state */
    Resource_Handler = 0;
    Resource_userdata = 0;
    Resource_resource = 0;
    Resource_timeout = 0;
    Resource_index = 0;
    Resource_fd = 0;
}

/*--------------------------------------------------------------*/
/* get the FD to which this window 'belongs' */
int GetResourceBaseFd(resource)
    XID resource;
{
    long ifd;
    unsigned long int c_res_base;

    for (ifd = 1; ifd <= HighestFD; ifd++) {
	c_res_base = resource & ~CS[ifd].ResourceIdMask;
	if (FDD[ifd].Busy) {
	    if (c_res_base == CS[ifd].ResourceIdBase)
		return ifd;
	} else {
	    if (c_res_base == CS[ifd].ResourceIdBase) {
		fprintf(stderr,
			"Playback Error. fd %ld for resource %x (%x) is not valid anymore\n",
		        ifd, (unsigned int) resource, OriginalWid(resource));
		return 0;
	    }
	}
    }
    /* maybe an external window */
    if ((resource != TheRootWindow) && (OriginalWid(resource) != 0))
	fprintf(stderr, "Playback Error. Couldn't find an fd for resource %x (%x)\n",
		(unsigned int) resource, OriginalWid(resource));
    return(0);
}

/*-------------------------------------------------------------*/
/*
 * this function can be called when somebody
 * wants to wait for a window to appear.
 */
void WaitForWindowMapped(fd, resource, handler, user_data)
    int fd;
    XID resource;
    void (*handler) ();
    void *user_data;

{
    int ifd;
    int res_index = resource & CS[fd].ResourceIdMask;


    if (Resource_Handler)
	panic("already waiting for a resource....only one allowed!");

    if (res_index >= MAXRESOURCESTATUS) {
	warn("res_index beyond MAXRESOURCESTATUS in WaitForWindowMapped.");
	/* beyond the resource count; just assume it's available */
	if (handler)
	    (*handler) (resource, user_data);
	return;
    }
    /* No check for first EnterNotify event */
    if (first_event_in_replay && synth_ev.type == EnterNotify) {
	first_event_in_replay = 0;
	if (handler)
	    (*handler) (resource, user_data);
	return;
    }
    ifd = GetResourceBaseFd(resource);

    /* 
     * a completeness test 
     */
    if (resource != TheRootWindow) { /** 'res_unmapped' **/
	if (ifd == 0 && OriginalWid(resource) != 0) {
	    fprintf(stderr, "Playback Error. Contradiction ifd=0 resource %x (%x)\n",
		    (unsigned int) resource, OriginalWid(resource));
	    ReadPlaybackEvent(20L, fd);
	    return;
	}
	if ((ifd != 0) && OriginalWid(resource) == 0) {
	    fprintf(stderr,
		    "Playback Error.Probably window destroyed. ifd = %d resource %x (0)\n",
		    ifd, (unsigned int) resource);
	    CS[ifd].ResourceStatus[res_index] = 0;
	    ReadPlaybackEvent(20L, fd);
	    return;
	}
    }
    /*
     * if the window is mapped or its is an external window 
     */
    if ((ifd == 0) || CS[fd].ResourceStatus[res_index] == RES_Mapped ||
	CS[ifd].ResourceStatus[res_index] == RES_Mapped) {
	(*handler) (resource, user_data);
	return;
    }
    CS[ifd].ResourceStatus[res_index] |= RES_Waiting;
    Resource_Handler = handler;
    Resource_userdata = user_data;
    Resource_resource = resource;
    Resource_timeout = CreateTimer(2000, (void (*)()) ResourceNeverAppeared, (void *) resource);
    Resource_index = res_index;
    Resource_fd = ifd;
    debug(4, (stderr, "WAIT FOR %7x FD = %d IFD = %d\n", (unsigned int) resource, fd, ifd));
}

/*-------------------------------------------------------------*/
/* debugging aid */
char *getstat(status)
    int status;
{

    switch (status) {
    case 0:
	return "Unseen";
    case 1:
	return "Seen";
    case 2:
	return "Mapped";
    case 3:
	return "Unmapped";
    case 128:
	return "Wait";
    case 129:
	return "Wait&Seen";
    case 130:
	return "Wait&Mapped";
    case 131:
	return "Wait&Unmapp";
    default:
	return "illegal status";
    }
}

/*-------------------------------------------------------------*/
/*
 * this routine gets called from the event formatters
 * in decode11.c
 * it is used to maintain event status.
 */
void WindowStatusAnnounce(fd, buf, n, status)
    FD fd;
    unsigned char *buf;
    int n;
    ResStatus status;
{
    XID resource = ILong(&buf[n]);
    int res_index = resource & CS[fd].ResourceIdMask;
    int waiting = 0;
    ResStatus oldstatus;
    int ifd, i;

    enterprocedure("WindowStatusAnnounce");

    ifd = GetResourceBaseFd(resource);

    if ((resource == 0) || (ifd == 0))
	return;

    debug(4, (stderr,
	      "curwin=%7x resindx = %3d waitwin=%7x FD = %d IFD = %d \n",
	      (unsigned int) resource, res_index, (unsigned int) Resource_resource, fd, ifd));

    debug(4, (stderr,
              "                             enstat = %10s oldstat = %10s nwstat = ",
	      getstat(status),
	      getstat(CS[ifd].ResourceStatus[res_index])));

    if (res_index < MAXRESOURCESTATUS) {
	oldstatus = CS[ifd].ResourceStatus[res_index];
	if (oldstatus & RES_Waiting)
	    waiting = 1;
	CS[ifd].ResourceStatus[res_index] = status | (oldstatus & RES_Waiting);
    }
    /* if we were waiting for the resource and it is now mapped ... */
    if (resource == Resource_resource) {
	if (waiting & (status == RES_Mapped)) {
	    /* blast the wait timeout */
	    if (Resource_timeout)
		DeleteTimer(Resource_timeout);

	    /* call the resource handler, if any */
	    if (Resource_Handler)
		(*Resource_Handler) (Resource_resource, Resource_userdata);

	    /* zap the waiting bit in the resource map */
	    if (Resource_index < MAXRESOURCESTATUS) {
		CS[Resource_fd].ResourceStatus[Resource_index] &= ~RES_Waiting;
	    }
	    /* Restore the resource-wait information to initial state */
	    Resource_Handler = 0;
	    Resource_userdata = 0;
	    Resource_resource = 0;
	    Resource_timeout = 0;
	    Resource_index = 0;
	    Resource_fd = 0;
	}
    }
    if (status == RES_Mapped) {
	CS[ifd].ResourceStatus[res_index] |= RES_Mapped;

	/* indicate that the children of this window are also mapped */
	for (i = 0; i < maxwtab; i++) {
	    if (wintab[i].newpar == resource) {
		int ifdchild;
		int res_indexchild;

		ifdchild = GetResourceBaseFd(wintab[i].newid);
		res_indexchild = wintab[i].newid & CS[ifdchild].ResourceIdMask;

		if (ifdchild) {
		    CS[ifdchild].ResourceStatus[res_indexchild] &= ~RES_Waiting;
		    CS[ifdchild].ResourceStatus[res_indexchild] |= RES_Mapped;
		}
	    }
        }
    }
    debug(4, (stderr, "%10s Wait = %1d\n", getstat(CS[ifd].ResourceStatus[res_index]), waiting));
    return;
}

/*--------------------------------------------------------------*/
void AnnouncePlaybackError(dpy, erev)
    Display *dpy;
    XErrorEvent *erev;
{
    if (synth_ev.type != EnterNotify) {
	fprintf(stderr, "Playback failure\n");
	fprintf(stderr, "  Unable to move pointer into window %8.8x (%x)\n",
	 (unsigned int) erev->resourceid, OriginalWid(erev->resourceid));
	fprintf(stderr, "  %s\n", script_line);
	Errorcount++;
    }
}

/*--------------------------------------------------------------*/
void SendPlaybackEvent(timer, fd)
    TimerID timer;
    int fd;
{
    struct timeval tv;
    FD pairfd = FDPair(fd);
    unsigned short int seq = 0;
    Time timestamp, t;

    /* obtain the sequence number from the client-side CS structure */
    if (pairfd >= StaticMaxFD)
	pairfd = -1;
    if (pairfd >= 0)
	seq = CS[pairfd].SequenceNumber;

    /* compute timestamp based on previously noted time offset */
    gettimeofday(&tv, 0);
    t = tv.tv_sec * 1000;
    t += tv.tv_usec / 1000;
    timestamp = t + CS[fd].TimeOffset;

    /* it's time to send the event to the program under test..  */
    /* convert it to a wire-format event buffer */
    OShort(&synth_ev.evbuf[2], seq);
    OLong(&synth_ev.evbuf[4], timestamp);
    OShort(&synth_ev.evbuf[20], xRootPointer);
    OShort(&synth_ev.evbuf[22], yRootPointer);
    ReplaceWid(synth_ev.evbuf, 12);
    ReplaceWid(synth_ev.evbuf, 16);

    WriteToClient(fd, synth_ev.evbuf, 32);

    /* if Shift + Ctrl + Button 1, activate screen_capture */
    if ((synth_ev.type == ButtonRelease) &&
	(synth_ev.state & ShiftMask) &&
	(synth_ev.state & ControlMask))
	screen_capture();

    /* read the next event from the script file */
    ReadPlaybackEvent(timer, fd);
}

/*--------------------------------------------------------------*/
extern int whichGrab;
extern int status;

void WarpPlaybackEvent(session_window, fd)
    Window session_window;
    int fd;
{
    /*
     * This function decodes part of a stored script_line,
     * and warps the server pointer to the selected
     * position.  It is called when the window is known to be mapped.
     * This function ends by registering a callback to 
     * "SendPlaybackEvent" for 20 ms later.
     */
    Window saved_window;
    static int first = 0;
    /* more variables to query pointer */
    Window root_return, child_return;
    int win_x_return, win_y_return;
    unsigned int mask_Return;

    if (first == 0) {
	(void) XSetErrorHandler((int (*)()) AnnouncePlaybackError);
	first = 1;
    }
    saved_window = NewWid(synth_ev.win);

    /* EnterNotify events always causes problems */
    if (!(synth_ev.type == 7 &&
	  saved_window == TheRootWindow && synth_ev.subw == 0)) {

	if (saved_window) {
	    XWarpPointer(Dpy, None, saved_window, 0, 0, 0, 0, synth_ev.x, synth_ev.y);

            /* 
	     * This code does not work in all cases where you 'drag a window' 
             * because the server or pointer is grabbed by another client 
	     */
	    if (!((whichGrab == 4 /*server */  || whichGrab == 1 /*pntr */ ) && status == 1)) {
		XSync(Dpy, False);
		/* 
		 * Get root_x and root_y for the pointer, to set event coord accordingly 
		 */
		XQueryPointer(Dpy, saved_window, &root_return, &child_return,
			      &xRootPointer, &yRootPointer, &win_x_return, &win_y_return, &mask_Return);
	    } else {
		XFlush(Dpy);
		if (saved_window == TheRootWindow) {
		    xRootPointer = synth_ev.x;
		    yRootPointer = synth_ev.y;
		}
	    }
	    CreateTimer(20L, (void (*)()) SendPlaybackEvent, (void *) fd);
	    return;
	}
    }
    ReadPlaybackEvent(20L, fd);
}


/*--------------------------------------------------------------*/
void TranslatePlaybackEvent(timer, fd)
    TimerID timer;
    int fd;
{
    /*
     * This function decodes part of a stored script_line,
     * and waits for the named window to appear.
     */
    Window session_window = NewWid(synth_ev.win);

    if (session_window == 0)
	fprintf(stderr, "Error. session_window = 0 \n");
    WaitForWindowMapped(fd, session_window, (void (*)()) WarpPlaybackEvent, (void *) fd);
}

/*--------------------------------------------------------------*/
void end_replay()
{
    CloseFile(PlaybackFile);
    warn("Playback completed");
    Playback = PLAYBACK_COMPLETE;
    replay_counter--;
    if (replay_counter > 0) {
	warn("Activate next playback event");
	start_replay(PlaybackFileName);
    }
}

/*--------------------------------------------------------------*/
void ReadPlaybackEvent(timer, fd)
    TimerID timer;
    int fd;
{
    /*
     * This function reads an event from the script file.
     * It then establishes a timer call back to 
     * activate WarpPlaybackEvent at the appropriate time
     * (based on the elapsed time specified in the script file.)
     *
     * If NextScriptEvent comes back = -1, it's the end of
     * the script file;  forget about further timer callbacks.
     * However, don't stop the xscript pseudo-server, because
     * it should stay running until client-under-test disconnect.
     */
    long int linelen = NextScriptEvent(script_line, 256);
    unsigned long int delay, delay2;

    if (linelen < 0) {		
        /* 
	 * End of file ... wait 5 seconds to let application finish
         * windows closing.
         * Mark playback as complete 
         */
	warn("Playback completion in 5s");
	CreateTimer(5000, end_replay, (void *) 0);
    } else {
	/* 
	 * Process next event from file. 
	 */
	if (ReadSynthEvent(script_line) == 1) {
	    ReadPlaybackEvent(20L, fd);
	    return;
	}
	delay = synth_ev.time;

	switch (synth_ev.type) {
	case ButtonPress:
	    /*
	     *  to avoid fouling up double clicks,
	     *  don't mess around with delay times before ButtonPress < 250 ms 
	     */
	    if (delay < DoubleClickTime) {
		/* delay is shorter than doubleclick time, don't touch it */
		delay2 = delay;
	    } else {
		/*
		 * Original delay is longer than doubleclick time, 
		 * don't let recomputed delay end up shorter than doubleclick time
		 */
		delay2 = Dilate(delay);
		if (delay2 < DoubleClickTime)
		    delay2 = delay;
	    }
	    break;

	default:
	    /* 
	     * It's OK to compress other event delays more-or-less arbitrarily 
	     */
	    delay2 = Dilate(delay);
	    break;
	}

	/* subtract delay time we'll use for waiting for motion */
	if (delay2 > 20)
	    delay2 -= 20;
	else
	    delay2 = 0;

	if (delay2 < 5)
	    delay2 = 5;
	debug(128, (stderr, "%ld=delay %s\n", delay2, script_line));
	CreateTimer(delay2, (void (*)()) TranslatePlaybackEvent, (void *) synth_ev.fd);
    }
}

/*--------------------------------------------------------------*/
int start_replay(filename)
    char *filename;
{
    if (Record) {
	fprintf(stderr, "Cannot record and play back in same session.\n");
	return (1);
    }
    if (Playback != PLAYBACK_INACTIVE && Playback != PLAYBACK_COMPLETE) {
	fprintf(stderr, "Previous playback not yet completed.\n");
	return (1);
    }
    strcpy(PlaybackFileName, filename);

    /* ensure we can open playback file correctly */
    PlaybackFile = fopen(PlaybackFileName, "r");
    if (PlaybackFile == 0) {
	perror(PlaybackFileName);
	return (1);
    }
    fclose(PlaybackFile);
    PlaybackFile = 0;
    Playback = PLAYBACK_PENDING;

    /* read playbackfile to to extract Windows info */
    if (first_replay) {
	LoadWtab("PlaybackFileName");
	first_replay = 0;
    } else {
	CreateTimer(5000, (void (*)()) ReadPlaybackEvent, (void *) 0);
    }
    return (0);
}

/*--------------------------------------------------------------*/
void RecordEvent(fd, buf, n)
    FD fd;
    unsigned char *buf;
    long n;
{
    register short type = IByte(&buf[0]);	/* event type */

    if (RecordFile == 0) {
	/* open up the recording file if necessary, first time through */
	RecordFile = OpenFile(RecordFileName, "w");
    }
    /* high-order bit means SendEvent generated */
    if (type & 0x80) {
	type = type & 0x7F;
    }
    /*
     * note that we can't cast stuff to
     * the XEvent structure here, because
     * the information is in a message buffer,
     * not correctly unpacked and byteswapped.
     */
    switch (type) {
    case ButtonPress:
    case ButtonRelease:
    case KeyPress:
    case KeyRelease:
    case MotionNotify:
    case EnterNotify:
	/* process the event */
	event_to_ascii(fd, buf, RecordFile);
	break;
    default:
	/*EMPTY *//* ignore the event */
	break;
    }
}

/*-------------------------------------------------------------*/
void RecordRequest(fd, buf, n)
    FD fd;
    unsigned char *buf;
    long n;
{
    register short request = IByte(&buf[0]);	/* request type */
    unsigned long wid = ILong(&buf[4]);

    if (RecordFile == 0) {
	/* open up the recording file if necessary, first time through */
	RecordFile = OpenFile(RecordFileName, "w");
    }
    switch (request) {
    case 1:			/* CreateWindow */
	if (!MatchCreateWinRec(buf))
	    req_to_ascii(fd, buf, RecordFile);
	break;

    case 4:			/* DestroyWindow */
	SetInferiorsDelflag(wid, 1);
	SetDelflag(wid, 1);
	break;

    case 5:			/* DestroySubWindows */
	SetInferiorsDelflag(wid, 1);
	break;
    }
}

/*-------------------------------------------------------------*/
/*
 *returns 1 if a window is found, 0 otherwise
 */
int MatchCreateWinRec(buf)
    unsigned char *buf;
{
    unsigned long wid, parent, mask;
    int depth, x, y, width, height;
    long i;

    depth = IByte(&buf[1]);
    wid = ILong(&buf[4]);
    parent = ILong(&buf[8]);
    x = IShort(&buf[12]);
    y = IShort(&buf[14]);
    width = IShort(&buf[16]);
    height = IShort(&buf[18]);
    mask = ILong(&buf[28]);

    for (i = 0; i < maxwtab; i++) {
	if (parent == wintab[i].parent &&
	    depth == wintab[i].depth &&
	    mask == wintab[i].mask &&
	    x == wintab[i].x &&
	    y == wintab[i].y &&
	    width == wintab[i].width &&
	    height == wintab[i].height &&
	    wintab[i].delflag != 0) {
	    /* 
	     * could match this window: means that a previously
	     * destroyed window has been created again.
	     * alias it to the original and reset the deleted flag
	     */
	    wintab[i].delflag = 0;
	    if (maxalias < MAXALIAS) {
		winalias[maxalias].wid = wid;
		winalias[maxalias].alias = wintab[i].wid;
		maxalias++;
	    } else {
		fprintf(stderr, "Max number of window aliases reached, increase MAXALIAS.\n");
	    }
	    return (1);
	}
    }
    /* 
     * Couldn't match window, add it int the table.
     * Set newid and newpart to current wid and parent, 
     * to match directly if we replay in the same session.
     */
    wintab[maxwtab].wid = wid;
    wintab[maxwtab].newid = wid;
    wintab[maxwtab].parent = parent;
    wintab[maxwtab].newpar = parent;
    wintab[maxwtab].depth = depth;
    wintab[maxwtab].x = x;
    wintab[maxwtab].y = y;
    wintab[maxwtab].width = width;
    wintab[maxwtab].height = height;
    wintab[maxwtab].mask = mask;
    maxwtab++;
    return (0);
}

/*-------------------------------------------------------------*/
void LoadWtab(filename)
    char *filename;
{
    char line[256];
    int lsize = 256;
    unsigned int wid, parent, m;
    int d, x, y, w, h;
    char header[20];

    PlaybackFile = fopen(PlaybackFileName, "r");

    while (fgets(line, lsize, PlaybackFile) && maxwtab <= MAXWTAB) {
	if (line[0] == 'X') {
	    if (9 != sscanf(line, "%s %x %x %x %d %d %d %d %x",
			  header, &wid, &parent, &d, &x, &y, &w, &h, &m))
		warn("Error decoding in LoadWtab");

	    wintab[maxwtab].wid = wid;
	    wintab[maxwtab].parent = parent;
	    wintab[maxwtab].depth = d;
	    wintab[maxwtab].x = x;
	    wintab[maxwtab].y = y;
	    wintab[maxwtab].width = w;
	    wintab[maxwtab].height = h;
	    wintab[maxwtab].mask = m;
	    maxwtab++;
	}
    }
    if (maxwtab == MAXWTAB) {
	warn("LoadWtab: Number max of XCreateWindow reached, increase MAXWTAB");
    }
    fclose(PlaybackFile);
    PlaybackFile = 0;
}

/*-------------------------------------------------------------*/
/* mainly a debugging aid. get the original window */
int OriginalWid(wid)
    unsigned long wid;
{
    long i;

    for (i = 0; i < maxwtab; i++)
	if (wintab[i].newid == wid)
	    return wintab[i].wid;
    return 0;
}
/*-------------------------------------------------------------*/
int NewWid(wid)
    unsigned long wid;
{
    long i;

    for (i = 0; i < maxwtab; i++)
	if (wintab[i].wid == wid)
	    return(wintab[i].newid ? wintab[i].newid : wid);

    if (wid != TheRootWindow) {
	fprintf(stderr, "NewWid: Couldn't match Window id %lx\n", wid);
    }
    return wid;

}
/*-------------------------------------------------------------*/
void SetNewParent(old, new)
    unsigned long old, new;
{
    long i;

    for (i = 0; i < maxwtab; i++)
	if (wintab[i].parent == old)
	    wintab[i].newpar = new;
}

/*-------------------------------------------------------------*/
void SetNewid(old, new)
    unsigned long old, new;
{
    long i;

    for (i = 0; i < maxwtab; i++)
	if (wintab[i].newid == old)
	    wintab[i].newid = new;
}

/*-------------------------------------------------------------*/
unsigned long GetWinAlias(wid)
    unsigned long wid;
{
    long i;

    for (i = 0; i < maxalias; i++)
	if (winalias[i].wid == wid)
	    return (winalias[i].alias);

    return (wid);
}

/*-------------------------------------------------------------*/
void SetInferiorsNewid(wid, new)
    unsigned long wid, new;
{
    long i;

    for (i = 0; i < maxwtab; i++) {
	if (wintab[i].newpar == wid) {
	    SetInferiorsNewid(wintab[i].wid, new);
	    wintab[i].newid = new;
	}
    }
}

/*-------------------------------------------------------------*/
void SetInferiorsDelflag(wid, value)
    unsigned long wid;
    char value;
{
    long i;

    for (i = 0; i < maxwtab; i++) {
	if (wintab[i].newpar == wid) {
	    SetInferiorsDelflag(wintab[i].wid, value);
	    wintab[i].delflag = value;
	}
    }
}

/*-------------------------------------------------------------*/
void SetDelflag(wid, value)
    unsigned long wid;
    char value;
{
    long i;

    for (i = 0; i < maxwtab; i++) {
	if (wintab[i].wid == wid) {
	    wintab[i].delflag = value;
	}
    }
}

/*--------------------------------------------------------------*/
static char *state_to_ascii(state, scr_state)
    unsigned long int state;
    char scr_state[13];
{
    int k;
    /* clear the state mask string */
    for (k = 0; k < 12; k++)
	scr_state[k] = '-';

    if (state & ShiftMask)
	scr_state[0] = 'S';
    if (state & LockMask)
	scr_state[1] = 'L';
    if (state & ControlMask)
	scr_state[2] = 'C';
    if (state & Mod1Mask)
	scr_state[3] = 'M';
    if (state & Button1Mask)
	scr_state[4] = '1';
    if (state & Button2Mask)
	scr_state[5] = '2';
    if (state & Button3Mask)
	scr_state[6] = '3';
    k = 7;
    if (state & Mod2Mask)
	scr_state[k++] = 'N';
    if (state & Mod3Mask)
	scr_state[k++] = 'O';
    if (state & Mod4Mask)
	scr_state[k++] = 'P';
    if (state & Mod5Mask)
	scr_state[k++] = 'Q';
    if (state & Button4Mask)
	scr_state[k++] = '4';
    if (state & Button5Mask)
	scr_state[k++] = '5';
    scr_state[k] = 0;
    return scr_state;
}

/*--------------------------------------------------------------*/
void event_to_ascii(fd, buf, file)
    FD fd;
    unsigned char *buf;
    FILE *file;
{
    struct timeval tv;
    Time ts;
    static Time last_rec_ts = 0L;
    register unsigned short evtype = IByte(&buf[0]);
    short evdetail = 0;
    Window evroot = 0;
    unsigned long evstate = 0;
    Bool evsame = 0;
    /*  Script variables  */
    int scr_x = 0, scr_y = 0, scr_xroot = 0, scr_yroot = 0;
    char scr_state[13];
    char *scr_type;
    int scr_spec, scr_win = 0, scr_subw = 0;
    unsigned long int scr_time;
    char *scr_spec_str;
    char spec_str[40];

    scr_type = 0;		/* by default, no event type */
    scr_spec_str = "0";

    /* Compute time stamp from the last recorded event */
    gettimeofday(&tv, 0);
    ts = tv.tv_sec * 1000;
    ts += tv.tv_usec / 1000;

    /* first event recorded, delai must be 0 */
    if (last_rec_ts == 0L)
	last_rec_ts = ts;

    scr_time = ts - last_rec_ts;
    last_rec_ts = ts;

    /* high-order bit means SendEvent generated */
    evtype &= 0x7f;

    switch (evtype) {
    case ButtonPress:
    case ButtonRelease:
    case KeyPress:
    case KeyRelease:
    case MotionNotify:
    case EnterNotify:
    case LeaveNotify:
	/* unpack the contents of the event fields. */
	evdetail = IByte(&buf[1]);
	/* scr_time   = ILong(&buf[4]); */
	evroot = ILong(&buf[8]);
	scr_win = ILong(&buf[12]);
	scr_subw = ILong(&buf[16]);
	evroot = GetWinAlias(evroot);
	scr_win = GetWinAlias(scr_win);
	scr_subw = GetWinAlias(scr_subw);
	scr_xroot = IShort(&buf[20]);
	scr_yroot = IShort(&buf[22]);
	scr_x = IShort(&buf[24]);
	scr_y = IShort(&buf[26]);
	evstate = IShort(&buf[28]);
	state_to_ascii(evstate, scr_state);
	evsame = IBool(&buf[30]);
	break;

    default:
	/* EMPTY */
	break;
    }

    /*  code the event type */
    switch (evtype) {
    case ButtonPress:
	scr_type = "BP";
	break;
    case ButtonRelease:
	scr_type = "BR";
	break;
    case KeyPress:
	scr_type = "KP";
	break;
    case KeyRelease:
	scr_type = "KR";
	break;
    case MotionNotify:
	scr_type = "MN";
	break;
    case EnterNotify:
	scr_type = "EN";
	break;
    case LeaveNotify:
	scr_type = "LN";
	break;
    }

    /* special handling per event type */
    switch (evtype) {
    case ButtonPress:
    case ButtonRelease:
	/*   snatch the button number  */
	scr_spec = evdetail;
	sprintf(spec_str, "%-7d", scr_spec);
	scr_spec_str = spec_str;

	/* Shift + Ctrl + Button 1 activates screen capture */
	if ((evtype == ButtonRelease) && (evstate & ShiftMask) && (evstate & ControlMask))
	    screen_capture();
	break;

    case KeyPress:
    case KeyRelease:
	{
	    /* 
	     * fake an event for calling XLookupString, 
	     * to get Xlib-compatible KeySym lookup.
	     */
	    XKeyEvent ev;
	    char buff[10];
	    KeySym keysym;

	    ev.type = evtype;
	    ev.display = Dpy;
	    ev.keycode = evdetail;
	    ev.state = evstate;
	    XLookupString(&ev, buff, 10, &keysym, 0);
	    scr_spec_str = XKeysymToString(keysym);
	}
	break;

    case MotionNotify:
	scr_spec_str = evdetail ? "Hint   " : "Nohint ";
	break;

    case EnterNotify:
    case LeaveNotify:
	{
	    char *detail = "";
	    char *mode = "";
	    char evmode = IByte(&buf[30]);
	    char str[12];

	    /* handle detail and mode */
	    switch (evdetail) {
	    case 0:
		detail = "A";
		break;
	    case 1:
		detail = "V";
		break;
	    case 2:
		detail = "I";
		break;
	    case 3:
		detail = "N";
		break;
	    case 4:
		detail = "C";
		break;
	    }

	    switch (evmode) {
	    case 0:
		mode = "N";
		break;
	    case 1:
		mode = "G";
		break;
	    case 2:
		mode = "U";
		break;
	    case 3:
		mode = "W";
		break;
	    }

	    sprintf(str, "%s-%s", detail, mode);
	    scr_spec_str = str;
	}
    }				/* end switch (ev.type)   */

    if (scr_type)
	fprintf(file, "%s %-9s %s %d %8.8x %8.8x %8.8lx %4d %4d %4d %4d %7lu %4ld\n",
	    scr_type, scr_spec_str, scr_state, evsame, scr_win, scr_subw,
		evroot, scr_xroot, scr_yroot, scr_x, scr_y, scr_time, NumberFD(fd));
}

/*-------------------------------------------------------------*/
int ascii_to_event_type(line)
    char *line;
{
    char scr_type[20];
    int type = 0;
    int error = 0;
    int count = 0;

    count = sscanf(line, "%2s", scr_type);
    if (count != 1)
	error = 1;

    if (error)
	return 0;

    switch (scr_type[0]) {
    case 'B':
    case 'b':
	switch (scr_type[1]) {
	case 'P':
	case 'p':
	    type = ButtonPress;
	    break;
	case 'R':
	case 'r':
	    type = ButtonRelease;
	    break;
	default:
	    error = 1;
	    break;
	}
	if (error)
	    return 0;
	break;

    case 'K':
    case 'k':
	switch (scr_type[1]) {
	case 'P':
	case 'p':
	    type = KeyPress;
	    break;
	case 'R':
	case 'r':
	    type = KeyRelease;
	    break;
	default:
	    error = 1;
	    break;
	}
	if (error)
	    return 0;
	break;

    case 'M':
    case 'm':
	switch (scr_type[1]) {
	case 'N':
	case 'n':
	    type = MotionNotify;
	    break;
	default:
	    error = 1;
	    break;
	}
	if (error)
	    return 0;
	break;

    case 'E':
    case 'e':
	switch (scr_type[1]) {
	case 'N':
	case 'n':
	    type = EnterNotify;
	    break;
	default:
	    error = 1;
	    break;
	}
	if (error)
	    return 0;
	break;

    case 'L':
    case 'l':
	switch (scr_type[1]) {
	case 'N':
	case 'n':
	    type = LeaveNotify;
	    break;
	default:
	    error = 1;
	    break;
	}
	if (error)
	    return 0;
	break;

    case '#':	/* comment */
    case 'X':	/* XCreateWindow request */
	type = 0;
	break;
    }
    return type;
}

/*-------------------------------------------------------------*/
/* record XcreateWindow request to trace window chaining */
void req_to_ascii(fd, buf, file)
    FD fd;
    unsigned char *buf;
    FILE *file;
{
    unsigned long wid, parent, mask;
    int depth, x, y, width, height;

    depth = IByte(&buf[1]);
    wid = ILong(&buf[4]);
    parent = ILong(&buf[8]);
    x = IShort(&buf[12]);
    y = IShort(&buf[14]);
    width = IShort(&buf[16]);
    height = IShort(&buf[18]);
    mask = ILong(&buf[28]);

    fprintf(file, "%s %8.8lx %8.8lx %2.2x %4d %4d %4d %4d %8.8lx\n",
	 "XCreateWindow", wid, parent, depth, x, y, width, height, mask);
}

/*-------------------------------------------------------------*/
/* 
 * MatchWtab is called during the replay to match windows created
 * during replay against those created during recording.
 * If the request is CreateWindow, we try to find the corresponding
 * window in the recorded file (contained in wintab).
 * If the request is DestroyWindow or DestroySubWindows, we must remove
 * the corresponding wid from the table to allow a later CreateWindow of
 * the same window to match correctly.
 */

void MatchWtab(fd, buf)
    FD fd;
    unsigned char *buf;
{
    short request = IByte(&buf[0]);
    unsigned long wid, parent, mask;
    int depth, x, y, width, height;
    long i;
    static int first = 1;
    int found = 0;

    if (request == 4) {		/* DestroyWindow request */
	wid = ILong(&buf[4]);
	/* SetInferiorsNewid(wid, 0L); */
	SetNewid(wid, 0L);
	return;
    }
    if (request == 5) {		/* DestroySubWindows request */
	wid = ILong(&buf[4]);
	/* SetInferiorsNewid(wid, 0L); */
	return;
    }
    if (request != 1)		/* CreateWindow request */
	return;

    depth = IByte(&buf[1]);
    wid = ILong(&buf[4]);
    parent = ILong(&buf[8]);
    x = IShort(&buf[12]);
    y = IShort(&buf[14]);
    width = IShort(&buf[16]);
    height = IShort(&buf[18]);
    mask = ILong(&buf[28]);

    if (first == 1) {
	first = 0;
	/* treat the case of root window (parent of the 1st XCreateWindow) */
	SetNewParent(wintab[0].parent, parent);
    }
    /* 
     * First we try to find a similar window in the recorded file
     * with the same hierarchy.
     */
    for (i = 0; i < maxwtab; i++) {
	/* skip windows already allocated */
	if (wintab[i].newid != 0)
	    continue;

	if (parent == wintab[i].newpar &&
	    depth == wintab[i].depth &&
	    mask == wintab[i].mask &&
	    x == wintab[i].x &&
	    y == wintab[i].y &&
	    width == wintab[i].width &&
	    height == wintab[i].height) {
	    wintab[i].newid = wid;
	    SetNewParent(wintab[i].wid, wintab[i].newid);
	    found = 1;
	    return;
	}
    }

    /* 
     * If previous search doesn't match, assume it is a top window
     * (don't try to match parent), do the search again.
     */
    for (i = 0; i < maxwtab; i++) {
	/* skip windows already allocated */
	if (wintab[i].newid != 0)
	    continue;
	if (depth == wintab[i].depth &&
	    mask == wintab[i].mask &&
	    x == wintab[i].x &&
	    y == wintab[i].y &&
	    width == wintab[i].width &&
	    height == wintab[i].height) {
	    wintab[i].newid = wid;
	    SetNewParent(wintab[i].parent, parent);
	    found = 1;
	    return;
	}
    }
    warn("if nothing else helps.....\n");
    /* 
     * If previous search doesn't match, assume it is a top window
     * (don't try to match parent), do the search again.
     */
    for (i = 0; i < maxwtab; i++) {
	/* skip windows already allocated */
	if (wintab[i].newid != 0)
	    continue;

	if (parent == wintab[i].newpar &&
	    depth == wintab[i].depth &&
	    mask == wintab[i].mask &&
	    width == wintab[i].width &&
	    height == wintab[i].height) {
	    wintab[i].newid = wid;
	    SetNewParent(wintab[i].wid, wid);
	    found = 1;
	    return;
	}
    }

    /*
     * If not found here, we got a problem.
     */
    fprintf(stderr, "Window not matched %lx %lx %d %d %d %d\n",
	    wid, parent, x, y, width, height);

}

/*-------------------------------------------------------------*/
void ReplaceWid(buf, offset)
    unsigned char *buf;
    int offset;
{
    int wid, newid;

    wid = ILong(&buf[offset]);
    /* LM */
    if (wid == 0)
	return;
    newid = NewWid(wid);
    OLong(&buf[offset], newid);
}

/*-------------------------------------------------------------*/
int ReadSynthEvent(line)
    char *line;
{
    char str_type[3];
    char str_state[14];
    char str_spec[40];
    unsigned long int time;
    int x, y, xroot, yroot, same, fd;
    Window win, subw, rootw;
    char *p;
    static int first_debug = 0;

    if (13 != sscanf(line, "%s %s %s %d %lx %lx %lx %d %d %d %d %ld %u",
	       str_type, str_spec, str_state, &same, &win, &subw, &rootw,
		     &xroot, &yroot, &x, &y, &time, &fd))
	return (1);

    synth_ev.rootx = xroot;
    synth_ev.rooty = yroot;
    synth_ev.x = x;
    synth_ev.y = y;
    synth_ev.rootw = rootw;
    synth_ev.win = win;
    synth_ev.subw = subw;
    synth_ev.fd = FDClientFromNumber(fd);

    if (synth_ev.fd == -1) {
	fprintf(stderr,
		"Playback Error. No such client %d in current session\n", fd);
	return 1;
    }
    synth_ev.time = time;
    synth_ev.same = same;
    synth_ev.type = ascii_to_event_type(str_type);

    switch (synth_ev.type) {
    case ButtonPress:
    case ButtonRelease:
	/* get the button number */
	if (first_debug == 0) {
	    first_debug = 1;
	}
	synth_ev.detail = atoi(str_spec);
	break;

    case KeyPress:
    case KeyRelease:
	/* get the keycode number using tables from the server */
	synth_ev.detail = XKeysymToKeycode(Dpy, XStringToKeysym(str_spec));
	break;

    case MotionNotify:
	if (str_spec[0] == 'H')
	    synth_ev.detail = 1;
	else
	    synth_ev.detail = 0;
	break;

    case EnterNotify:
    case LeaveNotify:
	switch (str_spec[0]) {
	case 'A':
	    synth_ev.detail = 0;
	    break;
	case 'V':
	    synth_ev.detail = 1;
	    break;
	case 'I':
	    synth_ev.detail = 2;
	    break;
	case 'N':
	    synth_ev.detail = 3;
	    break;
	case 'C':
	    synth_ev.detail = 4;
	    break;
	default:
	    synth_ev.detail = 0;
	    break;
	}
	switch (str_spec[2]) {
	case 'N':
	    synth_ev.same = 0;
	    break;
	case 'G':
	    synth_ev.same = 1;
	    break;
	case 'U':
	    synth_ev.same = 2;
	    break;
	case 'W':
	    synth_ev.same = 3;
	    break;
	default:
	    synth_ev.same = 0;
	    break;
	}
	break;
    }

    synth_ev.state = 0;
    for (p = str_state; *p; p++) {
	switch (*p) {
	case 'S':
	    synth_ev.state |= ShiftMask;
	    break;
	case 'L':
	    synth_ev.state |= LockMask;
	    break;
	case 'C':
	    synth_ev.state |= ControlMask;
	    break;
	case 'M':
	    synth_ev.state |= Mod1Mask;
	    break;
	case 'N':
	    synth_ev.state |= Mod2Mask;
	    break;
	case 'O':
	    synth_ev.state |= Mod3Mask;
	    break;
	case 'P':
	    synth_ev.state |= Mod4Mask;
	    break;
	case 'Q':
	    synth_ev.state |= Mod5Mask;
	    break;
	case '1':
	    synth_ev.state |= Button1Mask;
	    break;
	case '2':
	    synth_ev.state |= Button2Mask;
	    break;
	case '3':
	    synth_ev.state |= Button3Mask;
	    break;
	case '4':
	    synth_ev.state |= Button4Mask;
	    break;
	case '5':
	    synth_ev.state |= Button5Mask;
	    break;
	}
    }
    OByte(&synth_ev.evbuf[0], synth_ev.type);
    OByte(&synth_ev.evbuf[1], synth_ev.detail);
    OLong(&synth_ev.evbuf[8], synth_ev.rootw);
    OLong(&synth_ev.evbuf[12], synth_ev.win);
    OLong(&synth_ev.evbuf[16], synth_ev.subw);
    OShort(&synth_ev.evbuf[20], synth_ev.rootx);
    OShort(&synth_ev.evbuf[22], synth_ev.rooty);
    OShort(&synth_ev.evbuf[24], synth_ev.x);
    OShort(&synth_ev.evbuf[26], synth_ev.y);
    OShort(&synth_ev.evbuf[28], synth_ev.state);
    OByte(&synth_ev.evbuf[30], synth_ev.same);
    return 0;
}

/*-------------------------------------------------------------*/
/* Screen capture function, activated by a keyboard sequence
 * during recording, and automatically reactivated during replay.
 * (this is really unstable at this moment).
 */
void screen_capture()
{
    static int ncap = 0;	/* increment this number at each capture */
    char cap_cmd[256];
    extern char *ScopeHost;
    extern int ScopePort;
    char this_disp[256];

    /* build "This" display name, so capture tool is controled by xlab */
    if (ncap == 0)
	sprintf(this_disp, "%s:%d.0", ScopeHost, ScopePort - 6000);

    /* build capture command */
    sprintf(cap_cmd, "xwd -display %s -out xlab_cap.%d &", this_disp, ncap++);
    system(cap_cmd);
}
