/*
 * $Id: scope.c,v 1.4 1998/07/24 11:59:17 marc Exp $
 *
 *  A script program to store and replay X input events
 * 
 *  Adapted from "xscope"
 *      James Peterson, 1988
 *      (c) Copyright MCC, 1988
 *      (c) Copyright Oliver Jones, 1993
 *      (c) Copyright Marc Vertes 1996, 1997, 1998
 *
 */
#include <stdio.h>
#include <stdlib.h>
#include <sys/uio.h>		/* for struct iovec, used by socket.h */
#include <sys/socket.h>		/* for AF_INET, SOCK_STREAM, ... */
#include <sys/ioctl.h>		/* for FIONCLEX, FIONBIO, ... */
#ifdef SVR4
#include <sys/filio.h>
#endif
#include <fcntl.h>
#include <netinet/in.h>		/* struct sockaddr_in */
#include <netdb.h>		/* struct servent * and struct hostent * */
#include <errno.h>		/* for EINTR, EADDRINUSE, ... */
extern int errno;

typedef struct TypeDef *TYPE;

#include "scope.h"
#include "table11.h"
#include "common.h"
#include "server.h"
#include "record.h"
#include "y.tab.h"

/*
 * we need a local X display connection to the server-under-test
 * to allow various queries to go through.
 * Hence, include the X definitions.
 */
#include <X11/Xlib.h>
#include <X11/Xutil.h>

short debuglevel;
short Verbose;			/* quiet (0) or increasingly verbose  ( > 0) */
int ScopePort;
char *ScopeHost;

/* path name of file to record, if any */
char RecordFileName[MAXPATHLEN];

/* FILE id of file name to record, if any */
FILE *RecordFile = 0;

/* path name of file to play back, if any */
char PlaybackFileName[MAXPATHLEN];

/* FILE id of file name to play back, if any */
FILE *PlaybackFile = 0;

/* user requested playback */
#define PLAYBACK_INACTIVE 0
#define PLAYBACK_PENDING 1
#define PLAYBACK_ACTIVE 2
#define PLAYBACK_COMPLETE 3
long Playback = PLAYBACK_INACTIVE;

/* non-zero if user requested recording  */
/* Record and Playback cannot both be nonzero */
long Record = 0;

/* number of seconds to wait after starting program to start playback */
long Wait = 15;

/* time dilation in percentage ... 200 means speed up x2 */
long TimeDilation = 100;

/* double click time in milliseconds */
long DoubleClickTime = 250;

/* grab timeout in seconds */
unsigned long int GrabTimeout = 30;

/* display connection */
Display *Dpy = 0;

/* playback error count */
unsigned long int Errorcount = 0;

#define DefaultPort 6000

char ServerHostName[255];
long ServerBasePort = DefaultPort;
long ServerInPort = 1;
long ServerOutPort = 0;
long ServerDisplay = 0;
char ScopeEnabled = 0;
char HandleSIGUSR1 = 0;

/*----------------------------------------------------------------------*/
short GetServerport()
{
    short port;

    enterprocedure("GetServerport");

    port = ServerBasePort + ServerOutPort + ServerDisplay;
    debug(4, (stderr, "Server service is on port %d\n", port));
    return (port);
}

/*----------------------------------------------------------------------*/
short GetScopePort()
{
    short port;

    enterprocedure("GetScopePort");

    port = ServerBasePort + ServerInPort + ServerDisplay;
    debug(4, (stderr, "scope service is on port %d\n", port));
    return (port);
}

/*----------------------------------------------------------------------*/
void Usage(xscript)
    char *xscript;
{
    fprintf(stderr, "Usage: %s [-r<script-file-to-record>]\n", xscript);
    fprintf(stderr, "               [-p<script-file-to-play-back>]\n");
    fprintf(stderr, "               [-w<seconds-to-wait-before-playback>]\n");
    fprintf(stderr, "               [-c<double-click-time>]\n");
    fprintf(stderr, "               [-t<time-contraction-percentage>]\n");
    fprintf(stderr, "               [-h<server-host>]\n");
    fprintf(stderr, "               [-i<in-port>]\n");
    fprintf(stderr, "               [-o<out-port>]\n");
    fprintf(stderr, "               [-d<display-number>]\n");
    fprintf(stderr, "               [-v<n>]  -- verbose output\n");
    fprintf(stderr, "               [-q]  -- quiet output\n");
    fprintf(stderr, "               [-D<debug-level>]\n");
    fprintf(stderr, "               [-S<n>] -- start/stop on SIGUSR1\n");
    fprintf(stderr, "               [-G<timeout>] -- grab timeout in seconds\n");

    fprintf(stderr, "To record a session, do this:\n");
    fprintf(stderr, "    (1)  setenv DISPLAY `hostname`:0\n");
    fprintf(stderr, "    (2)  %s -rscript-file&\n", xscript);
    fprintf(stderr, "    (3)  setenv DISPLAY `hostname`:1\n");
    fprintf(stderr, "    (4)  run the program under test, and operate it.\n");
    fprintf(stderr, "To play back a session at 50%% speed, do this:\n");
    fprintf(stderr, "    (1)  setenv DISPLAY `hostname`:0\n");
    fprintf(stderr, "    (2)  %s -t50 -pscript-file&\n", xscript);
    fprintf(stderr, "    (3)  setenv DISPLAY `hostname`:1\n");
    fprintf(stderr, "    (4)  run the program under test, sit back, and watch.\n");
    exit(1);
}


/*----------------------------------------------------------------------*/
static char *xscript = NULL;

void ScanArgs(argc, argv)
    int argc;
    char **argv;
{
    xscript = argv[0];

    Verbose = 0 /* default verbose-ness level */ ;

    /* Scan argument list */
    while (--argc > 0) {
	++argv;
	if (**argv == '-')
	    switch (*++*argv) {
		/*
		 * debug levels:
		 *  2 - trace each procedure entry
		 *  4 - I/O, connections
		 *  8 - Scope internals
		 *  16 - Message protocol
		 *  32 - 64 - malloc
		 *  128 - 256 - really low level
		 */
	    case 'D':
		debuglevel = atoi(++*argv);
		if (debuglevel == 0)
		    debuglevel = 255;
		debuglevel |= 1;
		Verbose = 7;
		debug(1, (stderr, "debuglevel = %d\n", debuglevel));
		break;

	    case 'S':
		HandleSIGUSR1 = 1;
		ScopeEnabled = atoi(++*argv);
		break;

	    case 'G':
		GrabTimeout = atoi(++*argv);
		if (GrabTimeout == 0)
		    GrabTimeout = 30;
		break;

	    case 'q':		/* quiet mode */
		Verbose = 0;
		debug(1, (stderr, "Verbose = %d\n", Verbose));
		break;

	    case 'v':		/* verbose mode */
		Verbose = atoi(++*argv);
		debug(1, (stderr, "Verbose = %d\n", Verbose));
		break;

	    case 'o':
		ServerOutPort = atoi(++*argv);
		if (ServerOutPort <= 0)
		    ServerOutPort = 0;
		debug(1, (stderr, "ServerOutPort = %ld\n", ServerOutPort));
		break;

	    case 'd':
		ServerDisplay = atoi(++*argv);
		if (ServerDisplay <= 0)
		    ServerDisplay = 0;
		debug(1, (stderr, "ServerDisplay=%ld\n", ServerDisplay));
		break;

	    case 'i':
		ServerInPort = atoi(++*argv);
		if (ServerInPort <= 0)
		    ServerInPort = 0;
		debug(1, (stderr, "ServerInPort = %ld\n", ServerInPort));
		break;

	    case 'h':
		if (++*argv != NULL && **argv != '\0') {
		    (void) strcpy(ServerHostName, OfficialName(*argv));
		    debug(1, (stderr, "ServerHostName=%s\n", ServerHostName));
		} else {
		    /* -h by itself means print usage message */
		    Usage(xscript);
		}
		break;

	    case 'r':
		if (Playback != PLAYBACK_INACTIVE) {
		    fprintf(stderr, "Cannot record and play back in same session.\n");
		    Usage(xscript);
		}
		if (++*argv != NULL && **argv != '\0')
		    (void) strcpy(RecordFileName, *argv);
		debug(1, (stderr, "RecordFileName=%s\n", RecordFileName));

		/* ensure we can open record file correctly */
		RecordFile = fopen(RecordFileName, "w");
		if (RecordFile == 0) {
		    perror(RecordFileName);
		    exit(1);
		}
		fclose(RecordFile);
		RecordFile = 0;
		Record = 1;
		break;

	    case 'p':
		if (Record) {
		    fprintf(stderr, "Cannot record and play back in same session.\n");
		    Usage(xscript);
		}
		if (++*argv != NULL && **argv != '\0')
		    (void) strcpy(PlaybackFileName, *argv);
		debug(1, (stderr, "PlaybackFile<Name=%s\n", PlaybackFileName));

		/* ensure we can open playback file correctly */
		PlaybackFile = fopen(PlaybackFileName, "r");
		if (PlaybackFile == 0) {
		    perror(PlaybackFileName);
		    exit(1);
		}
		fclose(PlaybackFile);
		PlaybackFile = 0;
		Playback = PLAYBACK_PENDING;
		/* read playbackfile to to extract Windows info */
		LoadWtab("PlaybackFileName");
		break;

	    case 'w':
		Wait = atoi(++*argv);
		if (Wait <= 0)
		    Wait = 10;
		debug(1, (stderr, "Wait = %ld\n", Wait));
		break;

	    case 't':
		TimeDilation = atoi(++*argv);
		if (TimeDilation <= 0)
		    TimeDilation = 100;
		debug(1, (stderr, "TimeDilation = %ld\n", TimeDilation));
		break;

	    case 'c':
		DoubleClickTime = atoi(++*argv);
		if (DoubleClickTime <= 0)
		    DoubleClickTime = 250;
		debug(1, (stderr, "DoubleClickTime = %ld\n", DoubleClickTime));
		break;

	    default:
		fprintf(stderr, "Unknown option %c\n", **argv);
		Usage(xscript);
		break;

	} else {
	    /* file argument to scope -- error */
	    Usage(xscript);
	}
    }

    /* check for different port numbers or different machines */
    if (ServerInPort == ServerOutPort)
	if (ServerHostName[0] == '\0') {
	    fprintf(stderr, "Can't have xlab on same port as server (%ld)\n",
		    ServerInPort);
	    Usage(xscript);
	}
}


/*----------------------------------------------------------------------*/
int main(argc, argv)
    int argc;
    char **argv;
{
    /* grab input args */
    ScanArgs(argc, argv);

    /* initialize cross table of FDinfo */
    initFDbynum();

    /* initialize the file-descriptor-handling system */
    InitializeFD();

    /* initialize the symbol tables, etc, for X */
    InitializeX11();

    /* set up input stuff */
    SetUpStdin();
    SetUpXDisplay();
    SetUpConnectionSocket(GetScopePort());
    SetSignalHandling();

    MainLoop();
    return (0);
}


/*----------------------------------------------------------------------*/
char *mystring;

void ReadStdin(fd)
    FD fd;
{
    long n;
    char in_buf[2048];
    extern FILE *yyin;
    extern int yyparse();
    extern int yyrestart();

    enterprocedure("ReadStdin");
    n = read(fd, in_buf, 2048);
    in_buf[n] = 0;
    debug(4, (stderr, "read %ld bytes from stdin\n", n));
    if (n == 0) {
	NotUsingFD(fileno(stdin));
	CreateTimer(2000, (void (*)()) SetUpStdin, (void *) fd);
	return;
    }
    mystring = in_buf;
    yyparse();
    yyrestart(yyin);
}

/*----------------------------------------------------------------------*/
void SetUpStdin()
{
    enterprocedure("SetUpStdin");
    UsingFD(fileno(stdin), (int (*)()) ReadStdin);
}

/*----------------------------------------------------------------------*/
void ReportAndExit()
{

    if (Errorcount > 0 && Playback != PLAYBACK_INACTIVE) {
	fprintf(stderr, "xlab: Playback summary\n");
	fprintf(stderr, "  %ld errors encountered during playback\n", Errorcount);
	exit(2);
    }
    exit(0);
}

/*----------------------------------------------------------------------*/
/*
   here is where we would add code to allow control from
   the keyboard.  We would want to read a command and
   interpret it.  Possibilties:

   (a) verbose level setting
   (b) reset time
   (c) save X requests to a file.
   (d) replay X requests from a file.
   (e) allow fake events, errors to be generated.
 */

/*
   xscope is really meant to look at one client at a time.  However,
   it can easily handle multiple clients and servers.  To do so,
   we need to have a pair of FDs: one for the client and one for the
   server for that client.  If either goes away, so does the other.
   We need to be able to identify the other FD of a pair, so that if
   we get input from one, we can write it to the other.
 */

struct fdinfo {
    Boolean Server;
    long ClientNumber;
    FD pair;
};

static long ClientNumber = 0;
static long ClientsConnected = 0;
struct fdinfo FDinfo[StaticMaxFD];

/* table to retrieve the client from the ClientNumber field of FDinfo */
FD FDbynum[StaticMaxFD];

/*----------------------------------------------------------------------*/
void SetUpPair(client, server)
    FD client;
    FD server;
{
    if (client >= 0) {
	ClientNumber += 1;
	ClientsConnected += 1;
	FDinfo[client].Server = false;
	FDinfo[client].pair = server;
	FDinfo[client].ClientNumber = ClientNumber;
	if (server >= 0) {
	    FDinfo[server].Server = true;
	    FDinfo[server].pair = client;
	    FDinfo[server].ClientNumber = FDinfo[client].ClientNumber;
	    FDbynum[ClientNumber] = server;
	}
    } else if (server >= 0) {
	(void) close(server);
	NotUsingFD(server);
    }
}

/*----------------------------------------------------------------------*/
void CloseConnection(fd)
    FD fd;
{
    debug(4, (stderr, "close %d and %d\n", fd, FDPair(fd)));
    StopClientConnection(ServerHalf(fd));
    StopServerConnection(ClientHalf(fd));

    (void) close(fd);
    NotUsingFD(fd);
    (void) close(FDPair(fd));
    NotUsingFD(FDPair(fd));
    ClientsConnected -= 1;
}

/*----------------------------------------------------------------------*/
FD FDPair(fd)
    FD fd;
{
    return (FDinfo[fd].pair);
}

/*----------------------------------------------------------------------*/
FD ClientHalf(fd)
    FD fd;
{
    if (FDinfo[fd].Server)
	return (FDinfo[fd].pair);
    return (fd);
}

/*----------------------------------------------------------------------*/
FD ServerHalf(fd)
    FD fd;
{
    if (FDinfo[fd].Server)
	return (fd);
    return (FDinfo[fd].pair);
}

/*----------------------------------------------------------------------*/
char *ClientName(fd)
    FD fd;
{
    static char name[12];

    (void) sprintf(name, " %ld", FDinfo[fd].ClientNumber);
    return (name);
}

/*----------------------------------------------------------------------*/
void initFDbynum()
{
    long i;
    for (i = 0; i < StaticMaxFD; i++)
	FDbynum[i] = -1;
}

/*----------------------------------------------------------------------*/
extern struct WinTab wintab[MAXWTAB];
extern struct WinAlias winalias[MAXWTAB];

void initwintab()
{
    int i;

    for (i = 0; i < MAXWTAB; i++) {
	wintab[i].wid = 0;
	wintab[i].parent = 0;
	wintab[i].newid = 0;
	wintab[i].newpar = 0;
	wintab[i].x = -1;
	wintab[i].y = -1;
	wintab[i].width = -1;
	wintab[i].height = -1;
	wintab[i].delflag = 0;
	winalias[i].wid = 0;
	winalias[i].alias = 0;
    }
}

/*----------------------------------------------------------------------*/
FD FDClientFromNumber(clientNb)
    long clientNb;
{
    return FDbynum[clientNb];
    /*
       long i;

       for (i = 0; i < StaticMaxFD; i++) {
       if ((FDinfo[i].ClientNumber == clientNb) && (FDinfo[i].Server == false))
       return i;
       }
       return -1;
     */
}

/*----------------------------------------------------------------------*/
long NumberFD(fd)
    FD fd;
{
    return FDinfo[fd].ClientNumber;
}


/*----------------------------------------------------------------------*/
/* It's possible for a write syscall to fail with
   EWOULDBLOCK.  In this case, we want to do a select
   until the file descriptor is writeable once again  */

static void WaitUntilWriteable(fd)
    FD fd;
{
    long int rfds, wfds, xfds;
    short int nfds;
    /* assert (fd < 32); */
    rfds = 0;
    wfds = 1 << fd;
    xfds = 0;
    debug(128, (stderr, "select %d, wfds = 0%lo\n", fd + 1, wfds));
    nfds = select(fd + 1, (fd_set *) & rfds, (fd_set *) & wfds, (fd_set *) & xfds,
		  (struct timeval *) NULL);
    debug(128, (stderr, "select nfds = 0%o, wfds 0%lo\n", nfds, wfds));
}


/*----------------------------------------------------------------------*/
/* when we get data from a client, we read it in, copy it to the
 * server for this client, then dump it to the client. Note, we don't
 * have to have a server, if there isn't one. 
 */
void DataFromClient(fd)
    FD fd;
{
    unsigned char buf[2048];
    long n;
    FD ServerFD;

    /*MV enterprocedure("DataFromClient"); */
    n = read(fd, (char *) buf, 2048);
    debug(4, (stderr, "read %ld bytes from Client%s\n", n, ClientName(fd)));
    if (n < 0 && errno == ECONNRESET)
	n = 0;
    if (n < 0) {
	PrintTime();
	perror("Client --> read error:");
	CloseConnection(fd);
	return;
    }
    if (n == 0) {
	if (ScopeEnabled) {
	    PrintTime();
	    fprintf(stdout, "Client%s --> EOF\n", ClientName(fd));
	}
	if (RecordFile) {
	    fflush(RecordFile);
	    if (ClientsConnected == 0) {
		fclose(RecordFile);
		RecordFile = 0;
	    }
	}
	CloseConnection(fd);

/*
   if ((Record || (Playback != PLAYBACK_INACTIVE)) && (ClientsConnected == 0))
   ReportAndExit ();
 */
	return;
    }
    ServerFD = FDPair(fd);
    if (ServerFD < 0) {
	ServerFD = ConnectToServer(false);
	SetUpPair(fd, ServerFD);
    }
    /* write bytes from client to server, allow for server to fail */
    if (ServerFD >= 0) {
	long BytesToWrite = n;
	unsigned char *p = buf;

	while (BytesToWrite > 0) {
	    int n1 = write(ServerFD, (char *) p, (int) BytesToWrite);
	    debug(4, (stderr, "write %d bytes to Server%s\n", n1, ClientName(fd)));
	    if (n1 < 0 && (errno == EWOULDBLOCK || errno == EAGAIN)) {
		if (errno == EAGAIN)
		    sleep(1);
		WaitUntilWriteable(ServerFD);
		n1 = 0;
	    }
	    if (n1 >= 0) {
		BytesToWrite -= n1;
		p += n1;
	    } else {
		perror("Error on write to Server");
		CloseConnection(fd);
		BytesToWrite = 0;
	    }
	}
    }
    /* also report the bytes to standard out */
    ReportFromClient(fd, buf, n);
}

/*----------------------------------------------------------------------*/
/* push bytes from server out to client */
void WriteToClient(fd, buf, n)
    FD fd;
    unsigned char *buf;
    int n;
{
    FD ClientFD = FDPair(fd);
    long BytesToWrite = n;
    unsigned char *p = buf;

    while (BytesToWrite > 0) {
	int n1 = write(ClientFD, (char *) p, (int) BytesToWrite);
	debug(4, (stderr, "write %d bytes to Client%s\n", n1, ClientName(fd)));

	if (n1 < 0 && (errno == EWOULDBLOCK || errno == EAGAIN)) {
	    if (errno == EAGAIN)
		sleep(1);
	    WaitUntilWriteable(ClientFD);
	    n1 = 0;
	}
	if (n1 >= 0) {
	    BytesToWrite -= n1;
	    p += n1;
	} else {
	    perror("Error on write to Client");
	    CloseConnection(fd);
	    BytesToWrite = 0;
	}
    }
}

/*----------------------------------------------------------------------*/
/* similar situation for the server, but note that if there is no client,
 *  we close the connection down -- don't need a server with no client. 
 */
void DataFromServer(fd)
    FD fd;
{
    unsigned char buf[2048];
    long n;
    FD ClientFD;

    /*MV enterprocedure("DataFromServer"); */
    n = read(fd, (char *) buf, 2048);
    debug(4, (stderr, "read %ld bytes from Server%s\n", n, ClientName(fd)));
    if (n < 0) {
	PrintTime();
	perror("read error <- Server");
	CloseConnection(fd);
	return;
    }
    if (n == 0) {
	if (ScopeEnabled) {
	    PrintTime();
	    fprintf(stdout, "EOF <-- Server%s\n", ClientName(fd));
	}
	if (RecordFile) {
	    fflush(RecordFile);
	    if (ClientsConnected == 0) {
		fclose(RecordFile);
		RecordFile = 0;
	    }
	}
	CloseConnection(fd);

/*
   if ((Record || (Playback != PLAYBACK_INACTIVE)) && (ClientsConnected == 0))
   ReportAndExit ();
 */
	return;
    }
    ClientFD = FDPair(fd);
    if (ClientFD < 0) {
	CloseConnection(fd);
	return;
    }
    if (Playback == PLAYBACK_INACTIVE) {
	/*
	 * if we're not playing stuff back from a file,
	 * write bytes from server to client, allow for client to fail.
	 * Note...if we ARE playing server stuff back, we need to 
	 *   crack the protocol first, because we don't pass throug
	 *   all events from the server; only those which don't
	 *   come from the playback file.
	 */
	WriteToClient(fd, buf, n);
    }
    /* also report the bytes to standard out */
    ReportFromServer(fd, buf, n);
}



/*----------------------------------------------------------------------*/
/*     Create New Connection to a client program and to Server  */

#ifndef linux
static int ON = 1 /* used in ioctl */ ;
#endif

void NewConnection(fd)
    FD fd;
{
    FD ServerFD = -1;
    FD ClientFD = -1;

    ClientFD = ConnectToClient(fd);
    ServerFD = ConnectToServer(true);
    SetUpPair(ClientFD, ServerFD);
}

/*----------------------------------------------------------------------*/
FD ConnectToClient(ConnectionSocket)
    FD ConnectionSocket;
{
    FD ClientFD;
    struct sockaddr_in from;
    int len = sizeof(from);

    enterprocedure("ConnectToClient");

    ClientFD = accept(ConnectionSocket, (struct sockaddr *) &from, &len);
    debug(4, (stderr, "Connect To Client: FD %d\n", ClientFD));
    if (ClientFD < 0 && errno == EWOULDBLOCK) {
	debug(4, (stderr, "Almost blocked accepting FD %d\n", ClientFD));
	panic("Can't connect to Client");
    }
    if (ClientFD < 0) {
	debug(4, (stderr, "NewConnection: error %d\n", errno));
	panic("Can't connect to Client");
    }
    UsingFD(ClientFD, (int (*)()) DataFromClient);
#ifdef FD_CLOEXEC
    (void) fcntl(ClientFD, F_SETFD, FD_CLOEXEC);
#else
    (void) ioctl(ClientFD, FIOCLEX, 0);
#endif
    /* ultrix reads hang on Unix sockets, hpux reads fail */
#if defined(O_NONBLOCK) && (!defined(ultrix) && !defined(hpux))
    (void) fcntl(ClientFD, F_SETFL, O_NONBLOCK);
#else
#ifdef FIOSNBIO
    ioctl(ClientFD, FIOSNBIO, &ON);
#else
    (void) fcntl(ClientFD, F_SETFL, FNDELAY);
#endif
#endif
    StartClientConnection(ClientFD);
    return (ClientFD);
}

/*----------------------------------------------------------------------*/
FD ConnectToServer(report)
    Boolean report;
{
    FD ServerFD;
    struct sockaddr_in sin;
    struct hostent *hp;
    short port;
    int reuse_addr = 1;
#ifndef	SO_DONTLINGER
    struct linger linger;
#endif				/* SO_DONTLINGER */

    enterprocedure("ConnectToServer");

    /* establish a socket to the name server for this host */
    bzero((char *) &sin, sizeof(sin));
    ServerFD = socket(AF_INET, SOCK_STREAM, 0);
    if (ServerFD < 0) {
	perror("socket() to Server failed");
	debug(1, (stderr, "socket failed\n"));
	panic("Can't open connection to Server");
    }
    (void) setsockopt(ServerFD, SOL_SOCKET, SO_REUSEADDR, &reuse_addr, 0);
#ifndef linux
    (void) setsockopt(ServerFD, SOL_SOCKET, SO_USELOOPBACK, (char *) NULL, 0);
#endif
#ifdef	SO_DONTLINGER
    (void) setsockopt(ServerFD, SOL_SOCKET, SO_DONTLINGER, (char *) NULL, 0);
#else				/* SO_DONTLINGER */
    linger.l_onoff = 0;
    linger.l_linger = 0;
    (void) setsockopt(ServerFD, SOL_SOCKET, SO_LINGER, (char *) &linger, sizeof linger);
#endif				/* SO_DONTLINGER */

    /* determine the host machine for this process */
    if (ServerHostName[0] == '\0')
	(void) gethostname(ServerHostName, sizeof(ServerHostName));
    debug(4, (stderr, "try to connect on %s\n", ServerHostName));

    hp = gethostbyname(ServerHostName);
    if (hp == 0) {
	perror("gethostbyname failed");
	debug(1, (stderr, "gethostbyname failed for %s\n", ServerHostName));
	panic("Can't open connection to Server");
    }
    sin.sin_family = AF_INET;
    bcopy((char *) hp->h_addr, (char *) &sin.sin_addr, hp->h_length);
    port = GetServerport();

    if (port == ScopePort
	&& strcmp(ServerHostName, ScopeHost) == 0) {
	char error_message[100];
	(void) sprintf(error_message, "Trying to attach to myself: %s,%d\n",
		       ServerHostName, sin.sin_port);
	panic(error_message);
    }
    sin.sin_port = htons(port);

    /* ******************************************************** */
    /* try to connect to Server */

    if (connect(ServerFD, (struct sockaddr *) &sin, sizeof(sin)) < 0) {
	debug(4, (stderr, "connect returns errno of %d\n", errno));
	if (errno != 0)
	    if (report)
		perror("connect");
	switch (errno) {
	case ECONNREFUSED:
	    /* experience says this is because there is no Server
	       to connect to */
	    (void) close(ServerFD);
	    debug(1, (stderr, "No Server\n"));
	    if (report)
		warn("Can't open connection to Server");
	    return (-1);

	default:
	    (void) close(ServerFD);
	    panic("Can't open connection to Server");
	}
    }
    debug(4, (stderr, "Connect To Server: FD %d\n", ServerFD));
    if (ServerFD >= 0) {
	UsingFD(ServerFD, (int (*)()) DataFromServer);
	StartServerConnection(ServerFD);
    }
    return (ServerFD);
}

/*----------------------------------------------------------------------*/
char *OfficialName(name)
    char *name;
{
    struct hostent *HostEntry;

    HostEntry = gethostbyname(name);
    if (HostEntry == NULL) {
	fprintf(stderr, "%s: Cannot understand host name as specified: -h%s\n", xscript, name);
	Usage(xscript);
    }
    debug(4, (stderr, "Official name of %s is %s\n", name, HostEntry->h_name));
    return (HostEntry->h_name);
}

/*----------------------------------------------------------------------*/
void DataOnXDisplay(fd)
    FD fd;
{
    XEvent ev;

    enterprocedure("DataOnXDisplay");

    if (fd != ConnectionNumber(Dpy)) {
	panic("impossible...fd mismatch in DataOnXDisplay");
    }
    /* chew up and discard events.... */
    while (XEventsQueued(Dpy, QueuedAfterReading)) {
	XNextEvent(Dpy, &ev);
	debug(4, (stderr, "event type %d in DataOnXDisplay\n", ev.type));
	switch (ev.type) {
	    /*  process keyboard mapping changes  */
	case MappingNotify:
	    XRefreshKeyboardMapping((XMappingEvent *) & ev);
	    break;

	default:
	    break;
	}
    }
}

/*----------------------------------------------------------------------*/
void SetUpXDisplay()
{
    FD displayFD;
    char displayName[256];
    long int displayPort = 0;
    long int myPort = 0;

    enterprocedure("SetUpXDisplay");

    /* determine the host machine for this process */
    if (ServerHostName[0] == '\0')
	(void) gethostname(ServerHostName, sizeof(ServerHostName));

    displayPort = GetServerport() - ServerBasePort;
    myPort = ScopePort - ServerBasePort;

    (void) sprintf(displayName, "%s:%ld", ServerHostName, displayPort);
    debug(4, (stderr, "try to establish local X conn to %s\n", displayName));

    if (displayPort == myPort
	&& strcmp(ServerHostName, ScopeHost) == 0) {
	char error_message[100];
	(void) sprintf(error_message, "Trying to attach to myself: %s\n",
		       displayName);
	panic(error_message);
    }
    Dpy = XOpenDisplay(displayName);
    if (Dpy == 0) {
	char error_message[100];
	(void) sprintf(error_message, "Unable to establish X connection to %s\n",
		       displayName);
	panic(error_message);
    }
    displayFD = ConnectionNumber(Dpy);
    UsingFD(displayFD, (int (*)()) DataOnXDisplay);
}
