/*
 * $Id: server.c,v 1.4 1998/07/24 11:59:18 marc Exp $
 *
 *  Code to decode and print X11 protocol
 *
 *      James Peterson, 1988
 *      (c) Copyright MCC, 1988
 *      (c) Copyright Marc Vertes 1998
 */

#include <malloc.h>

#include "scope.h"
#include "common.h"
#include "record.h"
#include "server.h"
#include "decode11.h"
#include "print11.h"
#include "x11.h"

extern char ScopeEnabled;

struct ConnState CS[StaticMaxFD];

/*----------------------------------------------------------------------*/
void ReportFromClient(fd, buf, n)
    FD fd;
    unsigned char *buf;
    long n;
{
    if (ScopeEnabled) {
	PrintTime();
	fprintf(stdout, "Client%s --> %4ld %s\n",
		ClientName(fd), n, (n == 1 ? "byte" : "bytes"));
    }
    ProcessBuffer(fd, buf, n);
}

/*----------------------------------------------------------------------*/
void ReportFromServer(fd, buf, n)
    FD fd;
    unsigned char *buf;
    long n;
{
    if (ScopeEnabled) {
	PrintTime();
	fprintf(stdout, "\t\t\t\t\t%4ld %s <-- X11 Server%s\n",
		n, (n == 1 ? "byte" : "bytes"), ClientName(fd));
    }
    ProcessBuffer(fd, buf, n);
}

/*----------------------------------------------------------------------*/
static long ZeroTime1 = -1;
static long ZeroTime2 = -1;
static struct timeval tp;

/*----------------------------------------------------------------------*/
/* 
 * Print the time since we started in hundredths (1/100) of seconds 
 */
void PrintTime()
{
    static long lastsec = 0;
    long sec /* seconds */ ;
    long hsec /* hundredths of a second */ ;

    (void) gettimeofday(&tp, (struct timezone *) NULL);
    if (ZeroTime1 == -1 || (tp.tv_sec - lastsec) >= 1000) {
	ZeroTime1 = tp.tv_sec;
	ZeroTime2 = tp.tv_usec / 10000;
    }
    lastsec = tp.tv_sec;
    sec = tp.tv_sec - ZeroTime1;
    hsec = tp.tv_usec / 10000 - ZeroTime2;
    if (hsec < 0) {
	hsec += 100;
	sec -= 1;
    }
    fprintf(stdout, "%2ld.%02ld: ", sec, hsec);
}

/*----------------------------------------------------------------------*/
/* 
 * We will need to be able to interpret the values stored in the
 * requests as various built-in types.  The following routines
 * support the types built into X11
 */
long pad(n)
    long n;
{
    /* round up to next multiple of 4 */
    return ((n + 3) & ~0x3);
}

/*----------------------------------------------------------------------*/
extern int littleEndian;

unsigned long int ILong(buf)
    unsigned char buf[];
{
    /* check for byte-swapping */
    if (littleEndian)
	return ((((((buf[3] << 8) | buf[2]) << 8) | buf[1]) << 8) | buf[0]);
    return ((((((buf[0] << 8) | buf[1]) << 8) | buf[2]) << 8) | buf[3]);
}

/*----------------------------------------------------------------------*/
void OLong(buf, val)
    unsigned char buf[];
    unsigned long int val;
{
    if (littleEndian) {
	buf[0] = val & 0xff;
	val >>= 8;
	buf[1] = val & 0xff;
	val >>= 8;
	buf[2] = val & 0xff;
	val >>= 8;
	buf[3] = val;
    } else {
	buf[3] = val & 0xff;
	val >>= 8;
	buf[2] = val & 0xff;
	val >>= 8;
	buf[1] = val & 0xff;
	val >>= 8;
	buf[0] = val;
    }
}

/*----------------------------------------------------------------------*/
unsigned short IShort(buf)
    unsigned char buf[];
{
    /* check for byte-swapping */
    if (littleEndian)
	return (buf[1] << 8) | buf[0];
    return ((buf[0] << 8) | buf[1]);
}

/*----------------------------------------------------------------------*/
void OShort(buf, val)
    unsigned char buf[];
    unsigned short int val;
{
    if (littleEndian) {
	buf[0] = val & 0xff;
	val >>= 8;
	buf[1] = val & 0xff;
    } else {
	buf[1] = val & 0xff;
	val >>= 8;
	buf[0] = val & 0xff;
    }
}

/*----------------------------------------------------------------------*/
unsigned short IChar2B(buf)
    unsigned char buf[];
{
    /* CHAR2B is like an IShort, but not byte-swapped */
    return ((buf[0] << 8) | buf[1]);
}

/*----------------------------------------------------------------------*/
unsigned short IByte(buf)
    unsigned char buf[];
{
    return (buf[0]);
}

/*----------------------------------------------------------------------*/
void OByte(buf, val)
    unsigned char buf[];
    unsigned short int val;
{
    buf[0] = val & 0xff;
}

/*----------------------------------------------------------------------*/
Boolean IBool(buf)
    unsigned char buf[];
{
    if (buf[0] != 0)
	return (true);
    else
	return (false);
}

/*----------------------------------------------------------------------*/
/* 
 * we will need to save bytes until we get a complete request to
 * interpret.  The following procedures provide this ability 
 */
void SaveBytes(fd, buf, n)
    FD fd;
    unsigned char *buf;
    long n;
{
    /* check if there is enough space to hold the bytes we want */
    if (CS[fd].NumberofSavedBytes + n > CS[fd].SizeofSavedBytes) {
	/* not enough room so far; malloc more space and copy */
	long SizeofNewBytes = (CS[fd].NumberofSavedBytes + n + 1);
	unsigned char *NewBytes = (unsigned char *) Malloc(SizeofNewBytes);
	if (CS[fd].NumberofSavedBytes > 0) {
	    bcopy( /* from  */ (char *) CS[fd].SavedBytes,
	    /* to    */ (char *) NewBytes,
	    /* count */ (int) CS[fd].NumberofSavedBytes);
	    Free((char *) CS[fd].SavedBytes);
	}
	CS[fd].SavedBytes = NewBytes;
	CS[fd].SizeofSavedBytes = SizeofNewBytes;
    }
    /* now copy the new bytes onto the end of the old bytes */
    bcopy( /* from  */ (char *) buf,
    /* to    */ (char *) (CS[fd].SavedBytes + CS[fd].NumberofSavedBytes),
    /* count */ (int) n);
    CS[fd].NumberofSavedBytes += n;
}

/*----------------------------------------------------------------------*/
void RemoveSavedBytes(fd, n)
    FD fd;
    long n;
{
    /* check if all bytes are being removed -- easiest case */
    if (CS[fd].NumberofSavedBytes <= n)
	CS[fd].NumberofSavedBytes = 0;
    else if (n == 0)
	return;
    else {
	/* not all bytes are being removed -- shift the remaining ones down  */
	register unsigned char *p = CS[fd].SavedBytes;
	register unsigned char *q = CS[fd].SavedBytes + n;
	register long i = CS[fd].NumberofSavedBytes - n;
	while (i-- > 0)
	    *p++ = *q++;
	CS[fd].NumberofSavedBytes -= n;
    }
}

/*----------------------------------------------------------------------*/
/* following are the possible values for ByteProcessing */
/* forward declarations */
long StartSetUpMessage();
long FinishSetUpMessage();
long StartRequest();
long FinishRequest();

long StartSetUpReply();
long FinishSetUpReply();
long ServerPacket();
long FinishReply();

int littleEndian;

/*----------------------------------------------------------------------*/
void ProcessBuffer(fd, buf, n)
    FD fd;
    unsigned char *buf;
    long n;
{
    unsigned char *BytesToProcess;
    long NumberofUsedBytes;

    /* as long as we have enough bytes to do anything -- do it */

    while (CS[fd].NumberofSavedBytes + n >= CS[fd].NumberofBytesNeeded) {
	/*
	 * we have enough bytes to do something.  We want the bytes to be
	 * grouped together into one contiguous block of bytes. We have three
	 * cases:
	 *
	 * (1) NumberofSavedBytes = 0; so all needed bytes are in the
	 * read buffer, buf.
	 *
	 * (2) NumberofSavedBytes >= NumberofBytesNeeded; in this case we
	 * will not need to copy any extra bytes into the save buffer.
	 *
	 * (3) 0 < NumberofSavedBytes < NumberofBytesNeeded; so
	 * some bytes are in the save buffer and others are in the read
	 * buffer.  In this case we need to copy some of the bytes from the
	 * read buffer to the save buffer to get as many bytes as we need,
	 * then use these bytes.
	 */

	if (CS[fd].NumberofSavedBytes == 0) {
	    /* 
	     * no saved bytes, so just process the first bytes in the
	     * read buffer 
	     */
	    BytesToProcess = buf /* address of request bytes */ ;
	} else {
	    if (CS[fd].NumberofSavedBytes < CS[fd].NumberofBytesNeeded) {
		/* first determine the number of bytes we need to
		   transfer; then transfer them and remove them from
		   the read buffer. (there may be additional requests
		   in the read buffer) 
		 */
		long m;
		m = CS[fd].NumberofBytesNeeded - CS[fd].NumberofSavedBytes;
		SaveBytes(fd, buf, m);
		buf += m;
		n -= m;
	    }
	    BytesToProcess = CS[fd].SavedBytes /* address of request bytes */ ;
	}

	/*
	   BytesToProcess points to a contiguous block of NumberofBytesNeeded
	   bytes that we should process.  The type of processing depends upon
	   the state we are in. The processing routine should return the
	   number of bytes that it actually used.
	 */
	NumberofUsedBytes = (*CS[fd].ByteProcessing)
	    (fd, BytesToProcess, CS[fd].NumberofBytesNeeded);

	/* the number of bytes that were actually used is normally (but not
	   always) the number of bytes needed.  Discard the bytes that were
	   actually used, not the bytes that were needed. The number of used
	   bytes must be less than or equal to the number of needed bytes. */

	if (NumberofUsedBytes > 0) {
	    if (CS[fd].NumberofSavedBytes > 0)
		RemoveSavedBytes(fd, NumberofUsedBytes);
	    else {
		/* there are no saved bytes, so the bytes that were
		   used must have been in the read buffer */
		buf += NumberofUsedBytes;
		n -= NumberofUsedBytes;
	    }
	}
    }				/* end of while (NumberofSavedBytes + n >= NumberofBytesNeeded) */

    /* not enough bytes -- just save the new bytes for more later */
    if (n > 0)
	SaveBytes(fd, buf, n);
    return;
}


/*----------------------------------------------------------------------*/
/*
 * Byte Processing Routines.  Each routine MUST set NumberofBytesNeeded
 * and ByteProcessing.  It probably needs to do some computation first.
 */
void StartClientConnection(fd)
    FD fd;
{
    enterprocedure("StartClientConnection");
    /* when a new connection is started, we have no saved bytes */
    CS[fd].SavedBytes = NULL;
    CS[fd].SizeofSavedBytes = 0;
    CS[fd].NumberofSavedBytes = 0;

    /* when a new connection is started, we have no reply Queue */
    FlushReplyQ(fd);

    /* each new connection gets a request sequence number */
    CS[fd].SequenceNumber = 0;

    /* we need 12 bytes to start a SetUp message */
    CS[fd].ByteProcessing = StartSetUpMessage;
    CS[fd].NumberofBytesNeeded = 12;
}

/*----------------------------------------------------------------------*/
void StopClientConnection(fd)
    FD fd;
{
    enterprocedure("StopClientConnection");
    /* when a new connection is stopped, discard the old buffer */

    if (CS[fd].SizeofSavedBytes > 0) {
	Free((char *) CS[fd].SavedBytes);
	CS[fd].SizeofSavedBytes = 0;
    }
}

/*----------------------------------------------------------------------*/
long StartSetUpMessage(fd, buf, n)
    FD fd;
    unsigned char *buf;
    long n;
{
    short namelength;
    short datalength;

    enterprocedure("StartSetUpMessage");
    /*
       we need the first 12 bytes to be able to determine if, and how many,
       additional bytes we need for name and data authorization.  However, we
       can't process the first 12 bytes until we get all of them, so
       return zero bytes used, and increase the number of bytes needed
     */

    namelength = IShort(&buf[6]);
    datalength = IShort(&buf[8]);
    CS[fd].ByteProcessing = FinishSetUpMessage;
    CS[fd].NumberofBytesNeeded = n
	+ pad((long) namelength) + pad((long) datalength);
    debug(8, (stderr, "need %ld bytes to finish startup\n",
	      CS[fd].NumberofBytesNeeded - n));
    return (0);
}

/*----------------------------------------------------------------------*/
long FinishSetUpMessage(fd, buf, n)
    FD fd;
    unsigned char *buf;
    long n;
{
    enterprocedure("FinishSetUpMessage");
    CS[fd].littleEndian = (buf[0] == 'l');
    CS[ServerHalf(fd)].littleEndian = CS[fd].littleEndian;
    littleEndian = CS[fd].littleEndian;
    if (ScopeEnabled)
	PrintSetUpMessage(buf);

    /* after a set-up message, we expect a string of requests */
    CS[fd].ByteProcessing = StartRequest;
    CS[fd].NumberofBytesNeeded = 4;
    return (n);
}

/*----------------------------------------------------------------------*/
long StartRequest(fd, buf, n)
    FD fd;
    unsigned char *buf;
    long n;
{
    unsigned short requestlength;
    /*MV enterprocedure("StartRequest"); */

    /* bytes 0,1 are ignored now; bytes 2,3 tell us the request length */
    requestlength = IShort(&buf[2]);
    CS[fd].ByteProcessing = FinishRequest;
    CS[fd].NumberofBytesNeeded = 4 * requestlength;
    debug(8, (stderr, "need %ld more bytes to finish request\n",
	      CS[fd].NumberofBytesNeeded - n));
    return (0);
}

/*----------------------------------------------------------------------*/
long FinishRequest(fd, buf, n)
    FD fd;
    unsigned char *buf;
    long n;
{

    /* CS[fd].SequenceNumber += 1; */

    /*MV enterprocedure("FinishRequest"); */

    if (ScopeEnabled)
	DecodeRequest(fd, buf, n);

    if (Record)
	RecordRequest(fd, buf, n);

    switch (Playback) {
    case PLAYBACK_PENDING:
    case PLAYBACK_ACTIVE:
	MatchWtab(fd, buf);
	break;
    }

    ProcessGrab(fd, buf, n);

    CS[fd].ByteProcessing = StartRequest;
    CS[fd].NumberofBytesNeeded = 4;
    return (n);
}

/*----------------------------------------------------------------------*/
void StartServerConnection(fd)
    FD fd;
{
    enterprocedure("StartServerConnection");
    /* when a new connection is started, we have no saved bytes */
    CS[fd].SavedBytes = NULL;
    CS[fd].SizeofSavedBytes = 0;
    CS[fd].NumberofSavedBytes = 0;

    /* when a new connection is started, we have no reply Queue */
    FlushReplyQ(fd);

    /* we need 8 bytes to start a SetUp reply */
    CS[fd].ByteProcessing = StartSetUpReply;
    CS[fd].NumberofBytesNeeded = 8;
}

/*----------------------------------------------------------------------*/
void StopServerConnection(fd)
    FD fd;
{
    enterprocedure("StopServerConnection");
    /* when a new connection is stopped, discard the old buffer */

    if (CS[fd].SizeofSavedBytes > 0) {
	Free((char *) CS[fd].SavedBytes);
	CS[fd].SizeofSavedBytes = 0;
    }
}

long StartSetUpReply(fd, buf, n)
    FD fd;
    unsigned char *buf;
    long n;
{
    short replylength;

    /*MV enterprocedure("StartSetUpReply"); */
    replylength = IShort(&buf[6]);
    CS[fd].ByteProcessing = FinishSetUpReply;
    CS[fd].NumberofBytesNeeded = n + 4 * replylength;
    debug(8, (stderr, "need %ld bytes to finish startup reply\n",
	      CS[fd].NumberofBytesNeeded - n));
    return (0);
}

/*----------------------------------------------------------------------*/
long FinishSetUpReply(fd, buf, n)
    FD fd;
    unsigned char *buf;
    long n;
{

    unsigned long int ResourceIdBase;
    unsigned long int ResourceIdMask;

    /* enterprocedure("FinishSetUpReply"); */
    if (ScopeEnabled)
	PrintSetUpReply(buf);

    /* pass along to client if playing back from file */
    if (Playback != PLAYBACK_INACTIVE)
	WriteToClient(fd, buf, n);

    CS[fd].ByteProcessing = ServerPacket;
    CS[fd].NumberofBytesNeeded = 32;

    ResourceIdBase = ILong(&buf[12]);
    ResourceIdMask = ILong(&buf[16]);

    CS[fd].ResourceIdBase = ResourceIdBase;
    CS[fd].ResourceIdMask = ResourceIdMask;

    CS[fd].TimeInitialized = 0;

    return (n);
}

/*----------------------------------------------------------------------*/
long ErrorPacket(fd, buf, n)
    FD fd;
    unsigned char *buf;
    long n;
{
/*
   fprintf(stdout, "Error: ");
   DecodeError(fd, buf, n);
 */
    /* pass along to client if playing back from file */
    if (Playback != PLAYBACK_INACTIVE)
	WriteToClient(fd, buf, n);

    /* prepare for next packet */
    CS[fd].ByteProcessing = ServerPacket;
    CS[fd].NumberofBytesNeeded = 32;
    return (n);
}

/*----------------------------------------------------------------------*/
long EventPacket(fd, buf, n)
    FD fd;
    unsigned char *buf;
    long n;
{

    if (ScopeEnabled)
	DecodeEvent(fd, buf, n);

    if (Record)
	RecordEvent(fd, buf, n);

    /*
     * if we're playing back, start things happening 
     * after the first event arrives, meaning the
     * client program under test has been initiated.
     * apply the "wait" parameter specified in the
     * xscript command line.
     */
    if (Playback == PLAYBACK_PENDING) {
	static int first_time = 1;
	if (first_time) {
	    warn("Activate next playback event");
	    CreateTimer(1000 * Wait, (void (*)()) ReadPlaybackEvent, 0);
	    first_time = 0;
	}
    }
    /*
     * if we're playing stuff back, 
     * we want to monitor the appearance
     * of resources by watching events
     * We also want to decide whether to
     * pass the events received from the server
     * along to the client (we don't pass along
     * the kinds of events we're synthesizing.)
     */
    switch (Playback) {
	extern Boolean MonitorEvent();

    case PLAYBACK_INACTIVE:
	break;

    case PLAYBACK_PENDING:
    case PLAYBACK_COMPLETE:
	/* nonselective delivery of events */
	/* MonitorEvent (fd, buf, n); */
	WriteToClient(fd, buf, n);
	break;

    case PLAYBACK_ACTIVE:
	/* selective delivery of events */
	if (MonitorEvent(fd, buf, n)) {
	    WriteToClient(fd, buf, n);
	}
	break;

    default:
	panic("bad Playback value.");
	break;

    }

    CS[fd].ByteProcessing = ServerPacket;
    CS[fd].NumberofBytesNeeded = 32;
    return (n);
}

/*----------------------------------------------------------------------*/
long ReplyPacket(fd, buf, n)
    FD fd;
    unsigned char *buf;
    long n;
{
    short replylength;

    replylength = ILong(&buf[4]);

    /*
       Replies may need more bytes, so we compute how many more
       bytes are needed and ask for them, not using any of the bytes
       we were given (return(0) to say that no bytes were used).
       If the replylength is zero (we don't need any more bytes), the
       number of bytes needed will be the same as what we have, and
       so the top-level loop will call the next routine immediately
       with the same buffer of bytes that we were given.
     */

    CS[fd].ByteProcessing = FinishReply;
    CS[fd].NumberofBytesNeeded = n + 4 * replylength;
    debug(8, (stderr, "need %d bytes to finish reply\n", (4 * replylength)));
    return (0);
}

/*----------------------------------------------------------------------*/
long ServerPacket(fd, buf, n)
    FD fd;
    unsigned char *buf;
    long n;
{
    short PacketType;
    /*MV enterprocedure("ServerPacket"); */

    PacketType = IByte(&buf[0]);
    if (PacketType == 0)
	return (ErrorPacket(fd, buf, n));
    if (PacketType == 1)
	return (ReplyPacket(fd, buf, n));
    return (EventPacket(fd, buf, n));
}

/*----------------------------------------------------------------------*/
long FinishReply(fd, buf, n)
    FD fd;
    unsigned char *buf;
    long n;
{
    FD pairfd = FDPair(fd);

    /*MV enterprocedure("FinishReply"); */

    CS[pairfd].SequenceNumber = IShort(&buf[2]);

    if (ScopeEnabled)
	DecodeReply(fd, buf, n);

    /* pass along playback events if necessary */
    if (Playback != PLAYBACK_INACTIVE)
	WriteToClient(fd, buf, n);

    CS[fd].ByteProcessing = ServerPacket;
    CS[fd].NumberofBytesNeeded = 32;
    return (n);
}
