/*
 * lpf.c		- lost+found management for e2fsck
 *
 * Copyright (C) 1992, 1993  Remy Card <card@masi.ibp.fr>
 *
 * This file is based on the minix file system programs fsck and mkfs
 * written and copyrighted by Linus Torvalds <Linus.Torvalds@cs.helsinki.fi>
 *
 * This file can be redistributed under the terms of the GNU General
 * Public License
 */

/*
 * History:
 * 93/05/26	- Creation from e2fsck
 * 93/06/29	- Don't move a directory to lost+found is size < 24
 */

#include <stdio.h>
#include <string.h>
#include <sys/stat.h>

#include <sys/stat.h>

#include <linux/ext2_fs.h>

#include "bitops.h"
#include "e2fsprogs.h"
#include "ckfunc.h"
#include "ckvar.h"

/*
 * Look for the lost+found directory
 */
static int find_lpf_inode (int dev, struct ext2_inode * lpf_inode)
{
	struct ext2_inode root_inode;
	char * buffer = blkbuf;
	unsigned long block;
	struct ext2_dir_entry * de;

	read_inode (dev, EXT2_ROOT_INO, &root_inode);
	block = 0;
	while ((block * block_size) < root_inode.i_size)
	{
		mapped_read_block (dev, &root_inode, block, buffer);
		de = (struct ext2_dir_entry *) buffer;
		while ((char *) de < buffer + block_size)
			if (de->inode && de->name_len == 10 &&
			    strncmp (de->name, "lost+found", 10) == 0)
			{
				read_inode (dev, de->inode, lpf_inode);
				return de->inode;
			}
			else
				de = (struct ext2_dir_entry *) ((char *) de +
								de->rec_len);
		block++;
	}
	return 0;
}

/*
 * Add an entry to the lost+found directory
 */
void set_lost_dir_ent (int dev, unsigned long ino, int mode)
{
	static struct ext2_inode lpf_inode;
	static int lpf_inode_number = -1;
	int dir_changed;
	unsigned long block;
	char * buffer = blkbuf + block_size;
	struct ext2_dir_entry * de;
	struct ext2_dir_entry * de1;
	char * name = blkbuf + block_size * 2;
	int rec_len;

	if (lpf_inode_number == -1)
	{
		lpf_inode_number = find_lpf_inode (dev, &lpf_inode);
		if (lpf_inode_number == 0)
		{
			printf ("No directory lost+found\n");
			no_lpf = 1;
			return;
		}
	}
	block = 0;
	if (mapped_read_block (dev, &lpf_inode, block, buffer))
		write_inode (dev, lpf_inode_number, &lpf_inode);
	de = (struct ext2_dir_entry *) buffer;
	sprintf (name, "%lu", ino);
	rec_len = EXT2_DIR_REC_LEN(strlen (name));
	while (block * block_size < lpf_inode.i_size)
	{
		if (de->rec_len < EXT2_DIR_REC_LEN(1) ||
		    de->rec_len % 4 != 0 ||
		    de->rec_len < EXT2_DIR_REC_LEN(de->name_len))
		{
			printf ("lost+found is corrupted, cannot fix\n");
			lpf_corrupted = 1;
			return;
		}
		if ((!de->inode && de->rec_len >= rec_len) ||
		    de->rec_len >= EXT2_DIR_REC_LEN(de->name_len) + rec_len)
		{
			if (de->inode)
			{
				de1 = (struct ext2_dir_entry *) ((char *) de + EXT2_DIR_REC_LEN(de->name_len));
				de1->rec_len = de->rec_len - EXT2_DIR_REC_LEN(de->name_len);
				de->rec_len = EXT2_DIR_REC_LEN(de->name_len);
			}
			else
				de1 = de;
			de1->inode = ino;
			de1->name_len = strlen (name);
			strncpy (de1->name, name, strlen (name));
			write_block (dev, last_block_read, buffer);
			strcpy (name_list[name_depth], name);
			name_depth++;
			check_file (&lpf_inode, &dir_changed,
				    (block * block_size) +
				    ((char *) de1 - buffer));
			name_depth--;
			if (S_ISDIR(mode)) 
			{
				lpf_inode.i_links_count++;
				dir_changed = 1;
			}
			if (dir_changed)
				write_inode (dev, lpf_inode_number, &lpf_inode);
			return;
		}
		else
		{
			de = (struct ext2_dir_entry *) ((char *) de + de->rec_len);
			if ((char *) de > buffer + block_size)
			{
				block++;
				if (block * block_size < lpf_inode.i_size)
				{
					mapped_read_block (dev, &lpf_inode, block,
							   buffer);
					de = (struct ext2_dir_entry *) buffer;
				}
			}
		}
	}
	printf ("directory lost+found is full\n");
	return;
}

void check_connectivity (int dev)
{
	unsigned long ino;
	struct ext2_inode inode;

	DEBUG(("check_connectivity ()\n"));
	if (verbose)
		printf ("Searching for lost inodes\n");
	name_depth = 1;
	strcpy (name_list[0], "lost+found");
	for (ino = 1; ino < INODES; ino++)
	{
		if (ino == EXT2_BAD_INO ||
		    (ino > EXT2_ROOT_INO && ino < EXT2_FIRST_INO))
			continue;
		if (!inode_count[ino] && inode_in_use (ino))
		{
			printf("Inode %lu not used"
			       ", not counted in the bitmap. ", ino);
			read_inode (dev, ino, &inode);
			if (!inode.i_dtime && inode.i_links_count &&
			    (!S_ISDIR (inode.i_mode) || inode.i_size >= 24) &&
			    !no_lpf && !lpf_corrupted)
			{
				if (ask ("Connect to lost+found", 1))
					set_lost_dir_ent (dev, ino, 
							  inode.i_mode);
			}
			else
			{
				if (lpf_corrupted)
					printf ("(lost+found is corrupted). ");
				if (no_lpf)
					printf ("(no directory lost+found). ");
				if (ask ("Unmark in the bitmap", 1))
					unmark_inode (ino);
			}
		}
	}
}
