/*
 * tables1.c		- File system tables management for e2fsck
 *
 * Copyright (C) 1992, 1993  Remy Card <card@masi.ibp.fr>
 *
 * This file is based on the minix file system programs fsck and mkfs
 * written and copyrighted by Linus Torvalds <Linus.Torvalds@cs.helsinki.fi>
 *
 * This file can be redistributed under the terms of the GNU General
 * Public License
 */

/*
 * History:
 * 93/05/26	- Creation from e2fsck
 * 93/11/01	- Can use an alternate super block
 *		  Write super block and descriptor backups
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <time.h>
#include <unistd.h>

#include <linux/ext2_fs.h>

#include "e2fsprogs.h"
#include "ckfunc.h"
#include "ckvar.h"

extern int retval;

static char * padding_bits = NULL;

static void init_padding (void)
{
	if (padding_bits)
		return;
	padding_bits = malloc (block_size);
	memset (padding_bits, -1, block_size);
}

static void write_inode_bitmap (int dev)
{
	int i, bytes = INODESPERGROUP / 8;
	char * inode_bitmap = inode_map;

	DEBUG(("write_inode_bitmap()\n"));
	init_padding ();
	for (i = 0; i < group_desc_count; i++)
	{
		if (lseek (dev, group_desc[i].bg_inode_bitmap * block_size,
			   SEEK_SET) !=
			group_desc[i].bg_inode_bitmap * block_size)
			die ("seek failed in write_inode_bitmap", EXIT_ERROR);
		if (write (dev, inode_bitmap, bytes) != bytes)
			die ("write failed in write_inode_bitmap", EXIT_ERROR);
		if ((bytes != block_size) &&
		    write (dev, padding_bits, block_size - bytes) != block_size - bytes)
			die ("write failed in write_inode_bitmap", EXIT_ERROR);
		inode_bitmap += bytes;
	}
}

static void write_block_bitmap (int dev)
{
	int i, bytes = BLOCKSPERGROUP / 8;
	char * block_bitmap = block_map;

	DEBUG(("write_block_bitmap()\n"));
	init_padding();
	for (i = 0; i < group_desc_count; i++)
	{
		if (lseek (dev, group_desc[i].bg_block_bitmap * block_size,
			   SEEK_SET) !=
			group_desc[i].bg_block_bitmap * block_size)
			die ("seek failed in write_block_bitmap", EXIT_ERROR);
		if (write (dev, block_bitmap, bytes) != bytes)
			die ("write failed in write_block_bitmap", EXIT_ERROR);
		if ((bytes != block_size) &&
		    write (dev, padding_bits, block_size - bytes) != block_size - bytes)
			die ("write failed in write_block_bitmap", EXIT_ERROR);
		block_bitmap += bytes;
	}
}

void write_super_block (int dev)
{
	int i;
	unsigned long block;

	STATE |= EXT2_VALID_FS;
	if (retval & EXIT_UNCORRECTED)
		STATE |= EXT2_ERROR_FS;
	else
		STATE &= ~EXT2_ERROR_FS;
	MNTCNT = 0;
	WTIME = time(NULL);
	/* Write master copy of the super block */
	if (lseek (dev, EXT2_MIN_BLOCK_SIZE, SEEK_SET) != EXT2_MIN_BLOCK_SIZE)
		die ("seek (first sb) failed in write_tables", EXIT_ERROR);
	if (write (dev, (char *) Super, EXT2_MIN_BLOCK_SIZE) !=
	    EXT2_MIN_BLOCK_SIZE)
		die ("unable to write super-block", EXIT_ERROR);
	/* Write copies of the super block */
	block = FIRSTBLOCK + BLOCKSPERGROUP;
	for (i = 1; i < group_desc_count; i++)
	{
		if (lseek (dev, block_size * block, SEEK_SET) !=
		    block_size * block)
			die ("seek (sb backup) failed in write_tables", EXIT_ERROR);
		if (write (dev, (char *) Super, EXT2_MIN_BLOCK_SIZE) !=
		    EXT2_MIN_BLOCK_SIZE)
			die ("unable to write super-block backup", EXIT_ERROR);
		block += BLOCKSPERGROUP;
	}
}

void write_tables (int dev)
{
	int i;
	unsigned long block;

	DEBUG(("DEBUG: write_tables()\n"));
	write_super_block (dev);
	block = FIRSTBLOCK + 1;
	for (i = 0; i < group_desc_count; i++)
	{
		if (lseek (dev, block_size * block, SEEK_SET) !=
		    block_size * block)
			die ("seek (desc) failed in write_tables", EXIT_ERROR);
		if (write (dev, (char *) group_desc, group_desc_size) !=
		    group_desc_size)
			die ("write of group descriptors failed\n", EXIT_ERROR);
		block += BLOCKSPERGROUP;
	}
	write_inode_bitmap (dev);
	write_block_bitmap (dev);
}

static void read_inode_bitmap (int dev)
{
	int i;
	char * inode_bitmap = inode_map;

	DEBUG(("DEBUG: read_inode_bitmap()\n"));
	for (i = 0; i < group_desc_count; i++)
	{
		DEBUG(("Group %d: inode bitmap at %d read at %d\n",
			i, group_desc[i].bg_inode_bitmap,
			inode_bitmap - inode_map));
		if (lseek (dev, group_desc[i].bg_inode_bitmap * block_size,
			   SEEK_SET) !=
		    group_desc[i].bg_inode_bitmap * block_size)
			die ("seek failed in read_inode_bitmap", EXIT_ERROR);
		if (read (dev, inode_bitmap, INODESPERGROUP / 8) !=
		    INODESPERGROUP / 8)
			die ("read failed in read_inode_bitmap", EXIT_ERROR);
		inode_bitmap += INODESPERGROUP / 8;
	}
}

static void read_block_bitmap (int dev)
{
	int i;
	char * block_bitmap = block_map;

	DEBUG(("DEBUG: read_block_bitmap()\n"));
	for (i = 0; i < group_desc_count; i++)
	{
		DEBUG(("Group %d: block bitmap at %d read at %d\n",
			i, group_desc[i].bg_block_bitmap,
			block_bitmap - block_map));
		if (lseek (dev, group_desc[i].bg_block_bitmap * block_size,
			   SEEK_SET) !=
			group_desc[i].bg_block_bitmap * block_size)
			die ("seek failed in read_block_bitmap", EXIT_ERROR);
		if (read (dev, block_bitmap, BLOCKSPERGROUP / 8) !=
		    BLOCKSPERGROUP / 8)
			die ("read failed in read_block_bitmap", EXIT_ERROR);
		block_bitmap += BLOCKSPERGROUP / 8;
	}
}

static void check_desc (void)
{
	int i;
	unsigned long block = FIRSTBLOCK;

	for (i = 0; i < group_desc_count; i++)
	{
		if (group_desc[i].bg_block_bitmap < block ||
		    group_desc[i].bg_block_bitmap > block + BLOCKSPERGROUP)
		{
			printf ("Block bitmap for group %d"
				" not in group (block %lu)!",
				i, group_desc[i].bg_block_bitmap);
			die ("descriptors corrupted", EXIT_ERROR);
		}
		if (group_desc[i].bg_inode_bitmap < block ||
		    group_desc[i].bg_inode_bitmap > block + BLOCKSPERGROUP)
		{
			printf ("Inode bitmap for group %d"
				" not in group (block %lu)!",
				i, group_desc[i].bg_inode_bitmap);
			die ("descriptors corrupted", EXIT_ERROR);
		}
		if (group_desc[i].bg_inode_table < block ||
		    group_desc[i].bg_inode_table + inode_blocks_per_group >
		    block + BLOCKSPERGROUP)
		{
			printf ("Inode table for group %d"
				" not in group (block %lu)!",
				i, group_desc[i].bg_inode_table);
			die ("descriptors corrupted", EXIT_ERROR);
		}
		block += BLOCKSPERGROUP;
	}
}

int read_tables (int dev, long sb)
{
	int i;
	long desc_loc;

	DEBUG(("DEBUG: read_tables()\n"));
	if (lseek (dev, sb * EXT2_MIN_BLOCK_SIZE, SEEK_SET) !=
	    sb * EXT2_MIN_BLOCK_SIZE)
		die ("seek failed", EXIT_ERROR);
	if (read (dev, (char *) Super, EXT2_MIN_BLOCK_SIZE) !=
	    EXT2_MIN_BLOCK_SIZE)
		die ("unable to read super block", EXIT_ERROR);
	if (MAGIC != EXT2_SUPER_MAGIC)
		die ("bad magic number in super-block", EXIT_ERROR);
	if (!(STATE & EXT2_VALID_FS))
		printf ("Filesystem not clean, checking it\n");
	else if (STATE & EXT2_ERROR_FS)
		printf ("Filesystem with errors, checking it\n");
	else if (MNTCNT >= MAXMNTCNT)
		printf ("Maximal mounts count of the filesystem reached, "
			"checking it\n");
	else if (!force)
		return 0;
/*	if (BLOCKSIZE != 0 || EXT2_MIN_BLOCK_SIZE != 1024)
		die ("Only 1k blocks supported", EXIT_ERROR); */

	addr_per_block = EXT2_ADDR_PER_BLOCK (Super);
	block_size = EXT2_BLOCK_SIZE (Super);
	inode_blocks_per_group = INODESPERGROUP /
				 EXT2_INODES_PER_BLOCK (Super);

	group_desc_count = (BLOCKS - NORM_FIRSTBLOCK) / BLOCKSPERGROUP;
	if ((group_desc_count * BLOCKSPERGROUP) != (BLOCKS - NORM_FIRSTBLOCK))
		group_desc_count++;
	if (group_desc_count % EXT2_DESC_PER_BLOCK (Super))
		desc_blocks = (group_desc_count / EXT2_DESC_PER_BLOCK (Super)) + 1;
	else
		desc_blocks = group_desc_count / EXT2_DESC_PER_BLOCK (Super);
	group_desc_size = desc_blocks * block_size;
	group_desc = malloc (group_desc_size);
	if (!group_desc)
		die ("Unable to allocate buffers for group descriptors", EXIT_ERROR);
	desc_loc = (((sb * EXT2_MIN_BLOCK_SIZE) / block_size) + 1) * block_size;
	if (lseek (dev, desc_loc, SEEK_SET) != desc_loc)
		die ("seek failed", EXIT_ERROR);
	if (read (dev, group_desc, group_desc_size) != group_desc_size)
		die ("Unable to read group descriptors", EXIT_ERROR);

	check_desc ();

	block_count = malloc(BLOCKS * sizeof (*block_count));
	if (!block_count)
		die ("Unable to allocate buffer for block count", EXIT_ERROR);

	inode_map = malloc ((INODES / 8) + 1);
	if (!inode_map)
		die ("Unable to allocate inodes bitmap", EXIT_ERROR);
	memset (inode_map, 0, (INODES / 8) + 1);
	read_inode_bitmap (dev);

	i = EXT2_BLOCK_SIZE (Super) * group_desc_count;
	block_map = malloc (i);
	if (!block_map)
		die ("Unable to allocate blocks bitmap", EXIT_ERROR);
	memset (block_map, 0, i);
	read_block_bitmap (dev);

	bad_map = malloc (((BLOCKS - FIRSTBLOCK) / 8) + 1);
	if (!bad_map)
		die ("Unable to allocate bad block bitmap", EXIT_ERROR);
	memset (bad_map, 0, ((BLOCKS - FIRSTBLOCK) / 8) + 1);

	inode_count = malloc ((INODES + 1) * sizeof (*inode_count));
	if (!inode_count)
		die ("Unable to allocate buffer for inode count", EXIT_ERROR);

	if (NORM_FIRSTBLOCK != FIRSTBLOCK)
		printf ("Warning: First block (%lu)"
			" != Normal first block (%lu)\n",
			FIRSTBLOCK, NORM_FIRSTBLOCK);

	if (show)
	{
		printf ("Block size = %d\n", EXT2_BLOCK_SIZE (Super));
		printf ("Fragment size = %d\n", EXT2_FRAG_SIZE (Super));
		printf ("%lu inode%s\n", INODES, (INODES != 1) ? "s" : "");
		printf ("%lu block%s\n", BLOCKS, (BLOCKS != 1) ? "s" : "");
		printf ("%lu reserved block%s\n", RBLOCKS,
			(RBLOCKS != 1) ? "s" : "");
		printf ("First data block = %lu (normally %lu)\n",
			FIRSTBLOCK, NORM_FIRSTBLOCK);
		printf ("Mount counts = %d (maximum %d)\n",
			MNTCNT, MAXMNTCNT);
		printf ("Last mounted on %s", ctime (&MTIME));
		printf ("Super block last written on %s", ctime (&WTIME));
		printf ("%lu free block%s\n", FREEBLOCKSCOUNT,
			(FREEBLOCKSCOUNT != 1) ? "s" : "");
		printf ("%lu free inode%s\n", FREEINODESCOUNT,
			(FREEINODESCOUNT != 1) ? "s" : "");
		printf ("%lu group%s (%lu descriptors block%s)\n",
			group_desc_count, (group_desc_count != 1) ? "s" : "",
			desc_blocks, (desc_blocks != 1) ? "s" : "");
		for (i = 0; i < group_desc_count; i++)
			printf ("  Group %d: block bitmap at %lu, "
				"inode bitmap at %lu, "
				"inode table at %lu\n"
				"           %d free block%s, "
				"%d free inode%s, "
				"%d used director%s\n",
				i, group_desc[i].bg_block_bitmap,
				group_desc[i].bg_inode_bitmap,
				group_desc[i].bg_inode_table,
				group_desc[i].bg_free_blocks_count,
				group_desc[i].bg_free_blocks_count != 1 ? "s" : "",
				group_desc[i].bg_free_inodes_count,
				group_desc[i].bg_free_inodes_count != 1 ? "s" : "",
				group_desc[i].bg_used_dirs_count,
				group_desc[i].bg_used_dirs_count != 1 ? "ies" : "y");
		printf ("%lu blocks per group\n", BLOCKSPERGROUP);
		printf ("%lu fragments per group\n", FRAGSPERGROUP);
		printf ("%lu inodes per group\n", INODESPERGROUP);
	}
	return 1;
}
