/*
 * tune2fs.c		- Change the reserved blocks percentage on
 *			  an unmounted second extended file system
 *
 * Copyright (C) 1992, 1993  Remy Card <card@masi.ibp.fr>
 *
 * This file can be redistributed under the terms of the GNU General
 * Public License
 */

/*
 * History:
 * 93/06/01	- Creation
 * 93/10/31	- Added the -c option to change the maximal mount counts
 */

#include <stdio.h>
#include <unistd.h>
#include <string.h>
#include <fcntl.h>
#include <stdlib.h>
#include <time.h>
#include <getopt.h>

#include <linux/ext2_fs.h>

#include "e2fsprogs.h"

char * program_name = "tune2fs";
char * device_name = NULL;
int dev = -1;
int c_flag = 0;
int m_flag = 0;
int max_mount_count;
int reserved_ratio = 0;

static char super_block_buffer[EXT2_MIN_BLOCK_SIZE];

struct ext2_super_block * Super = (struct ext2_super_block *) super_block_buffer;

static unsigned long group_desc_count;

static volatile void fatal_error (const char * fmt_string, int errcode)
{
	fprintf (stderr, fmt_string, program_name, device_name);
	exit (errcode);
}

#define usage()	fatal_error ("usage: %s [-c max-mounts-count]" \
			     " [-m reserved-blocks-percent] /dev/name\n", \
			     EXIT_USAGE);

static volatile void die (const char * str, int errcode)
{
	char string [200];

	strcpy (string, "%s: ");
	strcat (string, str);
	strcat (string, "\n");
	fatal_error (string, errcode);
}

static void read_super (int dev)
{
	if (lseek (dev, EXT2_MIN_BLOCK_SIZE, SEEK_SET) != EXT2_MIN_BLOCK_SIZE)
		die ("seek failed", EXIT_ERROR);
	if (read (dev, (char *) Super, EXT2_MIN_BLOCK_SIZE) != EXT2_MIN_BLOCK_SIZE)
		die ("unable to read super block", EXIT_ERROR);
	if (MAGIC != EXT2_SUPER_MAGIC)
		die ("bad magic number in super-block", EXIT_ERROR);
	group_desc_count = (BLOCKS - NORM_FIRSTBLOCK)  / BLOCKSPERGROUP;
	if ((group_desc_count * BLOCKSPERGROUP) != (BLOCKS - NORM_FIRSTBLOCK))
		group_desc_count++;
}

static void write_super (int dev)
{
	int i;
	unsigned long block;

	WTIME = time (NULL);
	block = FIRSTBLOCK;
	for (i = 0; i < group_desc_count; i++)
	{
		if (lseek (dev, EXT2_MIN_BLOCK_SIZE * block, SEEK_SET) !=
		    EXT2_MIN_BLOCK_SIZE * block)
			die ("seek failed in write_tables", EXIT_ERROR);
		if (write (dev, (char *) Super, EXT2_MIN_BLOCK_SIZE) !=
		    EXT2_MIN_BLOCK_SIZE)
			die ("unable to write super-block", EXIT_ERROR);
		block += BLOCKSPERGROUP;
	}
}

void main (int argc, char ** argv)
{
	char c;
	char * tmp;

	fprintf (stderr, "tune2fs %s, %s for EXT2 FS %s, %s\n",
		 E2FSPROGS_VERSION, E2FSPROGS_DATE,
		 EXT2FS_VERSION, EXT2FS_DATE);
	if (argc && *argv)
		program_name = *argv;
	while ((c = getopt (argc, argv, "c:m:")) != EOF)
		switch (c)
		{
			case 'c':
				max_mount_count = strtol (optarg, &tmp, 0);
				if (*tmp || max_mount_count < 1 ||
				    max_mount_count > 32000)
				{
					printf ("Bad mount count : %s\n", optarg);
					usage ();
				}
				c_flag = 1;
				break;
			case 'm':
				reserved_ratio = strtol (optarg, &tmp, 0);
				if (*tmp || reserved_ratio > 50)
				{
					printf ("bad reserved block ratio : %s\n", optarg);
					usage ();
				}
				m_flag = 1;
				break;
			default:
				usage ();
		}
	if (optind < argc - 1 || optind == argc)
		usage ();
	if (!c_flag && !m_flag)
		usage ();
	device_name = argv[optind];
	dev = open (device_name, O_RDWR);
	if (dev < 0)
		die("unable to open %s", EXIT_ERROR);
	read_super (dev);
	if (c_flag)
	{
		MAXMNTCNT = max_mount_count;
		printf ("Setting maximal mount count to %d\n", max_mount_count);
	}
	if (m_flag)
	{
		RBLOCKS = (BLOCKS / 100) * reserved_ratio;
		printf ("Setting reserved blocks percentage to %d (%lu blocks)\n",
			reserved_ratio, RBLOCKS);
	}
	write_super (dev);
	exit (EXIT_OK);
}
